package pms.shape;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.net.*;
import java.util.*;

import pms.*;
import pms.table.*;
import pms.tools.*;

public abstract class ShapeFile {
 
  public static ShapeFile create(String path) throws IOException {

    return create(path, true);
  }

  public static ShapeFile create(String path, boolean index) 
    throws IOException {

    ShapeFile shapeFile = null;

    switch (getShapeType(path)) {
      case POINT:   shapeFile = new PointFile();   break;
      case ARC:     shapeFile = new LineFile();    break;
      case POLYGON: shapeFile = new PolygonFile(); break;

      case POINT_FLOAT:   shapeFile = new PointFloatFile();   break;
      case ARC_FLOAT:     shapeFile = new LineFloatFile();    break;
      case POLYGON_FLOAT: shapeFile = new PolygonFloatFile(); break;
    }
    shapeFile.open(path, index);
    
    return shapeFile;
  }

  public static int getShapeType(String path) throws IOException {

    RandomAccessFile file = null;
    int shapeType;

    try {

      File extPath = new File(StringTools.setExtension(path, "shq"));
      if (! extPath.exists())
        extPath = new File(StringTools.setExtension(path, "shp"));

      file = new RandomAccessFile(extPath, "r");
      file.seek(32);
      shapeType = LittleEndian.toInt(file.readInt());
      file.close();

    } catch (IOException e) {

      if (file != null) file.close();
      throw new IOException(e.getMessage() + ": " + path);
    }

    return shapeType;
  }

  protected void open(String path, boolean index) throws IOException {

    this.path = path;

    try {
      readHeaders();
  
      if (index) {
        File tesselationFile = new File(
          StringTools.setExtension(path, "tes"));
        if (! tesselationFile.exists())
          createTesselation();
        globalIndex = new FileTesselation(path);
      }
    } catch (IOException e) {
      close();
      throw new IOException(e.getMessage() + ": " + path);
    }
  }

  protected void readHeaders() throws IOException {

    File extPath = new File(StringTools.setExtension(path, "shq"));
    boolean isConcatenated = true;

    if (! extPath.exists()) {
      extPath = new File(StringTools.setExtension(path, "shp"));
      isConcatenated = false;
    }

    shapeFile = new RandomAccessFile(extPath, "r");
    fileCode = shapeFile.readInt();

    shapeFile.seek(24);
    fileLength = shapeFile.readInt();
    version = LittleEndian.toInt(shapeFile.readInt());
    shapeType = LittleEndian.toInt(shapeFile.readInt());
    extent = readBounds(shapeFile);

    if (isConcatenated) {
      shapeIndex = shapeFile;
      indexOffset = fileLength * 2;
    } else {
      shapeIndex = new RandomAccessFile(
        StringTools.setExtension(path, "shx"), "r");
      indexOffset = 0;
    }

    shapeIndex.seek(indexOffset + 24);
    int indexLength = shapeIndex.readInt();
    recordCount = (indexLength - 50) / 4;

    dbFile = new DBaseFile();
    if (isConcatenated) {
      long dbOffset = indexOffset + (indexLength * 2);
      dbFile.open(path, shapeFile, dbOffset);
    } else {
      dbFile.open(path);
    }
  }

  public String getPath() {

    return path;
  }

  public Tesselation getIndex() {
  
    return globalIndex;
  }

  public Rectangle2D.Double getExtent() {

    return extent;
  }

  public int getRecordCount() {

    return recordCount;
  }

  public int getWordLength() {

    return fileLength;
  }

  public int getShapeType() {

    return shapeType;
  }

  public DBaseFile getTable() {

    return dbFile;
  }

  public DataInput readShapeRecord(int i, int skipBytes) throws IOException {
    
    if (recordIndex == i) {
      recordStream.reset();
      recordStream.skip(skipBytes);
      return (DataInput) new DataInputStream(recordStream);
    }

    recordIndex = i;
    int indexPos = FILE_HEADER_LENGTH + (i - 1) * 8;
    
    // if (Debug.verbose)
    //  System.err.println("Seeking shape index " + (indexOffset + indexPos));

    shapeIndex.seek(indexOffset + indexPos);
    int offset = shapeIndex.readInt() * 2 + RECORD_HEADER_LENGTH;
    int length = shapeIndex.readInt() * 2;

    // if (Debug.verbose)
      // System.err.println(path + ":Seeking shape to " + offset);

    shapeFile.seek(offset);

    byte[] recordBuffer = new byte[length];
    shapeFile.read(recordBuffer);

    recordStream = new ByteArrayInputStream(recordBuffer);
    recordStream.skip(skipBytes);

    return (DataInput) new DataInputStream(recordStream);
  }

  protected Rectangle2D.Double readBounds(DataInput in) throws IOException {

    double Xmin = LittleEndian.toDouble(in.readLong());
    double Ymin = LittleEndian.toDouble(in.readLong());
    double Xmax = LittleEndian.toDouble(in.readLong());
    double Ymax = LittleEndian.toDouble(in.readLong());
    double width = Xmax - Xmin;
    double height = Ymax - Ymin;

    return new Rectangle2D.Double(Xmin, Ymax, width, height);
  }

  public void createTesselation() throws IOException {

    double aspectRatio = extent.width / extent.height; 
    double rows = Math.sqrt(recordCount / aspectRatio);
    double cols = recordCount / rows;
    double cellSize = Math.max(extent.height / rows, extent.width / cols);

    createTesselation(cellSize);
  }

  public void createTesselation(double cellSize) throws IOException {

    MemoryTesselation index = new MemoryTesselation(extent, cellSize);

    for (int i = 1; i <= recordCount; i++) {
      Rectangle2D.Double bounds = getBounds(i);
      if (bounds != null)
        index.add(i, bounds);
    }

    index.write(path);
  }

  public ArrayList selectByRect(Rectangle2D.Double rect) throws IOException {

    ArrayList selection = new ArrayList();

    if (! Tesselation.intersects(extent, rect))
      return selection;

    for (Enumeration e = globalIndex.iterate(rect); e.hasMoreElements();) {
      int[] cell = (int[]) e.nextElement();
      for (int k = 0; k < cell.length; k++) {
        Rectangle2D.Double bounds = getBounds(cell[k]);
        if (Tesselation.intersects(rect, bounds)) {
          Integer recNo = new Integer(cell[k]);
          if (selection.indexOf(recNo) == -1)
            selection.add(new Integer(cell[k]));
        }
      }
    }

    return selection;
  }

  public boolean iterate(Rectangle2D.Double selectExtent) throws IOException {

    if (! Tesselation.intersects(extent, selectExtent))
      return false;
    this.selectExtent = selectExtent;
    cellEnumeration = globalIndex.iterate(selectExtent);
    if (! cellEnumeration.hasMoreElements())
      return false;

    cellArray = (int[]) cellEnumeration.nextElement();
    cellIndex = 0;
    recordSet = new BitSet(recordCount + 1);

    return true;
  }

  public boolean hasMoreRecords() {

    if (cellIndex < cellArray.length) return true;
    if (cellEnumeration.hasMoreElements()) {
      cellArray = (int[]) cellEnumeration.nextElement();
      cellIndex = 0;
      return true;
    }

    return false;
  }

  public int nextRecord() throws IOException {

    while (hasMoreRecords()) {

      int r = cellArray[cellIndex++];
      if (recordSet.get(r)) continue;
      Rectangle2D.Double bounds = getBounds(r);

      if (Tesselation.intersects(selectExtent, bounds)) {
        recordSet.set(r);
        return r;
      }
    }

    return -1;
  }

  public void close() throws IOException {

    closeShape();
    closeTable();
  }

  public void closeShape() throws IOException {

    if (shapeFile != null) shapeFile.close();
    if (shapeIndex != null) shapeIndex.close();
    if (globalIndex != null) globalIndex.close();
  }

  public void closeTable() throws IOException {

    if (dbFile != null) dbFile.close();
  }

  public abstract Rectangle2D.Double getBounds(int r) throws IOException;
  public abstract Object getShape(int r) throws IOException;
  public abstract Object getShape(int r, AffineTransform transform) 
    throws IOException;
  public abstract Object getCoords(Area viewPort);
  public abstract Integer identify(double x, double y, double tolerance)
    throws IOException;

  public static void main(String[] args) {

    try {

      ShapeFile shapeFile = create(args[0]);
      int r = shapeFile.getRecordCount();
      System.out.println("Record count: " + shapeFile.getRecordCount());
      Object shape = shapeFile.getShape(r);
      System.out.println("Last record read successfully");

      System.out.println("Opening index...");
      FileTesselation tess = new FileTesselation(args[0]);
      System.out.println(tess);

    } catch (Exception e) {
      e.printStackTrace();
      System.exit(0);
    }

  }

  public final static int VERSION = 1000;

  public final static int NULL_SHAPE = 0;
  public final static int POINT = 1;
  public final static int ARC = 3;
  public final static int POLYGON = 5;
  public final static int MULTIPOINT = 8;

  public final static int NULL_SHAPE_FLOAT = 100;
  public final static int POINT_FLOAT = 101;
  public final static int ARC_FLOAT = 103;
  public final static int POLYGON_FLOAT = 105;

  protected final static int FILE_HEADER_LENGTH = 100;

  protected int RECORD_HEADER_LENGTH = 8;

  private BitSet recordSet;
  private Enumeration cellEnumeration;
  private int[] cellArray;
  private int cellIndex;
  private Rectangle2D.Double selectExtent;

  // cache last record to avoid reading the shape twice when rendering 
  // (checking the extent and then retrieving the entire shape)
  private int recordIndex = -1;
  private ByteArrayInputStream recordStream;

  protected String path;
  protected int fileCode;
  protected int fileLength;
  protected int version;
  protected int shapeType;

  protected Rectangle2D.Double extent;

  protected int recordCount;
  protected Tesselation globalIndex = null;

  protected RandomAccessFile shapeFile = null;
  protected RandomAccessFile shapeIndex = null;
  protected long indexOffset = 0;
  protected DBaseFile dbFile = null;
}
