/**

Copyright (C) 1999 Karl Goldstein

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

**/

package pms.shape;

import java.awt.*;
import java.awt.geom.*;
import java.io.*;
import java.util.*;

import pms.tools.StringTools;

public class MemoryTesselation extends Tesselation {

  private int[][][] tesselation;

  protected MemoryTesselation() {}

  public MemoryTesselation(Rectangle2D.Double extent, double cellSize,
    int cols, int rows) {

    this.extent = (Rectangle2D.Double) extent.clone();
    this.cellSize = cellSize;

    this.cols = cols;
    this.rows = rows;

    tesselation = new int[cols][][];
  }

  public MemoryTesselation(Rectangle2D.Double extent, double cellSize) {

    this.extent = (Rectangle2D.Double) extent.clone();
    this.cellSize = cellSize;

    cols = (int) Math.ceil(extent.width / cellSize);
    rows = (int) Math.ceil(extent.height / cellSize);

    tesselation = new int[cols][][];
  }

  public void add(int record, Rectangle2D.Double bounds) {

    if (bounds == null) return;

    Point min = pointToTesselation(bounds.x, bounds.y - bounds.height);
    if (min.x < 0) min.x = 0; 
    if (min.y < 0) min.y = 0;

    Point max = pointToTesselation(bounds.x + bounds.width, bounds.y);
    if (max.x >= cols) max.x = cols - 1; 
    if (max.y >= rows) max.y = rows - 1; 

    for (int j = min.x; j <= max.x; j++) {
      for (int k = min.y; k <= max.y; k++) {

        if (tesselation[j] == null)
          tesselation[j] = new int[rows][];

        int[] cell = tesselation[j][k];
        if (cell == null)
          cell = new int[0];

        int[] refit = new int[cell.length + 1];
        for (int l = 0; l < cell.length; l++)
          refit[l] = cell[l];
        refit[cell.length] = record;
        tesselation[j][k] = refit;
      }
    }
  }
 
  public void write(String path) throws IOException {

    DataOutputStream indexStream = new DataOutputStream(
      new FileOutputStream(StringTools.setExtension(path, "tsx")));
    writeHeader(indexStream);

    DataOutputStream mainStream = new DataOutputStream(
      new FileOutputStream(StringTools.setExtension(path, "tes")));

    int offset = 0;

    for (int i = 0; i < rows; i++) {
      for (int j = 0; j < cols; j++) {
        indexStream.writeInt(offset);
        int len = writeCell(j, i, mainStream);
        indexStream.writeInt(len);
        offset += len * 4;
      }
    }

    indexStream.close();
    mainStream.close();
  }

  private void writeHeader(DataOutputStream out) throws IOException {

    out.writeDouble(extent.x);
    out.writeDouble(extent.y);
    out.writeDouble(extent.width);
    out.writeDouble(extent.height);
    out.writeDouble(cellSize);
    out.writeInt(rows);
    out.writeInt(cols);
  }

  private int writeCell(int col, int row, DataOutputStream out)
    throws IOException {

    int offset = 0;
    if (tesselation[col] != null) {
      int[] cell = tesselation[col][row];
      if (cell != null) {
        for (int k = 0; k < cell.length; k++) {
          out.writeInt(cell[k]);
          offset++;
        }
      }
    }

    return offset;
  }

  public int[] getCell(int col, int row) throws IOException {

    int[] cell = new int[0];
    int[][] cellColumn;

    try {
      cellColumn = tesselation[col];
    } catch (ArrayIndexOutOfBoundsException e) {
      throw new IOException("Column out of bounds at row " + row + ": " + 
        col + " (of " + tesselation.length + " columns");
    }

    try {
      if (cellColumn != null) {
        int[] c = cellColumn[row];
        if (c != null) cell = c;
      }
    } catch (ArrayIndexOutOfBoundsException e) {
      throw new IOException("Row out of bounds at column " + col + ": " + 
        row + " (of " + cellColumn.length + " rows");
    }

    return cell;
  }

  public void close() throws IOException {} 
}
