/**

Copyright (C) 1999-2001 Karl Goldstein

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

**/

package pms.server;

import java.awt.*;
import java.io.*;
import java.net.URLDecoder;
import java.util.*;
import java.util.zip.*;

import javax.servlet.http.*;

import pms.*;
import pms.tools.*;
import pms.template.*;

import org.apache.log4j.Category;

public class MapApplet extends HttpServlet {

  private static Category log = 
    Category.getInstance(MapApplet.class.getName());

  public void doPost(HttpServletRequest request, 
		    HttpServletResponse response) throws IOException {
    doGet(request, response);
  }

  public void doGet(HttpServletRequest request, 
		    HttpServletResponse response) throws IOException {

    log.info("Applet request: " + URLDecoder.decode(request.getQueryString()));

    String targetPath = getServletContext().getRealPath("");

    HashMap queryParams = RequestTools.getParameters(request);

    CartaProperties props = new CartaProperties(targetPath, queryParams);

    doResponse(response, props, targetPath);
  }

  void doResponse(HttpServletResponse response,
		  CartaProperties props, 
		  String targetPath) throws IOException {

    long start = System.currentTimeMillis();

    Carta carta = new Carta(props);

    log.info("Construct carta: " + (System.currentTimeMillis() - start));
    start = System.currentTimeMillis();

    log.info("Render carta: " + System.currentTimeMillis());
    
    carta.render();

    log.info("Render carta: " + System.currentTimeMillis());

    log.info("Render carta: " + (System.currentTimeMillis() - start));
    start = System.currentTimeMillis();

    ViewFrame.drawArrows(carta.getImage(), null);

    log.info("Draw arrows: " + (System.currentTimeMillis() - start));
    start = System.currentTimeMillis();

    ContentBuffer imageOut = new ContentBuffer();
    carta.writeImage(imageOut);
    imageOut.close();

    log.info("Write image: " + (System.currentTimeMillis() - start));
    start = System.currentTimeMillis();

    ContentBuffer out = new ContentBuffer();

    DataOutputStream dataOut = new DataOutputStream(out);
    dataOut.writeInt(imageOut.size());
    imageOut.writeTo(dataOut);
    carta.flush();

    GZIPOutputStream zipOut = new GZIPOutputStream(out);
    PrintStream printOut = new PrintStream(zipOut);
    printOut.print("View: " + props.getViewPath() + CRLF);
    printOut.print("Views: " + props.getViewsParam() + CRLF);
    printOut.print("Title: " + props.getTitle() + CRLF);
    printOut.print("Extent: " + props.getExtentParam() + CRLF);
    printOut.print(CRLF);

    for (Iterator i = carta.getLayers().iterator(); i.hasNext();) {
      Layer layer = (Layer) i.next();
      addLayer(printOut, layer, targetPath);
    }

    log.info("Write info: " + (System.currentTimeMillis() - start));
    start = System.currentTimeMillis();
    
    printOut.close();
    zipOut.close();
    
    dataOut.close();

    Debug.printMemory();

    response.setHeader("Content-type", getResponseType());
    response.setIntHeader("Content-length", out.size());

    out.writeTo(response.getOutputStream());
  }

  public String getResponseType() {

    return "*/*";
  }

  private void addLayer(PrintStream printOut, Layer layer, String targetPath) 
    throws IOException {

    printOut.print("LAYER" + CRLF);
    LayerProperties layerProps = layer.getProperties();

    String layerPath = 
      StringTools.getRelativePath(targetPath, layerProps.getPath());
    //    System.err.println("Adding " + layerPath);
    printOut.print("Path: " + layerPath + CRLF);
    printOut.print("Type: " + layerProps.getGenericType() + CRLF);

    String labelParam = layerProps.getLabelParam();
    if (labelParam != null)
      printOut.print("Label: " + labelParam + CRLF);
      
    String reportParam = layerProps.getReportParam();
    if (reportParam != null)
      printOut.print("Reports: " + reportParam + CRLF);

    String chartParam = layerProps.getChartParam();
    if (chartParam != null)
      printOut.print("Chart: " + chartParam + CRLF);

    Legend legend = layerProps.getLegend();
    if (legend != null) {
      printOut.print("Title: " + layerProps.getTitle() + CRLF);
      printOut.print("Subtitle: " + layerProps.getSubtitle() + CRLF);
      printOut.print("Classes: " + legend.getClassesParam() + CRLF);
    }

    printOut.print(CRLF);
    addFeatures(printOut, layer);
    //    System.err.println("Handled " + layerPath);
  }

  private void addFeatures(OutputStream out, Layer layer)
    throws IOException {

    DataOutputStream dataOut = new DataOutputStream(out);

    Layer.FeatureTable featureTable = layer.getFeatureTable();
    if (featureTable == null) {
      dataOut.writeInt(0);
      return;
    }

    dataOut.writeInt(featureTable.count());
    dataOut.writeBytes(featureTable.getHeader() + CRLF);

    for (int r = 0; r < featureTable.count(); r++) {
      writeCoords(featureTable.getCoords(r), dataOut);
      writeAttributes(featureTable.getAttributes(r), dataOut);
    }
    //    dataOut.close();
  }
 
  private void writeCoords(Object coords, DataOutputStream dataOut)
    throws IOException {

    if (coords instanceof Point) {

      Point point = (Point) coords;
      dataOut.writeShort((short) point.x);
      dataOut.writeShort((short) point.y);

    } else {

      ArrayList rings = (ArrayList) coords;
      dataOut.writeShort(rings.size());

      for (int i = 0; i < rings.size(); i++) {

        ArrayList ring = (ArrayList) rings.get(i);
        dataOut.writeShort(ring.size());

        for (int j = 0; j < ring.size(); j++) {
          Point vertex = (Point) ring.get(j);
          dataOut.writeShort(vertex.x);
          dataOut.writeShort(vertex.y);
        }
      }
    }
  }

  private void writeAttributes(String[] attribs, DataOutputStream dataOut)
    throws IOException {

    String recString = "";

    for (int i = 0; i < attribs.length; i++) {
     if (i > 0) recString += "|";
     recString += attribs[i];
    }

    dataOut.writeBytes(recString + CRLF);
    //    System.err.println(recString);
  }

  private static final String CRLF = "\r\n";
}












