package pms;

import java.awt.*;
import java.io.*;
import java.util.*;

import pms.tools.*;

public class ViewProperties extends Properties {

  public int getMaxScale()         { return maxScale; }
  public int getMinScale()         { return minScale; }
  public Extent getExtent()        { return extent; }
  public String getUnits()         { return units; }
  public String getTitle()         { return title; }
  public ArrayList getLayerPaths() { return layerPaths; }
  public Color getBackgroundColor() { return backgroundColor; }

  public String getViewsParam() { return viewsParam; }

  public Legend[] getLegends() { return legends; }
  public int[] getScales() { return scales; }

  protected ViewProperties() {}

  protected ViewProperties(String path) throws IOException {

    setPath(path);
  }

  protected void setPath(String path) throws IOException {

    HashMap topMap = parseFile(path);
    parseMain(getOneMap("", topMap, "MAIN", true));
    parseLayers(getOneMap("", topMap, "LAYERS", true));
    parseViews(getOneMap("", topMap, "VIEWS", false));
    parseScales(getOneMap("", topMap, "SCALES", true));
  }

  private void parseMain(HashMap params) throws FormatException {
 
    if (params == null)
      throw new FormatException("MAIN block must be declared in view");
  
    String backgroundParam = 
      getOneString("MAIN", params, "BACKGROUND_COLOR", null, false); 
    if (backgroundParam != null)
      backgroundColor = Symbol.parseColor(backgroundParam); 

    units = getOneString("MAIN", params, "UNITS", null, true); 

    title = getOneString("MAIN", params, "TITLE", "", true); 

    extentLayer = getOneString("MAIN", params, "EXTENT_LAYER", null, false);
  }

  private void parseLayers(HashMap params) throws IOException {
    
    if (params == null)
      throw new FormatException("LAYERS block must be declared in view");

    ArrayList paths = getAllStrings("LAYERS", params, "LAYER", true);
    legends = new Legend[paths.size()];
    int index = 0;

    for (Iterator i = paths.iterator(); i.hasNext();) {
      String layerPath = (String) i.next();
      String absPath = StringTools.resolvePath(path, layerPath);
      LayerProperties layerProps = LayerProperties.retrieve(absPath);
      legends[index++] = layerProps.getLegend();
      
      if ((extentLayer != null) && (extentLayer.equals(layerPath)))
        extent = new Extent(layerProps.getExtent(), units);
	
      calcGlobals(layerProps);
      layerPaths.add(absPath);
    }
  }

  private void parseViews(HashMap params) throws IOException {
    
    if (params == null)
      return;

    viewPaths = getAllStrings("VIEWS", params, "VIEW", true);

    for (Iterator i = viewPaths.iterator(); i.hasNext();) {
      String viewPath = (String) i.next();
      viewPath = StringTools.resolvePath(path, viewPath);
      ViewProperties viewProps = ViewProperties.retrieve(viewPath);
       
      viewOptions.add(viewPath);
      viewsParam += viewPath + "|" + viewProps.getTitle();
      if (i.hasNext()) viewsParam += "|";
    }
  }

  private void parseScales(HashMap params) throws IOException {

    if (params == null)
      throw new FormatException("SCALES block must be declared in view");

    ArrayList scaleParams = getAllStrings("SCALES", params, "SCALE", true);
    if (scaleParams.size() > 10)
      throw new FormatException("More than 10 scales specified in view");

    scales = new int[scaleParams.size()];
    int index = 0;

    for (Iterator i = scaleParams.iterator(); i.hasNext();) {
      String scaleStr = (String) i.next();
      scales[index++] = StringTools.toInt(scaleStr);
    }
  }

  private void calcGlobals(LayerProperties layerProps) throws IOException {

    if (layerProps.getMinScale() < minScale) 
      minScale = layerProps.getMinScale();
    if (layerProps.getMaxScale() > maxScale) 
      maxScale = layerProps.getMaxScale();

    if (extentLayer == null) return;

    if (extent == null) 
      extent = new Extent(layerProps.getExtent(), units);
    else
      extent.union(layerProps.getExtent());
  }

  public static ViewProperties retrieve(String path) 
    throws IOException {
    
    ViewProperties props = (ViewProperties) cache.get(path);
    if (props == null) {
      props = new ViewProperties();
      cache.put(path, props);
      props.setPath(path);
    }

    return props;
  }

  public static synchronized void flushCache() {

    cache.clear();
  }

  public static void main(String[] args) {

    if (args.length == 0) {
      System.err.println(
        "Usage: java Map.ViewProperties [view file names...]");
      System.exit(0);
    }

    try {
      for (int i = 0; i < args.length; i++)
        retrieve(args[i]);
    } catch (IOException e) {
      e.printStackTrace();
    }
    System.exit(0);
  }

  private Color backgroundColor = new Color(255, 255, 255);
  private String units;
  private String title;
  private String extentLayer;
  private Legend[] legends;
  private ArrayList layerPaths = new ArrayList();

  private ArrayList viewPaths = new ArrayList();
  private ArrayList viewOptions = new ArrayList();
  private String viewsParam = "";

  private Extent extent;

  private int[] scales;

  private int minScale;
  private int maxScale;

  private static HashMap cache = new HashMap();
}
