/**

Copyright (C) 1999 Karl Goldstein

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

**/

package pms;

import java.awt.*;
import java.awt.font.*;
import java.awt.geom.*;

public class TextLabel implements Label {

  public void setMask(Label[] mask) {

    this.mask = mask;
  }

  public void setText(String text, Font font) {
   
    this.text = text;
    this.font = (font == null) ? defaultFont : font;
  }
    
  public void setLocation(Object shape) {

    if (shape instanceof Point2D.Float) {
      Point2D.Float p = (Point2D.Float) shape;
      this.x = (int) p.x;
      this.y = (int) p.y;
    }
  }

  public boolean nextExtent() {
    if ((i == extents.length - 1) || (extents[i + 1] == null)) 
      return false;
    i++;
    return true;
  }

  /**
    Returns a label already in the mask that intersects the
    current placement of this label.
  **/
  public Label getOverlappingLabel() {
    
    try {
      for (int r = 0; r < mask.length; r++) {
        if (mask[r] == null) break;
        if (extents[i].intersects(mask[r].getExtent())) 
          return mask[r];
      }
    } catch (ArrayIndexOutOfBoundsException e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
    Searches for a placement that does not intersect the free extent
    and does not overlap previously placed labels.  This allows for
    moving the placement of a higher-priority label in an attempt
    to make room for a label placement at free extent.
  **/
  public boolean freeExtent(Rectangle freeExtent) {
    
    try {
      for (int r = 0; r < extents.length; r++) {
        if (extents[r] == null) break;
        if (! extents[r].intersects(freeExtent)) {
          Label overlappingLabel = getOverlappingLabel();
          if (overlappingLabel == null) {
            i = r;
            return true;
          }
        }
      }
    } catch (ArrayIndexOutOfBoundsException e) {
      e.printStackTrace();
    }

    return false;
  }

  /**
    Tests (not very efficiently) all the possible label placements
    around the anchor point and saves the visible ones for mask
    calculation
  **/
  public void getVisibleExtents(Rectangle visExtent, FontRenderContext frc) {

    LineMetrics fm = font.getLineMetrics(text, frc);
    Rectangle2D bnd = font.getStringBounds(text, frc);
    int textWidth = (int) bnd.getWidth();
    int textHeight = (int) bnd.getHeight();

    int extentWidth = PAD + textWidth + PAD * 2;
    int extentHeight = textHeight + PAD * 2;

    int yHigh = y - textHeight - PAD;
    int yLow = y - PAD;
    int xRight = x - PAD - textWidth - PAD;
    int xLeft = x - PAD;

    i = -1;
    Rectangle r;

    r = new Rectangle(xLeft, yHigh, extentWidth, extentHeight);
    if (visExtent.contains(r)) {
      extents[++i] = r;
      anchors[i] = new Point(r.x + PAD, r.y + PAD + (int) fm.getAscent());
    }

    r = new Rectangle(xRight, yHigh, extentWidth, extentHeight);
    if (visExtent.contains(r)) {
      extents[++i] = r;
      anchors[i] = new Point(r.x + PAD * 2, r.y + PAD + (int) fm.getAscent());
    }
    
    r = new Rectangle(xLeft, yLow, extentWidth, extentHeight);
    if (visExtent.contains(r)) {
      extents[++i] = r;
      anchors[i] = new Point(r.x + PAD, r.y + PAD + (int) fm.getAscent());
    }

    r = new Rectangle(xRight, yLow, extentWidth, extentHeight);
    if (visExtent.contains(r)) {
      extents[++i] = r;
      anchors[i] = new Point(r.x + PAD, r.y + PAD + (int) fm.getAscent());
    } 

    i = -1;
  }

  public Point getTextAnchor() {

    return anchors[i];
  }

  public Rectangle getExtent() {

    return extents[i];
  }

  /**
    Draws a label with a white halo
  **/
  public void draw(Graphics g) {

    int x = anchors[i].x;
    int y = anchors[i].y;

    g.setFont(font);
    g.setColor(Color.white);
    g.drawString(text, x, y-1);
    g.drawString(text, x, y+1);
    g.drawString(text, x-1, y);
    g.drawString(text, x+1, y);
    g.drawString(text, x-1, y-1);
    g.drawString(text, x-1, y+1);
    g.drawString(text, x+1, y+1);
    g.drawString(text, x+1, y-1);

    g.fillOval(this.x - 3, this.y - 3, 6, 6);

    g.setColor(Color.black);
    g.drawString(text, x, y);
    g.drawOval(this.x - 3, this.y - 3, 6, 6);
  }

  /**
     Describes a label and its placement 
  **/
  public String toString() {

    String s =  "label: " + text + "  " + "index: " + i + "  ";

    if (i > -1) {
      s += "anchor: " + anchors[i].x + "," + anchors[i].y + "  " +
      "extent: " + extents[i].x + "," + extents[i].y + "," +
        extents[i].width + "," + extents[i].height;
    } else {
      s += "(no placement)";
    }

    return s;
  }

  public Label[] mask;

  public String text;
  public Font font;
  public int x;
  public int y;

  public Rectangle[] extents = new Rectangle[4];
  public Point[] anchors = new Point[4];
  public int i = -1;

  public static Font defaultFont = new Font("SansSerif", Font.PLAIN, 10);
  public static int PAD = 5;
}

