package pms;

import java.awt.geom.*;
import java.io.*;
import java.lang.reflect.*;
import java.net.*;
import java.util.*;

import pms.shape.*;
import pms.table.*;
import pms.tools.*;

public class LayerProperties extends Properties {

  public int getMaxScale()              { return maxScale; }
  public int getMinScale()              { return minScale; }
  public String getTitle()              { return title; }
  public String getSubtitle()           { return subtitle; }
  public String getDescription()        { return description; }
  public String getClassField()         { return classField; }
  public Symbol getSymbol()             { return defaultSymbol; }
  public String getShapePath()          { return shapePath; }
  public Rectangle2D.Double getExtent() { return extent; }
  public boolean getLegendVisibility()   { return legendVisibility; }

  public ArrayList getAttributeFields()  { return attributeFields; }

  public boolean hasAttributes()  { 
    return attributeFields.isEmpty() ? false : true; 
  }

  public boolean isTypePoint() { 
  
    return ((type == ShapeFile.POINT) || 
            (type == ShapeFile.POINT_FLOAT)) ? true : false;
  }

  public boolean isTypePolygon() { 
  
    return ((type == ShapeFile.POLYGON) || 
            (type == ShapeFile.POLYGON_FLOAT)) ? true : false;
  }

  public String getGenericType() {

    String genericType = "line";
    if (isTypePoint()) genericType = "point";
    if (isTypePolygon()) genericType = "polygon";

    return genericType;
  }

  public String getLabelField() {

    return labelField;
  }

  public String getLabelPriorityField() {

    return labelPriorityField;
  }

  public String getLabelParam() {

    if (labelDisplay != LABEL_DISPLAY_DYNAMIC) return null;
    if (labelField == null) return null;

    return labelPrefix + "|" + labelField;
  }

  public boolean hasStaticLabels() {

    return (labelClass == null) ? false : true;
  }

  public Label getLabel() throws FormatException {

    Label label = null; 
    try {

      label = (Label) labelClass.newInstance();

    } catch (Exception e) {
     throw new FormatException("Unable to instantiate label class.");
    }
    return label;
  }

  public Legend getLegend() { 

    return legendVisibility ? legend : null; 
  }

  public boolean hasReports() {

    return (reports.size() > 0) ? true : false;
  }

  public Report[] getReports(Layer layer, int record) throws IOException {
    
    Report[] featureReports = new Report[reports.size()];

    for (int i = 0; i < reports.size(); i++) {
      ReportProperties reportProps = (ReportProperties) reports.get(i);
      featureReports[i] = reportProps.getReport(layer, record);
    }

    return featureReports;
  }

  public String getChartParam() {

    return (chart == null) ? null : chart.getParam();
  }

  public String getReportParam() {

    String reportParam = "";

    for (int i = 0; i < reports.size(); i++) {
    
      if (i > 0) reportParam += "|";
      ReportProperties reportProps = (ReportProperties) reports.get(i);
      reportParam += reportProps.getParam();
    }

    return reportParam;
  }

  public boolean getVisibility(int scale) {

    return (scale >= minScale) && (scale <= maxScale);
  }

  public Integer getClassIndex(String classValue) {

    return (Integer) classes.get(classValue);
  }

  public Symbol getSymbol(String classValue) {

    Integer index = (Integer) classes.get(classValue);
    if (index == null) return defaultSymbol;

    return (Symbol) classSymbols.get(index.intValue());
  }

  public Query getQuery(DBaseFile keyFile, ArrayList selection) 
    throws IOException {

    if (sql == null)
      throw new IOException("Attempt to query nonexistent sql data source");

    return sql.getQuery(keyFile, selection);
  }

  public boolean dataSourceIsSQL() {

    return (sql == null) ? false : true;
  }

  public LayerProperties(String path) throws IOException {

    HashMap topMap = parseFile(path);

    parseMain(getOneMap("", topMap, "MAIN", true));
    parseSQL(getOneMap("", topMap, "SQL", false));

    parseScale(getOneMap("", topMap, "SCALE", false));
    parseLabel(getOneMap("", topMap, "LABEL", false));
    parseAlternateShapes(getOneMap("", topMap, "ALTERNATE_SHAPES", false));
    parseClassify(getOneMap("", topMap, "CLASSIFY", false));
    parseReports(getOneMap("", topMap, "REPORTS", false));
    parseChart(getOneMap("", topMap, "CHART", false));
  }
  
  private void parseMain(HashMap params) throws FormatException {
    
    shapePath = getOneString("MAIN", params, "PATH", null, true);
    shapePath = StringTools.resolvePath(path, shapePath);
    peekShape();

    title = getOneString("MAIN", params, "TITLE", "", false);
    subtitle = getOneString("MAIN", params, "SUBTITLE", "", false);
    description = getOneString("MAIN", params, "DESCRIPTION", "", false);
    String legendParam = 
      getOneString("MAIN", params, "LEGEND", "HIDE", false);
    legendVisibility = legendParam.equals("SHOW") ? true : false;

    HashMap symbolParams = getOneMap("MAIN", params, "SYMBOL", false);
    if (symbolParams != null) {
      defaultSymbol = Symbol.create(type, this, symbolParams);
    } else {
      defaultSymbol = Symbol.createDefault(type);
    }
  }

  private void parseScale(HashMap params) throws FormatException {

    if (params == null) return;

    String minParam = getOneString("SCALE", params, "MIN", "0", false);
    minScale = StringTools.toInt(minParam);
    if (minScale < 0)
      throw new FormatException(
        path, "SCALE", "Minimum scale is less than zero");

    String maxParam = getOneString("SCALE", params, "MAX", 
      Integer.toString(Integer.MAX_VALUE), false);
    maxScale = StringTools.toInt(maxParam);
    if (maxScale <= minScale)
      throw new FormatException(path, "SCALE", 
        "Maximum scale is less than minimum scale");
  }

  private void parseLabel(HashMap params) throws FormatException {

    if (params == null) return;

    labelField = getOneString("LABEL", params, "FIELD", null, true);
    peekAttribute("LABEL", labelField, false);

    labelPrefix = getOneString("LABEL", params, "PREFIX", "-", false);

    String displayParam = 
      getOneString("LABEL", params, "DISPLAY", "DYNAMIC", false);
    Object displayEnum = StringTools.toEnumeration(
      this, "LABEL_DISPLAY_" + displayParam.toUpperCase());
    labelDisplay = ((Integer) displayEnum).intValue();
    if (labelDisplay == LABEL_DISPLAY_DYNAMIC) {
      attributeFields.add(labelField);
      return;
    }
    
    String className = getOneString("LABEL", params, "CLASS", "Text", false);
    className = StringTools.toTitleCase(className);
    try {
      labelClass = Class.forName("pms." + className + "Label");
    } catch (ClassNotFoundException e) {
      throw new FormatException("No such label class: " + className);
    }

    labelPriorityField = 
      getOneString("LABEL", params, "PRIORITY", null, false);
    if (labelPriorityField == null)
      peekAttribute("LABEL", labelPriorityField, false);
  }

  private void parseAlternateShapes(HashMap params) throws FormatException {

    if (params == null) return;

    alternateShapePath = 
      getOneString("ALTERNATE_SHAPES", params, "PATH", null, true);
    alternateShapePath = StringTools.resolvePath(path, alternateShapePath);

    alternateFromKey = 
      getOneString("ALTERNATE_SHAPES", params, "FROM_KEY", null, true);
    alternateToKey =
      getOneString("ALTERNATE_SHAPES", params, "TO_KEY", null, true);
  }

  private void parseSQL(HashMap params) throws IOException {

    if (params == null) return;
    sql = new SQLProperties(this, params);
  }

  private void parseClassify(HashMap params) throws FormatException {

    if (params == null) {
      legend = new Legend(this, 1);
      legend.addClass(defaultSymbol.getLegendSymbol(), "-");
      // System.err.println("Added legend for " + title);
      return;
    }

    classField = 
      getOneString("CLASSIFY", params, "FIELD", null, true);
    peekAttribute("CLASSIFY", classField, false);

    ArrayList classBlocks = getAllMaps("CLASSIFY", params, "CLASS", true);
    legend = new Legend(this, classBlocks.size());

    for (Iterator i = classBlocks.iterator(); i.hasNext();) {
      HashMap classBlock = (HashMap) i.next();
      String values = getOneString("CLASS", classBlock, "VALUE", null, true);
      String label = getOneString("CLASS", classBlock, "LABEL", null, true);

      HashMap symbolParams = getOneMap("CLASS", classBlock, "SYMBOL", true);
      Symbol symbol = Symbol.create(type, this, symbolParams);
      legend.addClass(symbol.getLegendSymbol(), label);

      StringTokenizer s = new StringTokenizer(values, "|");
      while (s.hasMoreTokens()) {
        String value = s.nextToken();
        // System.err.println("Class value |" + value + "|");
        Integer index = new Integer(classes.size());
        classes.put(value, index);
        classSymbols.add(symbol);
      }
    }
  }

  private void parseReports(HashMap params) throws FormatException {

    if (params == null) {
      // System.err.println("No reports specified for " + path);
      return;
    }

    // System.err.println("Parsing reports for " + path);

    ArrayList reportBlocks = 
      getAllMaps("REPORTS", params, "REPORT", true);
    for (Iterator i = reportBlocks.iterator(); i.hasNext();) {

      HashMap reportBlock = (HashMap) i.next();
      reports.add(new ReportProperties(this, reportBlock));
    }
  }

  private void parseChart(HashMap params) throws FormatException {

    if (params == null) return;

    chart = new ChartProperties(this, params);
  }

  public void peekAttribute(String mapName, String field, 
    boolean addToFeatureSet) throws FormatException {

    // System.err.println("Checking " + field + " in " + path);
    if (sql != null) {
      sql.addField(mapName, field);
    } else {
      field = field.toUpperCase();
      Integer column = (Integer) shapeFields.get(field);
      if (column == null)
        throw new FormatException(path, mapName, 
          "Field " + field + " not found in attribute table(s)");
    }

    if (addToFeatureSet && (! attributeFields.contains(field)))
      attributeFields.add(field);
  }

  private void peekShape() throws FormatException {

    ShapeFile shapeFile;
    
    try {

      shapeFile = ShapeFile.create(shapePath);
      type = shapeFile.getShapeType();
      extent = shapeFile.getExtent();
      DBaseFile dbFile = shapeFile.getTable();
      for (int i = 1; i <= dbFile.getColumnCount(); i++) {
        shapeFields.put(dbFile.getColumnName(i), new Integer(i));
      }
      shapeFile.close();

    } catch (IOException e) {
      e.printStackTrace();
      throw new FormatException(e.getMessage());
    }
  }

  public static synchronized void flushCache() {
    cache.clear();
  }

  public static synchronized LayerProperties retrieve(String path) 
    throws IOException {
    
    LayerProperties props = (LayerProperties) cache.get(path);
    if (props == null) {
      props = new LayerProperties(path);
      cache.put(path, props);
    }

    return props;
  }

  public static void main(String[] args) {

    if (args.length == 0) {
      System.err.println(
        "Usage: java Map.LayerProperties [layer file names...]");
      System.exit(0);
    }

    try {
      for (int i = 0; i < args.length; i++)
        retrieve(args[i]);
    } catch (IOException e) {
      e.printStackTrace();
    }
    System.exit(0);
  }

  private int type;
  private String shapePath;
  private HashMap shapeFields = new HashMap();
  private Rectangle2D.Double extent;
  private String title;
  private String subtitle;
  private String description;

  private int minScale = 0;
  private int maxScale = Integer.MAX_VALUE;

  private String labelField;
  private String labelPrefix;
  private int labelDisplay;
  private Class labelClass;
  private String labelPriorityField;

  private String alternateShapePath;
  private String alternateFromKey;
  private String alternateToKey;

  private SQLProperties sql;

  private HashMap clientAttributes = new HashMap();

  private Symbol defaultSymbol;
  private String classField;
  private HashMap classes = new HashMap();
  private ArrayList classSymbols = new ArrayList();
  private Legend legend;
  private boolean legendVisibility;

  private ArrayList attributeFields = new ArrayList();

  private ArrayList reports = new ArrayList();
  private ChartProperties chart;

  protected static HashMap cache = new HashMap();
  public final static int LABEL_DISPLAY_DYNAMIC = 1;
  public final static int LABEL_DISPLAY_STATIC = 2;
}
