package pms;

import java.awt.*;
import java.awt.geom.*;
import java.io.*;
import java.net.*;
import java.util.*;

import pms.shape.*;
import pms.table.*;

public class Layer {

  public Layer(String path) throws IOException {

    props = LayerProperties.retrieve(path);
  }

  public boolean render(Graphics2D g, Extent geoExtent, Labeler labeler) 
    throws IOException {

    if (! props.getVisibility(geoExtent.getScale())) {
      //      System.err.println(
      // "Layer " + props.getPath() + " not visible.  Skipping...");
      return false;
    }
    // System.err.println("Rendering layer " + props.getPath() + "...");

    viewPort = geoExtent.getViewPort();
    if (props.hasStaticLabels()) this.labeler = labeler;

    try {

      shapeFile = ShapeFile.create(props.getShapePath());

      if (props.dataSourceIsSQL())
        select(g, geoExtent);
      else
        iterate(g, geoExtent);

      shapeFile.close();
      if (query != null) query.close();

    } catch (IOException e) {
      if (shapeFile != null) 
        shapeFile.close();
      if (query != null) query.close();
      e.printStackTrace();
      // throw e;
    }

    return true;
  }

  private void select(Graphics2D g, Extent geoExtent) throws IOException {

    AffineTransform transform = geoExtent.getTransform();
    ArrayList selection = shapeFile.selectByRect(geoExtent);
    if (selection.isEmpty()) return;

    query = props.getQuery(shapeFile.getTable(), selection);
    if (props.hasAttributes()) featureTable = new FeatureTable();

    classify();
    label();
    for (Iterator i = selection.iterator(); i.hasNext();) {

      Integer record = (Integer) i.next();
      int r = record.intValue();

      Object shape = shapeFile.getShape(r, transform);
      if (featureTable != null) 
        featureTable.add(shapeFile.getCoords(viewPort), r);
      if (labeler != null)
        addLabel(shape, r);
      getSymbol(r).render(shape, g);
    }
  }

  private void iterate(Graphics2D g, Extent geoExtent) throws IOException {

    AffineTransform transform = geoExtent.getTransform();

    if (props.hasAttributes()) featureTable = new FeatureTable();

    if (shapeFile.iterate(geoExtent)) {

      classify();
      label();

      int r;
      while ((r = shapeFile.nextRecord()) != -1) {

        Object shape = shapeFile.getShape(r, transform);
        if (featureTable != null) 
          featureTable.add(shapeFile.getCoords(viewPort), r);
        if (labeler != null)
          addLabel(shape, r);
        getSymbol(r).render(shape, g);
      }
    }
  }

  public Report[] identify(int scale, double x, double y, double tolerance)
    throws IOException {

    Report[] reports = null;
    if (! props.getVisibility(scale))
      return null;
    if (! props.hasReports())
      return null;

    try {
      shapeFile = ShapeFile.create(props.getShapePath());
      Integer record = shapeFile.identify(x, y, tolerance);

      if (record != null) {

        if (props.dataSourceIsSQL()) {
          ArrayList keyValues = new ArrayList();
          keyValues.add(record);
          query = props.getQuery(shapeFile.getTable(), keyValues);
        }
    
        reports = props.getReports(this, record.intValue());
      }

      if (query != null) query.close();
      shapeFile.close();
    } catch (IOException e) {
      if (shapeFile != null) 
        shapeFile.close();
      if (query != null) query.close();
      throw e;
    }

    return reports;
  }

  private void classify() throws IOException {

    String classField = props.getClassField();
    if (classField != null)
      classAttribute = getAttribute(classField);
  }

  private void label() throws IOException {

    if (labeler == null) return;

    String labelField = props.getLabelField();
    if (labelField != null)
      labelAttribute = getAttribute(labelField);

    String labelPriorityField = props.getLabelPriorityField();
    if (labelPriorityField != null)
      labelPriorityAttribute = getAttribute(labelPriorityField);
  }
    
  private Symbol getSymbol(int r) throws IOException {

    Symbol symbol;

    if (classAttribute != null) {
      String classValue = classAttribute.getValue(r);
      symbol = props.getSymbol(classValue);
    } else {
      symbol = props.getSymbol();
    }

    return symbol;
  }

  public Attribute getAttribute(String field) throws IOException {

    Attribute attribute;
    if (query == null) 
      attribute = shapeFile.getTable().getAttribute(field);
    else 
      attribute = query.getAttribute(field);

    return attribute;
  }

  public FeatureTable getFeatureTable() {

    return featureTable;
  }

  public LayerProperties getProperties() {

    return props;
  }

  public AttributeSource getAttributeSource() {

    AttributeSource source = null;

    if (query != null) 
      source = query;
    else if (shapeFile != null) 
      source = shapeFile.getTable();

    return source;
  }

  private void addLabel(Object shape, int r) throws IOException {

    String text = labelAttribute.getValue(r);
  
    int priority = 0;
    if (labelPriorityAttribute != null) {
      try {
        priority = Integer.parseInt(labelPriorityAttribute.getValue(r));
      } catch (NumberFormatException e) {
        throw new FormatException("Label priority must be integer");
      }
    }
    Label label = props.getLabel();
    label.setLocation(shape);
    label.setText(text, null);
    labeler.add(label, priority);
  }

  private LayerProperties props;
  private FeatureTable featureTable = null;
  private Area viewPort;
  private Labeler labeler;

  private ShapeFile shapeFile;
  private Query query;
  private Attribute classAttribute = null;
  private Attribute labelAttribute = null;
  private Attribute labelPriorityAttribute = null;

  public class FeatureTable {

    public FeatureTable() throws IOException {

      ArrayList attributeFields = props.getAttributeFields();
      if (attributeFields.isEmpty()) return;

      // System.err.println("Recording features for " + props.getPath());
      featureSource = getAttributeSource();
      featureSource.setColumns(attributeFields);

      header = "";
      for (int i = 0; i < attributeFields.size(); i++) {
         if (i > 0) header += "|";
         header += attributeFields.get(i);
      }
      // System.err.println(header);
    }

    public void add(Object shape, int r) throws IOException {

      String[] record = featureSource.getRecord(r);
      if (record != null) {

        attribs.add(record);
        coords.add(shape);
      }
    }

    public Object getCoords(int r) {

      return coords.get(r);
    }

    public String[] getAttributes(int r) {

      return (String[]) attribs.get(r);
    }

    public String getHeader() {

      return header;
    }

    public int count() {

      return coords.size();
    }

    private String header;
    private ArrayList coords = new ArrayList();
    private ArrayList attribs = new ArrayList();
    private AttributeSource featureSource;
  }
}



