/** 

Copyright (C) 1999-2001 Karl Goldstein

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

**/

package pms;

import java.awt.*;
import java.awt.Toolkit;
import java.awt.geom.*;
import java.util.*;

import pms.tools.*;

public class Extent extends Rectangle2D.Double {

  public int getDisplayWidth() throws FormatException {
    if (displayWidth == 0)
      throw new FormatException(
        "Attempt to get extent display width before it is set");

    return displayWidth;
  }

  public int getDisplayHeight() throws FormatException {
    if (displayHeight == 0)
      throw new FormatException(
        "Attempt to get extent display height before it is set");

    return displayHeight;
  }

  public Extent(Rectangle2D.Double extent, String units) 
    throws FormatException {
    
    super(extent.x, extent.y, extent.width, extent.height);
    parseUnits(units);
  }

  public Extent(String param, String units) throws FormatException {
    
    try {

      StringTokenizer s = new StringTokenizer(param, ",");
      x = java.lang.Double.parseDouble(s.nextToken());
      y = java.lang.Double.parseDouble(s.nextToken());
      double right = java.lang.Double.parseDouble(s.nextToken());
      double bottom = java.lang.Double.parseDouble(s.nextToken());
      width = right - x;
      height = y - bottom;

    } catch (Exception e) {
      throw new FormatException("Failed to parse extent: " + param + 
        " (" + e.getMessage() + ")");
    }

    parseUnits(units);
  }

  private void parseUnits(String units) throws FormatException {

    this.units = units;
    double pixelsPerInch =
      (double) Toolkit.getDefaultToolkit().getScreenResolution();
    unitsPerPixel =
      UnitConverter.measure( 1 / pixelsPerInch, "INCHES", units);
  }

  public void union(Rectangle2D.Double r) {

    double left = Math.min(r.x, x);
    double top =  Math.max(r.y, y);
    double right = Math.max(r.x + r.width, x + width);
    double bottom =  Math.min(r.y - r.height, y - height);

    x = left;
    y = top;
    width = right - left;
    height = top - bottom;

    calcAspect();
    calcTransform();
  }

  public AffineTransform getTransform() throws FormatException {

    if (transform == null)
      throw new FormatException(
        "Attempt to access extent transform before setting display size"); 

    return transform;
  }

  private void calcTransform() {

    if ((displayWidth == 0) || (displayHeight == 0)) return;

    double sx = displayWidth / width;
    double sy = displayHeight / height;
    double scaleFactor = (sx < sy) ? sx : sy;

    transform = new AffineTransform();
    transform.scale(scaleFactor, scaleFactor * -1);
    transform.translate(x * -1, y * -1);
  }
 
  private void calcAspect() {

    if ((displayWidth == 0) || (displayHeight == 0)) return;

    double displayAspect = (double) displayWidth / (double) displayHeight;
    double extentAspect = width / height;
    if (displayAspect < extentAspect) {

      double addY = (width / displayAspect) - height;
      y += addY / 2;
      height += addY;

    } else {

      double addX = (height * displayAspect) - width;
      x -= addX / 2;
      width += addX;
    }
  }

  public void setDisplaySize(int pixelWidth, int pixelHeight) {

    displayWidth = pixelWidth;
    displayHeight = pixelHeight;

    viewRect  = new Rectangle(0, 0, displayWidth, displayHeight);   
    viewPort = new Area(viewRect);
   
    calcAspect();

    pixelSize = (float) width / pixelWidth;

    calcTransform();

    // System.err.println("after: " + this);
  }

  public void setDisplayCenter(int cx, int cy) throws FormatException {

    if ((displayWidth == 0) || (displayHeight == 0)) {
      throw new FormatException(
        "Attempt to recenter display before setting dimensions");
    }

    Point2D.Double cp = new Point2D.Double((double) cx, (double) cy);
    try {
      transform.inverseTransform(cp, cp);
    } catch (NoninvertibleTransformException e) {
      throw new FormatException(e.getMessage());
    }

    x = cp.x - width / 2;
    y = cp.y + height / 2;

    calcTransform();
  }

  public int matchScaleLevel(int[] scales) {

    Debug.assertNotNull("Extent.matchScaleLevel", "scales", scales);

    int currentScale = getScale();
    int delta = Integer.MAX_VALUE;
    int matchLevel = 0;

    for (int i = 1; i < scales.length; i++) {
      int d = Math.abs(scales[i] - currentScale);
      if (d < delta) {
        matchLevel = i;
        delta = d;
      }
    }

    setScale(scales[matchLevel]);
    return matchLevel;
  }

  public int action(String param, int[] scales) throws FormatException {

    int level = 0;

    try {

      StringTokenizer s = new StringTokenizer(param, ",");
      String action = s.nextToken();

      if (action.equals("ZOOM")) {
        double factor = StringTools.toDouble(s.nextToken());
        zoom(factor);
      }

      if (action.equals("PAN")) {
        double fx = StringTools.toDouble(s.nextToken());
        double fy = StringTools.toDouble(s.nextToken());
        pan(fx, fy);
      }

      if (action.equals("SCALE")) {
        int scale = StringTools.toInt(s.nextToken());
        setScale(scale);
      }

      if (action.equals("LEVEL")) {
        level = Integer.parseInt(s.nextToken());
        setScale(scales[level - 1]);
      }
    } catch (Exception e) {
      throw new FormatException(e.getMessage());
    }

    return level;
  }

  public void center(double cx, double cy) {

    x = cx - width / 2;
    y = cy + height / 2;
  }

  public void zoom(double factor) {

    double cx = x + width / 2;
    double cy = y - height / 2;

    width /= factor;
    height /= factor;

    x = cx - width / 2;
    y = cy + height / 2;

    pixelSize = (float) width / displayWidth;
    calcTransform();
  }

  public void pan(double fx, double fy) {

    x += width * fx;
    y += height * fy;

    calcTransform();
  }

  public void setScale(int scale) {
    
    zoom((double) getScale() / (double) scale);
  }

  public void setRect(double x, double y, double w, double h) {

    super.setRect(x, y, w, h);
    pixelSize = (float) width / displayWidth;

    calcTransform();
  }

  public double getPixelSize() {
    return pixelSize;
  }

  public double getUnitsPerPixel() {
    return unitsPerPixel;
  }

  public int getScale() {

    return (int) (pixelSize / unitsPerPixel);
  }
 
  public Extent copy() throws FormatException {

    Extent copy = new Extent(this, units);
    if ((displayWidth != 0) && (displayHeight != 0))
      copy.setDisplaySize(displayWidth, displayHeight);

    return copy;
  }

  public Rectangle getViewRect() {
  
    return viewRect;
  }

  public Area getViewPort() {

    return viewPort;
  }

  public Object clipToView(Object shape) {

    if (shape instanceof Point2D.Float) {

      return shape;

    } else if (shape instanceof GeneralPath) {
      
      Area pathArea = new Area((GeneralPath) shape);
      pathArea.intersect(viewPort);
      return pathArea;
    }

    return shape;
  }

  private int displayWidth;
  private int displayHeight;
  private String units;
  private double pixelSize;
  private double unitsPerPixel; // units per pixel at 1:1 scale
  private AffineTransform transform;

  private Area viewPort;
  private Rectangle viewRect;
}
