/** 

Copyright (C) 1999-2001 Karl Goldstein

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

**/

package pms;

import java.awt.geom.*;
import java.io.*;
import java.util.*;

import pms.tools.*;

public class CartaProperties extends Properties {

  public int getWidth() {
    return width;
  }

  public int getHeight() {
    return height;
  }

  public String getSizeParam() {
    return sizeParam;
  }

  public Extent getExtent() {
    return extent;
  }

  public String getExtentParam() {

    double right = extent.x + extent.width;
    double bottom = extent.y - extent.height;

    return extent.x + "," + extent.y + "," + right + "," + bottom;
  }

  public String getViewPath() {
    return viewPath;
  }

  public String getAbsoluteViewPath() {
    return StringTools.resolvePath(path, viewPath);
  }

  public String getTitle() {
    return viewProps.getTitle();
  }

  public Legend[] getLegends() {
    return viewProps.getLegends();
  }

  public String getViewsParam() {
    return viewProps.getViewsParam();
  }

  public int getScaleLevel() {
    if (scaleLevel == 0)
      scaleLevel = extent.matchScaleLevel(viewProps.getScales());

    return scaleLevel;
  }

  public int getScale() {
    return extent.getScale();
  }

  public int[] getScaleLevels() {
    return viewProps.getScales();
  }

  public int getZoomInScaleLevel() {

    int scaleLevel = getScaleLevel();

    return (scaleLevel < 2) ? 1: scaleLevel - 1;
  }

  public int getZoomOutScaleLevel() {

    int[] scaleLevels = viewProps.getScales();
    int scaleLevel = getScaleLevel();

    return (scaleLevel >= scaleLevels.length) ? scaleLevels.length : 
      scaleLevel + 1;
  }

  public Report[] getReports() {

    return reports;
  }

  public String getAction() {

    return (actionParam == null) ? "" : actionParam;
  }

  public void setPath(String path) {

    this.path = path;
  }

  public void setAction(String actionParam) throws IOException {
    
    if (actionParam == null) { return ; }
    this.actionParam = actionParam;

    if (extent == null) {
      throw new IllegalStateException(
        "Extent must be set before performing an action on the map.");
    }

    if (! actionParam.equals("IDENTIFY")) {
      scaleLevel = extent.action(actionParam, viewProps.getScales());
    } else {
      Carta carta = new Carta(this);
      reports = carta.identify();
    }
  }

  public void setSize(String sizeParam) {

    if (extent == null) {
      throw new IllegalStateException(
      "Extent must be set before setting the size of the map.");
    }

    this.sizeParam = sizeParam;

    try {
      StringTokenizer s = new StringTokenizer(sizeParam, ",");
      width = Integer.parseInt(s.nextToken());
      height = Integer.parseInt(s.nextToken());
    } catch (Exception e) {
      throw new IllegalArgumentException(
        "Failed to parse size parameter: " + sizeParam);
    }

    extent.setDisplaySize(width, height);
  }

  public void setExtent(String extentParam) throws IOException {

    if (viewProps == null) {
      throw new IllegalStateException(
      "View must be set before setting the extent for the map.");
    }

    this.extentParam = extentParam;

    if (extentParam != null) {
      extent = new Extent(extentParam, viewProps.getUnits());
    }
  }

  public void setView(String viewPath) throws IOException {

    if (path == null) {
      throw new IllegalStateException(
      "Path must be set before setting the view for the map.");
    }

    this.viewPath = viewPath;

    String absViewPath = StringTools.resolvePath(path, viewPath);
    viewProps = ViewProperties.retrieve(absViewPath);
    extent = viewProps.getExtent().copy();
  }

  public void setDisplayCenter(String xParam, String yParam) 
    throws IOException {

    if (extent == null) {
      throw new IllegalStateException(
      "Extent must be set before recentering the map.");
    }

    try {
      int mapX = (xParam != null) ? Integer.parseInt(xParam) : width / 2;
      int mapY = (yParam != null) ? Integer.parseInt(yParam) : height / 2;
      extent.setDisplayCenter(mapX, mapY);
    } catch (Exception e) {
      throw new IllegalArgumentException(
        "Failed to parse map parameter(s): " + xParam + "," + yParam);
    }
  }

  public CartaProperties() {}

  public CartaProperties(String path, HashMap mainParams) 
    throws IOException {

    this.path = path;
    parseMain(mainParams);
  }

  public CartaProperties(String path) throws IOException {

    HashMap topMap = parseFile(path);
    parseMain(getOneMap("", topMap, "MAIN", true));
  }

  private void parseMain(HashMap params) throws IOException {
    
    String viewPath = getOneString("MAIN", params, "VIEW", null, true); 
    setView(viewPath);

    String extentParam = getOneString("MAIN", params, "EXTENT", null, false);
    setExtent(extentParam);

    String sizeParam = getOneString("MAIN", params, "SIZE", "360,360", false); 
    setSize(sizeParam);

    String xParam = getOneString("MAIN", params, "MAP.X", null, false);
    String yParam = getOneString("MAIN", params, "MAP.Y", null, false);
    setDisplayCenter(xParam, yParam);

    String actionParam = getOneString("MAIN", params, "ACTION", null, false);
    setAction(actionParam);
  }

  private Report[] reports = null;
  private String actionParam;
  private String viewPath;
  private String sizeParam;
  private String extentParam;
  private int width;
  private int height;
  private Extent extent;
  private int scaleLevel;
  private ViewProperties viewProps;
}
