import java.awt.*;
import java.util.*;

public class Popup extends Rectangle {

  public Popup(Component component, Messenger messenger) {

    this.component = component;
    this.messenger = messenger;
    fm = component.getFontMetrics(font);
    baseline = VERTICAL_PAD + fm.getAscent();
    itemHeight = fm.getAscent() + fm.getDescent() + (VERTICAL_PAD * 2);
  }

  public void setConstraint(Rectangle constraint) {

    this.constraint = constraint;
  }

  public void setParent(Item parentItem) {

    this.parentItem = parentItem;
    this.parent = parentItem.getPopup();
  }

  protected void setItemColors(int index) {}

  public void setItemChild(Popup child, int index) {

    Item item = (Item) items.elementAt(index);
    item.child = child;
    child.setParent(item);
  }

  public Point getAnchor() {

    return anchor;
  }

  public Item getActiveItem() {
 
    return activeItem;
  }

  public void setItemLabel(String label, int index) {

    Item item = (Item) items.elementAt(index);
    item.label = label;
    arrange();
  }

  public void setItemEnabled(boolean enabled, int index) {

    Item item = (Item) items.elementAt(index);
    item.enabled = enabled;
  }

  public void clearItems() {

    clearActiveItem();
    items.removeAllElements();   
    arrange();
  }

  public void clearActiveItem() {

    if (activeItem != null)
      activeItem.unhighlight();
    activeItem = null;
  }

  public boolean dispatch(int x, int y) {

    Item selectItem = getSelectItem();
    if (selectItem == null) {
      if (visible) {
        hide(true);
        return false;
      }
      return show(x, y, true);
    } else {
      selectItem.action();
      return true;
    }
  }

  public Item getSelectItem() {

    if ((! visible) || (activeItem == null))
      return null;

    if (activeItem.child == null)
      return activeItem;

    if (activeItem.child.getActiveItem() == null)
      return activeItem;

    return activeItem.child.getSelectItem();
  }
   
  public boolean show(int x, int y, boolean constrain) {

    if (constrain && (constraint != null) && (! constraint.contains(x, y))) {
      visible = false;
      return false;
    }

    visible = true;
    setAnchor(x, y);

    return true;
  }

  protected void setAnchor(int x, int y) {

    Dimension d = component.size();

    if (parentItem == null) {
      anchor = new Point(x, y);
      this.x = (x + width > d.width) ? (x - width) : x;
      this.y = (y + height > d.height) ? (y - height) : y;
    } else {
      anchor = new Point(parent.getAnchor().x, parent.getAnchor().y);
      this.x = (x + width > d.width) ? (x - width - parent.width + 3) : x - 3;
      this.y = (y + height > d.height) ? (y - height + itemHeight) : y;
    }
  }

  public void hide(boolean ascend) {

    visible = false;
    for (Enumeration e = items.elements(); e.hasMoreElements();) {
      Item item = (Item) e.nextElement();
      item.unhighlight();
    }

    if (ascend && (parent != null))
      parent.hide(true);
  }

  public boolean isVisible() {

    return visible;
  }

  public void highlight(int px, int py) {

    if (! visible) return;
    
    activeItem = null;
    for (Enumeration e = items.elements(); e.hasMoreElements();) {

      Item item = (Item) e.nextElement();
      Popup child = item.child;
      boolean childShowing = (child != null) && child.isVisible();

      if (item.contains(px, py)) {

        item.highlight();
        activeItem = item;
        if (childShowing)
          child.clearActiveItem();

      } else {

        if (childShowing && child.contains(px, py)) {
          item.highlight();
          activeItem = item;
          child.highlight(px, py);
        } else {
          item.unhighlight();
        }
      }
    } 
  }

  public void paint(Graphics g) {

    if (! visible) return;

    drawBezel(this, g);

    for (Enumeration e = items.elements(); e.hasMoreElements();) {
      Item item = (Item) e.nextElement();
      item.paint(g);
    }

    if ((activeItem != null) && 
        (activeItem.child != null) && activeItem.enabled)
      activeItem.child.paint(g);
  }
    
  protected void drawBezel(Rectangle r, Graphics g) {

    g.setColor(Color.black);
    g.drawRect(r.x - 2, r.y - 2, r.width + 4, r.height + 4);

    g.setColor(lightBezelColor);
    g.drawRect(r.x - 2, r.y - 2, r.width + 3, r.height + 3);

    g.setColor(darkBezelColor);
    g.drawRect(r.x - 1, r.y - 1, r.width + 2, r.height + 2);

    g.setColor(Color.white);
    g.drawRect(r.x - 1, r.y - 1, r.width + 1, r.height + 1);

    g.setColor(lightBezelColor);
    g.fillRect(r.x, r.y, r.width + 1, r.height + 1);
  }

  public void addItem(String label, int action, String tag) {
 
    Item item = new Item();
    item.label = label;
    item.action = action;
    item.tag = tag;
    item.bounds = new Rectangle();
    item.visible = true;

    items.addElement(item);
    setItemColors(items.size() - 1);

    arrange();
  }

  protected void arrange() {

    height = 0;
    baseleft = fm.getMaxAdvance();
    width = getMaxWidth() + fm.getMaxAdvance() * 2;

    for (Enumeration e = items.elements(); e.hasMoreElements();) {
      Item item = (Item) e.nextElement();
      item.bounds.setBounds(0, height, width, itemHeight);
      height += itemHeight;
    }
  }

  private int getMaxWidth() {

    int maxWidth = 0;

    for (Enumeration e = items.elements(); e.hasMoreElements();) {
      Item item = (Item) e.nextElement();
      int itemWidth = fm.stringWidth(item.label);
      if (itemWidth > maxWidth) maxWidth = itemWidth;
    }
    return maxWidth;
  }

  protected Component component;
  private Messenger messenger;
  protected Rectangle constraint;
  protected Vector items = new Vector();
  private Font font = new Font("Helvetica", Font.PLAIN, 12);
  protected FontMetrics fm;
  protected int baseline, baseleft, itemHeight;
  protected Point anchor = new Point(0,0);
  protected boolean visible = false;
  private Item activeItem = null;
  private Item parentItem = null;
  private Popup parent = null;

  protected Color lightBezelColor = new Color(204, 204, 204);
  protected Color darkBezelColor = new Color(102, 102, 102);

  private static int VERTICAL_PAD = 2;

  public class Item {

    Popup getPopup() {

      return Popup.this;
    }

    void action() {

      if ((messenger != null) && (child == null) && enabled) {
        hide(true);
        messenger.setMessage(tag, action);
      }
    }

    boolean contains(int px, int py) {

      Rectangle r = new Rectangle(x + bounds.x, y + bounds.y, 
        bounds.width, bounds.height);
      return r.contains(px, py);
    }

    void highlight() {

      highlighted = true;
      if (child != null)
        child.show(x + width, y + bounds.y, false); 
    }

    void unhighlight() {

      highlighted = false;
      if (child != null)
        child.hide(false);
    }

    void paint(Graphics g) {

      int ix = x + bounds.x;
      int iy = y + bounds.y;

      g.setColor(highlighted ? highlightFillColor : unhighlightFillColor);
      g.fillRect(ix, iy, bounds.width, bounds.height);

      if (enabled) {
        g.setColor(highlighted ? highlightTextColor : unhighlightTextColor);
        g.drawString(label, ix + baseleft, iy + baseline);
        if (child != null) drawChildMark(g, ix, iy);
      } else {
        if (! highlighted) {
          g.setColor(Color.white);
          g.drawString(label, ix + baseleft + 1, iy + baseline + 1);
        }
        g.setColor(disabledTextColor);
        g.drawString(label, ix + baseleft, iy + baseline);
      }

    }

    void drawChildMark(Graphics g, int ix, int iy) {

      ix += bounds.width - 6;
      iy += bounds.height / 2;
      int[] xp = { ix, ix - 4, ix - 4 };
      int[] yp = { iy, iy - 4, iy + 4 };
      g.fillPolygon( xp, yp, 3);
    }

    String label = "";
    int action;
    String tag = "";
    Rectangle bounds = new Rectangle();
    boolean visible = true;
    boolean highlighted = false;
    boolean enabled = true;
    Popup child;

    Color highlightTextColor = Color.white;
    Color unhighlightTextColor = Color.black;
    Color disabledTextColor = new Color(153, 153, 153);
    Color highlightFillColor = new Color(51, 51, 153);
    Color unhighlightFillColor = new Color(204, 204, 204);
  }
}
