import java.awt.*;
import java.io.*;
import java.net.*;
import java.util.*;
import java.util.zip.*;

/**
 * Makes requests to the server for new map images and associated
 * information, and parses the response.
 *
 * Requests are time-consuming and are therefore made in a separate
 * thread to ensure that the applet is always responsive to the user.
 **/
public class MapClient extends Thread implements Messenger {

  public MapClient(URL codeBase, Hashtable queryParams, Container container) 
    throws Exception {

    this(codeBase, queryParams, container, "MapApplet");
  }

  public MapClient(URL codeBase, Hashtable queryParams, 
		   Container container, String resource) 
    throws Exception {

    this.container = container;
    this.messenger = (Messenger) container;

    int port = codeBase.getPort();
    if (port == -1) port = 80;

    URL url = new URL(codeBase.getProtocol(), codeBase.getHost(),
       port, codeBase.getFile() + resource);
    System.err.println("Sending request to " + url);
    request = new HttpRequest(messenger, url, queryParams);
  }

  public void run() {

    for (int i = 0; i < 4; i++) {
      try {
        client();
        return;
      } catch (Exception e) {
        e.printStackTrace();
        String message = "Error: " + e.getMessage() + ".  ";
        message += (i < 3) ? "Retrying..." : "Failed.";
        setMessage(message, Messenger.ERROR);
      }
    }
  }

  private void client() throws Exception {

    request.connect();

    loadImage();

    GZIPInputStream zin = new GZIPInputStream(request.getContent());
    DataInputStream in = new DataInputStream(zin);
    mapParams = HttpRequest.parseParamBlock(in);

    String line;
    while ((line = in.readLine()) != null) {
      parseLayer(in);
    }
    setMessage("Retrieving legend symbols...", Messenger.PROGRESS);
    SymbolCache.loadSymbols();
    setMessage("Map request complete.", Messenger.SUCCESS);
  }

  public Layer[] getLayers(int add) {

    Layer[] layerArray = new Layer[layers.size() + add]; 
    layers.copyInto(layerArray);

    return layerArray;
  }

  public Image getImage() {

    return mapImage;
  }

  public String getViews() {

    return (String) mapParams.get("VIEWS");
  }

  public String getTitle() {

    return (String) mapParams.get("TITLE");
  }

  public String getExtent() {

    return (String) mapParams.get("EXTENT");
  }

  private void loadImage() throws Exception {

    setMessage("Creating map image...", Messenger.PROGRESS);

    DataInputStream in = new DataInputStream(request.getContent());
    int imageLength = in.readInt();
    byte[] imageBuffer = HttpRequest.readFully(in, imageLength);
    mapImage = 
      Toolkit.getDefaultToolkit().createImage(imageBuffer);
    loadImageFully(mapImage);
  }

  private void loadImageFully(Image image) {

    MediaTracker tracker = new MediaTracker(container);
    tracker.addImage(image, 0);
    try { 
      tracker.waitForAll();
    } catch (InterruptedException e) {}
  }

  private void parseLayer(DataInputStream in) throws Exception {

    Hashtable layerParams = HttpRequest.parseParamBlock(in);

    Layer layer = Layer.create(layerParams);
    System.err.println(layer.getPath());
 
    parseFeatures(in, layer);

    layers.addElement(layer);
  }

  private void parseFeatures(DataInputStream in, Layer layer) 
    throws Exception {

    int count = in.readInt();
    if (count == 0) return;

    String header = in.readLine();
    System.err.println(header);
    layer.setFields(header);

    for (int r = 0; r < count; r++)
      layer.readFeature(in);
  }

  public synchronized void setMessage(String message, int type) {

    if (messenger != null)
      messenger.setMessage(message, type);
  }

  private Container container = null;
  private Messenger messenger = null;
  private HttpRequest request = null;

  private Image mapImage = null;
  private Hashtable mapParams = null;

  private Vector layers = new Vector();
}
