import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.io.IOException;
import java.net.*;
import java.util.*;

public class MapApplet extends Applet implements Messenger {

  public void init() {

    SymbolCache.setApplet(this);
    Locale.init();

    arrange();
    initControls();
    initParams();
    request();
  }

  public void start() {

    buffer = createImage(size().width, size().height);

  }

  public void stop() {

    mapMouseTracker.stop();
  }

  public void paint(Graphics g) {

    Graphics bg = buffer.getGraphics();
    bg.setColor(Color.white);
    bg.fillRect(0, 0, size().width, size().height);

    titleBar.paint(bg);
    statusBar.paint(bg);
    mapView.paint(bg);
    chartView.paint(bg);
    legendView.paint(bg);

    mapPopup.paint(bg);
    viewMenu.paint(bg);

    if (activeLayer != null) {
      featureLabel.paint(bg);
    }

    bg.dispose();
    bg = null;

    g.drawImage(buffer, 0, 0, this);
    g.dispose();
    g = null;
  }

  public void update(Graphics g) {
    
    paint(g);
  }

  /**
   * Takes the appropriate action for each type of message.
   **/
  public synchronized void setMessage(String message, int type) {

    switch (type) {
    case Messenger.PROGRESS:
      statusBar.setStatus(message);
      break;
    case Messenger.SUCCESS:
      initMap();
      break;
    case Messenger.ERROR:
      statusBar.setStatus(message);
      break;
    case SELECT_VIEW:
      queryParams.put("VIEW", message);
      request();
      break;
    case SHOW_REPORT:
      report(message);
      break;
    case ZOOM:
      zoom(message);
      break;
    case ROBOT:
      request("MapAppletRobot");
      break;
    case PRINT:
      print();
      break;
    }
    repaint();
  }

  private Point translate(Point p, Rectangle r) {

    return new Point(p.x - r.x, p.y - r.y);
  }

  private void request() {
    request("MapApplet");
  }

  private void request(String resource) {

    if ((mapClient != null) && mapClient.isAlive()) {
      setMessage("Request is already pending.  Please wait...", 
        Messenger.PROGRESS);
      return;
    }

    if (mapMouseTracker != null) mapMouseTracker.stop();

    try {
      mapClient = new MapClient(getCodeBase(), queryParams, this, resource);
      mapClient.start();
    } catch (Exception e) {
      e.printStackTrace();
      redirect();
    }
  }

  private void pan() {
    String param = mapView.getPanParam();
    queryParams.put("ACTION", "PAN," + param);
    request();
  }

  private void zoom(String factor) {
    Point p = translate(mapPopup.getAnchor(), mapView);
    queryParams.put("MAP.X", Integer.toString(p.x));
    queryParams.put("MAP.Y", Integer.toString(p.y));
    if (factor.equals("0")) {
      queryParams.remove("EXTENT");
    } else {
      queryParams.put("ACTION", "ZOOM," + factor);
    }
    request();
  }

  private void report(String location) {
    try {
      URL url = new URL(location);
      getAppletContext().showDocument(url, "Report");
    } catch (MalformedURLException e) {
      e.printStackTrace();
    }
  }

  private void print() {
    try {

      int port = getCodeBase().getPort();
      if (port == -1) port = 80;

      String query = "?" + "VIEW=" + queryParams.get("VIEW") + "&" + 
	"EXTENT=" + queryParams.get("EXTENT") + "&" +
	"SIZE=" + "300,300";

      URL url = new URL(getCodeBase().getProtocol(), getCodeBase().getHost(),
			port, getCodeBase().getFile() + 
			"PrintPage.jsp" + query);

      getAppletContext().showDocument(url, "Print");

    } catch (MalformedURLException e) {
      e.printStackTrace();
    }
  }

  private void initMap() {

    updateParams();
    statusBar.setStatus(message);
    titleBar.setTitle(mapClient.getTitle());
    mapView.setImage(mapClient.getImage());
    initViews(mapClient.getViews());

    layers = mapClient.getLayers(1);
    legendView.setLayers(layers);
    try {
      layers[layers.length - 1] = mapView.getPanLayer();
    } catch (IOException e) {
      setMessage(e.getMessage(), Messenger.ERROR);
    }

    mapMouseTracker = new MapMouseTracker();
    mapMouseTracker.start();
  }

  private void initParams() {

    queryParams.put("EXTENT", getParameter("EXTENT"));
    queryParams.put("VIEW", getParameter("VIEW"));
    queryParams.put("SIZE", mapView.width + "," + mapView.height);

    String robotMode = getParameter("MODE");
    System.err.println("Mode: " + robotMode);
    if (robotMode != null && robotMode.equals("TRUE")) {
      startRobot();
    } else {
      System.err.println("Using interactive mode.");
    }
  }

  private void startRobot() {

    MapAppletRobot robot = new MapAppletRobot(this);
    System.err.println("Starting robot");
    robot.start();
  }

  private void updateParams() {

    queryParams.put("EXTENT", mapClient.getExtent());
    queryParams.remove("MAP.X");
    queryParams.remove("MAP.Y");
    queryParams.remove("ACTION");
  }

  private void updateReports() {

    mapPopup.setItemEnabled(false, 1);
    reportPopup.clearItems();
    if (activeLayer == null) return;

    if (activeLayer.setReports(reportPopup))
      mapPopup.setItemEnabled(true, 1);
  }
    
  private void initViews(String param) {

    if (param == null) return;

    viewMenu.clearItems();
    StringTokenizer s = new StringTokenizer(param, "|");
    while (s.hasMoreTokens()) {

      String path = s.nextToken();
      String title = s.nextToken();
      viewMenu.addItem(title, SELECT_VIEW, path);
    }
  }

  private void initControls() {

    mapPopup = new Popup(this, this);
    mapPopup.setConstraint(mapView);
    mapPopup.addItem("Zoom", 0, "");

    zoomPopup = new Popup(this, this);
    zoomPopup.addItem("all the way out", ZOOM, "0");
    zoomPopup.addItem("way out", ZOOM, "0.2");
    zoomPopup.addItem("out", ZOOM, "0.5");
    zoomPopup.addItem("in", ZOOM, "2");
    zoomPopup.addItem("way in", ZOOM, "5");
    mapPopup.setItemChild(zoomPopup, 0);

    mapPopup.addItem("Report", 0, "");
    mapPopup.addItem("Recenter", ZOOM, "1");
    mapPopup.addItem("Print", PRINT, "");

    reportPopup = new Popup(this, this);
    mapPopup.setItemChild(reportPopup, 1);
    mapPopup.setItemEnabled(false, 1);

    featureLabel = new Label(this);
    featureLabel.addItem("", 0, "");
    featureLabel.addItem("Click for popup menu", 0, "");
    featureLabel.setHintItem(1);
  }

  private void redirect() {
  }

  private void arrange() {

    titleBar = new TitleBar(this);
    statusBar = new StatusBar(this);

    int top = titleBar.y + titleBar.height + 10;
    mapView = new MapView(this, top, 0.5, 0.5);
    chartView = new ChartView(this, top, 0.5, 0.5);

    top = mapView.y + mapView.height + 10;
    int bottom = statusBar.y - 10;
    legendView = new LegendView(this, top, bottom, 0.0);

    viewMenu = new DropDown(this, this, titleBar.x + titleBar.width,
      titleBar.y + titleBar.height, "Select view...");
  }

  private String message = "";
  private Hashtable queryParams = new Hashtable();
  private MapClient mapClient = null;
  private Layer[] layers;

  private Image buffer;
  private TitleBar titleBar;
  private MapView mapView;
  private ChartView chartView;
  private LegendView legendView;
  private StatusBar statusBar;
  private DropDown viewMenu;

  private Popup mapPopup, reportPopup, zoomPopup;

  private Point mousePoint;
  private MapMouseTracker mapMouseTracker;
  private Layer activeLayer;
  private String[] activeRecord;
  private Label featureLabel;

  public static final int SELECT_VIEW = 101;
  public static final int ZOOM = 102;
  public static final int SHOW_REPORT = 103;
  public static final int ROBOT = 104;
  public static final int PRINT = 105;

  public boolean mouseDown(Event e, int x, int y) {
  
    if (layers == null) return false;

    if (! mapPopup.isVisible()) {
    
      activeRecord = mapMouseTracker.getActiveRecord();
      featureLabel.hide(false);
      if (activeLayer == layers[layers.length - 1]) {
        pan();
        return false;
      }
      updateReports();
    } 
    mapPopup.dispatch(x, y);
    viewMenu.dispatch(x, y);
    
    repaint();

    return false;
  }

  public boolean mouseMove(Event e, int x, int y) {

    mapPopup.highlight(x, y);
    viewMenu.highlight(x, y);

    mousePoint = new Point(x, y);

    repaint();

    return false;
  }

  class MapMouseTracker extends Thread {

    public void run() {

      while (running) {

        try {
          sleep(100);
          timeElapsed += 150;
          trackMouse();
        } catch (InterruptedException e) {}
      }
    }

    private void trackMouse() {

      if (mapPopup.isVisible() || (mousePoint == null)) return;

      int dx = Math.abs(activePoint.x - mousePoint.x);
      int dy = Math.abs(activePoint.y - mousePoint.y);
      int d = Math.max(dx, dy);

      if (d > 5) {

        activePoint = new Point(mousePoint.x, mousePoint.y);
        timeElapsed = 0;
        clearActive();

      } else if (timeElapsed > 2000) {

        if (d == 0) clearActive();
        
      } else if (timeElapsed > 150) {

         String[] newActiveRecord = getActiveRecord(); 
         if (newActiveRecord != activeRecord) {
           activeRecord = newActiveRecord;
           timeElapsed = 0;
           repaint();
         }
      }  
    }

    private void clearActive() {
      if (activeLayer != null) {
        activeLayer = null;
        activeRecord = null;
        repaint();
      }
    }

    public String[] getActiveRecord() {

      String[] record = null;
      Point searchPoint = translate(mousePoint, mapView);
      for (int i = layers.length - 1; i >= 0; i--) {
        if (layers[i].getFeatureAt(searchPoint.x, searchPoint.y)) {
          activeLayer = layers[i];
          record = activeLayer.getRecord();
          String labelText = activeLayer.getLabel();
          if (labelText != null) {
            featureLabel.setItemLabel(activeLayer.getLabel(), 0);
            featureLabel.setHint(activeLayer.getHint());
            featureLabel.show(mousePoint.x + 10, mousePoint.y + 20, false);
          }
	  activeLayer.setChart(chartView);
          break;
        }
      }
      activePoint = new Point(mousePoint.x, mousePoint.y);
      return record;
    }

    private Point activePoint = new Point(0,0);
    private int timeElapsed = 0;
    private boolean running = true;
  }
}
