import java.io.*;
import java.net.*;
import java.util.*;

/**
 * Constructs simple HTTP GET requests and reads the server response.
 **/
public class HttpRequest implements Messenger {

  public HttpRequest() {}

  public HttpRequest(URL baseURL) {
    this.baseURL = baseURL;
  }

  public HttpRequest(Messenger messenger, URL baseURL) {
    this.messenger = messenger;
    this.baseURL = baseURL;
  }

  public HttpRequest(Messenger messenger, URL baseURL, Hashtable queryParams) {
    this.messenger = messenger;
    this.baseURL = baseURL;
    this.queryParams = queryParams;
  }

  public void connect() throws Exception {

    setMessage("Opening connection...", Messenger.PROGRESS);
    Socket s = new Socket(baseURL.getHost(), baseURL.getPort());
    send(s);
    read(s);
    s.close();

    parseResponse();
    headerParams = parseParamBlock(content);
  }

  private void send(Socket s) throws Exception {

    setMessage("Sending request...", Messenger.PROGRESS);

    PrintStream p = new PrintStream(s.getOutputStream()); 

    p.print("GET " + baseURL.getFile());

    if (! queryParams.isEmpty()) 
      p.print("?" + buildQuery());

    p.print(" HTTP/1.1" + CRLF);
    p.print("Host: map.scorecard.org" + CRLF);
    p.print(CRLF);
  }

  private void read(Socket s) throws Exception {

    setMessage("Waiting for response...", Messenger.PROGRESS);

    ByteArrayOutputStream out = new ByteArrayOutputStream();
    byte[] buf = new byte[16384];
    int len;

    InputStream in = s.getInputStream();
    while ((len = in.read(buf)) >= 0)
      out.write(buf, 0, len);

    ByteArrayInputStream inb = new ByteArrayInputStream(out.toByteArray());
    content = new DataInputStream(inb);

    setMessage("Reading response...", Messenger.PROGRESS);
  }

  private String buildQuery() {

    String query = "";

    for (Enumeration e = queryParams.keys(); e.hasMoreElements();) {
      String param = (String) e.nextElement();
      String value = (String) queryParams.get(param);
      System.err.println(param + "=" + value);
      if (! query.equals("")) query += "&";
      query += URLEncoder.encode(param) + "=" + URLEncoder.encode(value);
    }

    return query;
  }

  private void parseResponse() throws Exception {

    String response = content.readLine();
    StringTokenizer s = new StringTokenizer(response, " ");
    version = s.nextToken();
    code = s.nextToken();

    if (! code.equals("200"))
      throw new IOException("Server returned error");
  }

  public static Hashtable parseParamBlock(DataInputStream in) 
    throws Exception {

    Hashtable block = new Hashtable();

    String line;
    while ((line = in.readLine()) != null) {

      if (line.trim().equals("")) break;
      int colonIndex = line.indexOf(":");
      if (colonIndex == -1)
        System.err.println("Bad content line: \n" + line);
      String param = line.substring(0, colonIndex).trim();
      String value = line.substring(colonIndex + 1).trim();
      block.put(param.toUpperCase(), value);
    }

    return block;
  }

  public static byte[] readFully(InputStream in, int count)
    throws Exception {

    byte[] buffer = new byte[count];
    int pos = 0;
    while (pos < count) {

      int remaining = count - pos;
      int len = (remaining > 1024) ? 1024 : remaining;
      len = in.read(buffer, pos, len);
      if (len < 0)
        throw new IOException("Unexpected EOF in byte stream.");
      pos += len;
    }

    return buffer;
  }

  public void setMessage(String message, int type) {

    if (messenger != null)
      messenger.setMessage(message, type);
  }

  public void setQueryParam(String param, String value) {

    queryParams.put(param, value);
  }

  public void setQueryParams(Hashtable queryParams) {

    this.queryParams = queryParams;
  }

  public String getResponseCode() {

    return code;
  }

  public String getHeaderParam(String param) {

    return (String) headerParams.get(param);
  }

  public DataInputStream getContent() {

    return content;
  }

  private Messenger messenger = null;
  private URL baseURL = null;
  private Hashtable queryParams = new Hashtable();

  private String code;
  private String version;

  private Hashtable headerParams = new Hashtable();
  private DataInputStream content = null;

  private String CRLF = "\r\n";
}
