import java.awt.*;
import java.util.*;

public class ChartView extends Rectangle {

  public ChartView(Container container, 
    int top, double xFactor, double yFactor) {

    Dimension d = container.size();

    x = (int) (d.width * xFactor) + 10;
    width = d.width - x - 10; 

    y = top;
    height = (int) (d.height * yFactor);

    fm = container.getFontMetrics(font);
    titleFM = container.getFontMetrics(titleFont);

    titleTop = y + 10 + titleFM.getAscent();
    subtitleTop = titleTop + titleFM.getDescent() + fm.getHeight();
    chartTop = subtitleTop + 10;
    chartLeft = x + 10;
  }

  private void sizeChart() {

    if (legendType == CHART) {

      chartWidth = width - 20;
      chartHeight = height - chartTop + y - 20;

    } else {

      chartWidth = x + (width * 2 / 3) - chartLeft;
      chartHeight = height - chartTop + y - 20;
      legendLeft = x + chartWidth + 20;
      legendMiddle = y + chartTop + (height - chartTop) / 2;
    }
  }

  public void paint(Graphics bg) {

    bg.setColor(Color.black);
    bg.drawRect(x, y, width - 1, height - 1);

    if (chartType != NONE) {

      int left = x + (width/2) - (titleFM.stringWidth(title)/2); 
      bg.setFont(titleFont);
      bg.drawString(title, left, titleTop);

      left = x + (width / 2) - (fm.stringWidth(subtitle) / 2); 
      bg.setFont(font);
      bg.drawString(subtitle, left, subtitleTop);

      if (legendType == KEY) drawLegend(bg);
      if (chartType == PIE) drawPie(bg); else drawBar(bg);

    } else {

      bg.setFont(font);
      bg.drawString("Move the cursor over the map", x + 30, y + 30);
      bg.drawString("to see charts in this pane", x + 40, y + 50);
      bg.drawString("Click on the map for a popup menu", 
        x + 30, y + 80);
      bg.drawString("with options to view reports", 
        x + 40, y + 100);
      bg.drawString("and zoom in or out", 
        x + 40, y + 120);
    }
  }

  public void setChart(String title, String subtitle, String[] labels, 
    Color[] colors, double[] values, String chartType, 
    String legendType, double maxScaleHint) {

    this.title = title;
    this.subtitle = subtitle;
    this.labels = labels;
    this.colors = colors;
    this.chartType = (chartType.equals("PIE")) ? PIE : BAR;
    this.legendType = (legendType.equals("CHART")) ? CHART : KEY;
    sizeChart();

    hasNonZeroValues = (Tools.max(values) == 0) ? false : true;

    if (chartType.equals("PIE")) {
      this.chartType = PIE;
      setPieValues(values);
    } else if (chartType.equals("BAR")) {
      this.chartType = BAR;
      setBarValues(values, maxScaleHint);
    }
  }

  public void clear() {
  
    chartType = NONE;
  }

  private void drawLegend(Graphics g) {

    int boxSize = fm.getAscent() + fm.getDescent();
    int legendTop = legendMiddle - ((labels.length * (boxSize + 4)) / 2);

    g.setFont(font);
    for (int i = 0; i < labels.length; i++) {
      g.setColor(colors[i]);
      int boxTop = legendTop + i * (boxSize + 4);
      g.fillRect(legendLeft, boxTop, boxSize, boxSize); 
      g.setColor(Color.black);
      g.drawRect(legendLeft, boxTop, boxSize, boxSize);
      g.drawString(labels[i], 
        legendLeft + boxSize + 4, boxTop + fm.getAscent());
    }
  }

  private void setPieValues(double[] values) {

    double total = 0;

    for (int i = 0; i < values.length; i++) {
      total += values[i];
    }

    this.values = new int[values.length];
  
    for (int i = 0; i < values.length; i++) {
      int sweep = (int) (values[i] / total * 360);
      this.values[i] = sweep; 
    }
  }

  private void drawPie(Graphics g) {

    int d = Math.min(chartWidth, chartHeight);
    int x = chartLeft + (chartWidth / 2) - (d / 2);
    int y = chartTop + (chartHeight / 2) - (d / 2);

    if (! hasNonZeroValues) {
      String s = "(no data)";
      Point p = centerString(s, fm, (x + d/2), (y + d/2) , 0, 0);
      g.drawString(s, p.x, p.y);
      return;
    }

    int start = 0; 

    for (int i = 0; i < values.length; i++) {
      g.setColor(colors[i]);
      g.fillArc(x, y, d, d, start, values[i] + 1);
      g.setColor(Color.black);
      g.drawArc(x, y, d, d, start, values[i] + 1);
      start += values[i];
    }
  }

  private void setBarValues(double[] values, double maxScaleHint) {

    this.values = new int[values.length];
    maxValue = Tools.base10RoundedMax(values);

    if (! hasNonZeroValues) {
      chartMax = 0;
      tickCount = 0;
      return;
    }

    double maxAbs = Math.log(maxValue);
    if (maxAbs < 0) maxAbs = -1 / maxAbs;
    double scaleFactor = (maxScaleHint > 0) ? 
      maxAbs / Math.log(maxScaleHint) : 1;

    chartMax = (int) (scaleFactor * chartHeight);
    double tickFactor = Tools.base10(maxValue) / maxValue;
    tickSize = (int) (chartMax * tickFactor);
    tickCount = (tickSize > 10) ? chartMax / tickSize : 1;

    for (int i = 0; i < values.length; i++) {
      double scaledValue  = (values[i] / maxValue) * scaleFactor;
      this.values[i] = (int) (scaledValue * chartHeight);
    }
  }

  private void drawBar(Graphics g) {

    if (values == null) return;
    int barWidth = chartWidth / values.length;

    g.setFont(font);

    g.setColor(Color.gray);
    int markY = chartTop + chartHeight - chartMax;

    String maxString = Tools.toString(maxValue);
    g.drawString(maxString, x + 10, markY - fm.getDescent());

    for (int i = 0; i < tickCount; i++) {
      g.fillRect(x, markY - 2, 8, 4);
      g.drawLine(x, markY, x + chartWidth, markY);
      markY += tickSize;
    }

    for (int i = 0; i < values.length; i++) {

      int barX = chartLeft + barWidth * i;
      int barY = chartTop + chartHeight - values[i];

      g.setColor(colors[i]);
      g.fillRect(barX, barY, barWidth, values[i]);
      g.setColor(Color.black);
      g.drawRect(barX, barY, barWidth, values[i]);

      if (legendType == CHART) {      
        int boxTop = chartTop + chartHeight;
        int boxHeight = y + height - boxTop;
        Point p = centerString(
          labels[i], fm, barX, boxTop, barWidth, boxHeight);
        g.drawString(labels[i], p.x, p.y);
      }
    }
    g.drawLine(chartLeft, chartTop + chartHeight, chartLeft + chartWidth,
      chartTop + chartHeight);
  }

  private Point centerString(String text,
    FontMetrics fm, int x, int y, int width, int height) {

    int sx = x + (width / 2) - (fm.stringWidth(text) / 2);

    // Figure how much extra vertical padding
    int pady = (height - (fm.getAscent() + fm.getDescent())) / 2;

    int sy = y + pady + fm.getAscent();

    return new Point(sx, sy);
  }

  String title;
  String subtitle;

  String[] labels;
  Color[] colors;
  int[] values;

  double maxValue;
  boolean hasNonZeroValues;

  int chartType;
  int legendType = NONE;

  Font titleFont = new Font("SansSerif", Font.BOLD, 12);
  Font font = new Font("SansSerif", Font.PLAIN, 12);
  FontMetrics fm, titleFM;

  int titleTop, subtitleTop, legendLeft, legendMiddle;
  int chartTop, chartLeft, chartWidth, chartHeight, chartMax;
  int tickSize, tickCount;

  public static final int NONE = 0;
  public static final int PIE = 1;
  public static final int BAR = 2;

  public static final int KEY = 0;
  public static final int CHART = 1;
}


