/* LiTE
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Denis Oliver Kropp <dok@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <pthread.h>
#include <directfb.h>

#include "lite_internal.h"

#include "util.h"
#include "font.h"
#include "window.h"


typedef struct {
     LiteFont *title_font;

     struct {
          struct {
               IDirectFBSurface *surface;
               int               width;
               int               height;
          } w;
          struct {
               IDirectFBSurface *surface;
               int               width;
               int               height;
          } o;
          struct {
               IDirectFBSurface *surface;
               int               width;
               int               height;
          } n;
          struct {
               IDirectFBSurface *surface;
               int               width;
               int               height;
          } s;

          struct {
               IDirectFBSurface *surface;
               int               width;
               int               height;
          } nw;
          struct {
               IDirectFBSurface *surface;
               int               width;
               int               height;
          } no;
          struct {
               IDirectFBSurface *surface;
               int               width;
               int               height;
          } sw;
          struct {
               IDirectFBSurface *surface;
               int               width;
               int               height;
          } so;
     } frame;
} WindowTheme;

static WindowTheme *theme = NULL;

static struct {
          IDirectFBSurface *surface;
          int               width;
          int               height;
} cursor = { NULL, 0, 0 };

static LiteWindow **windows   = NULL;
static int          n_windows = 0;

static void add_window (LiteWindow *window);
static void remove_window (LiteWindow *window);

static LiteWindow *find_window_by_id (DFBWindowID id);

static void flush_resize (LiteWindow *window);
static void flush_motion (LiteWindow *window);

static int handle_move (LiteWindow *window, DFBWindowEvent *ev);
static int handle_resize (LiteWindow *window, DFBWindowEvent *ev);
static int handle_close (LiteWindow *window);
static int handle_destroy (LiteWindow *window);
static int handle_enter (LiteWindow *window, DFBWindowEvent *ev);
static int handle_leave (LiteWindow *window, DFBWindowEvent *ev);
static int handle_lost_focus (LiteWindow *window);
static int handle_got_focus (LiteWindow *window);
static int handle_motion (LiteWindow *window, DFBWindowEvent *ev);
static int handle_button (LiteWindow *window, DFBWindowEvent *ev);
static int handle_key_up (LiteWindow *window, DFBWindowEvent *ev);
static int handle_key_down (LiteWindow *window, DFBWindowEvent *ev);
static int focus_traverse (LiteWindow *window);
static LiteBox *find_child (LiteBox *box, int *x, int *y);

static void render_title (LiteWindow *window);
static void render_border (LiteWindow *window);

static void draw_window (LiteBox *box, DFBRegion *region, DFBBoolean clear);


DFBResult lite_load_theme()
{
     DFBResult ret = DFB_FAILURE;

     theme = calloc (1, sizeof (WindowTheme));

     /* Load title font */
     theme->title_font = lite_get_font ("whiterabbit", LITE_FONT_PLAIN, 16);
     if (!theme->title_font) {
          fprintf (stderr, "window_load_theme: Could not load font 'whiterabbit'!\n");
          goto error;
     }

     ret = lite_util_load_image (DATADIR"/links.png",
                                 DSPF_UNKNOWN,
                                 &theme->frame.w.surface,
                                 &theme->frame.w.width,
                                 &theme->frame.w.height, NULL);
     if (ret)
          goto error;

     ret = lite_util_load_image (DATADIR"/rechts.png",
                                 DSPF_UNKNOWN,
                                 &theme->frame.o.surface,
                                 &theme->frame.o.width,
                                 &theme->frame.o.height, NULL);
     if (ret)
          goto error;

     ret = lite_util_load_image (DATADIR"/oben.png",
                                 DSPF_UNKNOWN,
                                 &theme->frame.n.surface,
                                 &theme->frame.n.width,
                                 &theme->frame.n.height, NULL);
     if (ret)
          goto error;

     ret = lite_util_load_image (DATADIR"/unten.png",
                                 DSPF_UNKNOWN,
                                 &theme->frame.s.surface,
                                 &theme->frame.s.width,
                                 &theme->frame.s.height, NULL);
     if (ret)
          goto error;

     ret = lite_util_load_image (DATADIR"/obenlinks.png",
                                 DSPF_UNKNOWN,
                                 &theme->frame.nw.surface,
                                 &theme->frame.nw.width,
                                 &theme->frame.nw.height, NULL);
     if (ret)
          goto error;

     ret = lite_util_load_image (DATADIR"/obenrechts.png",
                                 DSPF_UNKNOWN,
                                 &theme->frame.no.surface,
                                 &theme->frame.no.width,
                                 &theme->frame.no.height, NULL);
     if (ret)
          goto error;

     ret = lite_util_load_image (DATADIR"/untenlinks.png",
                                 DSPF_UNKNOWN,
                                 &theme->frame.sw.surface,
                                 &theme->frame.sw.width,
                                 &theme->frame.sw.height, NULL);
     if (ret)
          goto error;

     ret = lite_util_load_image (DATADIR"/untenrechts.png",
                                 DSPF_UNKNOWN,
                                 &theme->frame.so.surface,
                                 &theme->frame.so.width,
                                 &theme->frame.so.height, NULL);
     if (ret)
          goto error;

     return DFB_OK;

     error:
     lite_free_theme();

     return ret;
}

DFBResult lite_load_cursor()
{
     return lite_util_load_image (DATADIR"/cursor.png",
                                  DSPF_UNKNOWN,
                                  &cursor.surface,
                                  &cursor.width,
                                  &cursor.height, NULL);
}

void lite_free_theme()
{
     if (!theme)
          return;

     if (theme->title_font)
          lite_release_font (theme->title_font);
     if (theme->frame.w.surface)
          theme->frame.w.surface->Release (theme->frame.w.surface);
     if (theme->frame.o.surface)
          theme->frame.o.surface->Release (theme->frame.o.surface);
     if (theme->frame.n.surface)
          theme->frame.n.surface->Release (theme->frame.n.surface);
     if (theme->frame.s.surface)
          theme->frame.s.surface->Release (theme->frame.s.surface);
     if (theme->frame.nw.surface)
          theme->frame.nw.surface->Release (theme->frame.nw.surface);
     if (theme->frame.no.surface)
          theme->frame.no.surface->Release (theme->frame.no.surface);
     if (theme->frame.sw.surface)
          theme->frame.sw.surface->Release (theme->frame.sw.surface);
     if (theme->frame.so.surface)
          theme->frame.so.surface->Release (theme->frame.so.surface);

     free (theme);
     theme = NULL;
}

void lite_free_cursor()
{
     if (cursor.surface)
          cursor.surface->Release (cursor.surface);

     cursor.surface = NULL;
}

LiteWindow *lite_new_window (IDirectFBDisplayLayer *layer,
                             int                    width,
                             int                    height,
                             DFBWindowCapabilities  caps,
                             const char            *title)
{
     DFBResult              ret;
     DFBRectangle           rect;
     DFBWindowDescription   dsc;
     DFBDisplayLayerConfig  dlc;
     IDirectFBWindow       *window;
     IDirectFBSurface      *surface;
     IDirectFBSurface      *subsurface;
     DFBWindowID            id;
     LiteWindow            *lw;

     if (!layer)
          layer = lite_layer;

     layer->GetConfiguration( layer, &dlc );

     /* Create window */
     dsc.flags  = DWDESC_POSX | DWDESC_POSY |
                  DWDESC_WIDTH | DWDESC_HEIGHT | DWDESC_CAPS;
     dsc.width  = width;
     dsc.height = height;
     dsc.caps   = caps;

     if (theme) {
          dsc.width  += theme->frame.w.width  + theme->frame.o.width;
          dsc.height += theme->frame.n.height + theme->frame.s.height;
     }

     dsc.posx = ((int)dlc.width - (int)dsc.width) / 2;
     dsc.posy = ((int)dlc.height - (int)dsc.height) / 2;
     
     ret = layer->CreateWindow (layer, &dsc, &window);
     if (ret) {
          DirectFBError ("window_new: CreateWindow", ret);
          return NULL;
     }

     /* Get ID */
     window->GetID (window, &id);

     /* Set lite's cursor shape. */
     if (cursor.surface)
          window->SetCursorShape (window, cursor.surface, 0, 0);

     /* Get surface */
     ret = window->GetSurface (window, &surface);
     if (ret) {
          DirectFBError ("window_new: Window->GetSurface", ret);
          window->Release (window);
          return NULL;
     }

     /* Clear window surface */
     surface->Clear( surface, 0, 0, 0, 0 );

     /* Get sub surface */
     if (theme) {
          rect.x = theme->frame.w.width;
          rect.y = theme->frame.n.height;
     }
     else {
          rect.x = 0;
          rect.y = 0;
     }
     rect.w = width;
     rect.h = height;

     ret = surface->GetSubSurface (surface, &rect, &subsurface);
     if (ret) {
          DirectFBError ("window_new: Surface->GetSubSurface", ret);
          surface->Release (surface);
          window->Release (window);
          return NULL;
     }

     /* Set opaque content region. */
     window->SetOpaqueRegion (window, rect.x, rect.y,
                              rect.x + rect.w - 1, rect.y + rect.h - 1);
     
     /* allocate window struct */
     lw = calloc (1, sizeof (LiteWindow));

     lw->box.type    = LITE_TYPE_WINDOW;
     lw->box.rect    = rect;
     lw->box.surface = subsurface;

     lw->box.Draw    = draw_window;

     lw->id          = id;
     lw->width       = dsc.width;
     lw->height      = dsc.height;
     lw->window      = window;
     lw->surface     = surface;

     lw->bg.enabled  = DFB_TRUE;
     lw->bg.color.a  = 0xe6;
     lw->bg.color.r  = 0xee;
     lw->bg.color.g  = 0xee;
     lw->bg.color.b  = 0xee;

     /* give the child box focus to ourself */
     lw->focused_box = &lw->box;

     /* set window title */
     if (title)
          lw->title = strdup (title);

     /* Render title bar and borders */
     if (theme) {
          render_title (lw);
          render_border (lw);
     }

     /* Clear window content */
     lite_draw_box_and_children( LITE_BOX(lw), DFB_TRUE );

     /* Flip window surface */
     surface->Flip( surface, NULL, 0 );
     
     add_window (lw);

     return lw;
}

DFBResult
lite_window_event_loop (LiteWindow *window)
{
     DFBResult             ret;
     IDirectFBEventBuffer *buffer;
     IDirectFBWindow      *dfb_window = window->window;

     ret = dfb_window->CreateEventBuffer( dfb_window, &buffer );
     if (ret) {
          DirectFBError( "lite_window_event_loop: "
                         "IDirectFBWindow::CreateEventBuffer() failed", ret );
          return ret;
     }

     while (true) {
          DFBWindowEvent evt;

          buffer->WaitForEvent( buffer );

          while (buffer->GetEvent( buffer, DFB_EVENT(&evt) ) == DFB_OK) {
               if (lite_handle_window_event( window, &evt ))
                    continue;

               switch (evt.type) {
                    case DWET_CLOSE:
                         dfb_window->Destroy( dfb_window );
                         break;

                    case DWET_DESTROYED:
                         return DFB_DESTROYED;

                    default:
                         break;
               }
          }

          lite_flush_window_events( window );
     }

     /* shouldn't reach this */
     return DFB_BUG;
}

void lite_set_window_title (LiteWindow *window,
                            const char *title)
{
     char *new_title = NULL;

     if (!window->title && !title)
          return;

     /* make a copy of the new title first */
     if (title)
          new_title = strdup (title);

     /* free old title */
     if (window->title)
          free (window->title);

     /* set new title */
     window->title = new_title;

     /* render/flip title bar */
     if (theme) {
          DFBRegion         region;
          IDirectFBSurface *surface = window->surface;
          
          render_title (window);
          
          /* The title bar region */
          region.x1 = theme->frame.nw.width;
          region.y1 = 0;
          region.x2 = window->width - theme->frame.no.width - 1;
          region.y2 = theme->frame.n.height - 1;

          surface->Flip( surface, &region, 0 );
     }
}

void lite_set_window_opacity (LiteWindow *window,
                              __u8        opacity)
{
     IDirectFBWindow *dfb_window = window->window;

     if (window->opacity_mode == LITE_BLEND_NEVER && opacity && opacity < 0xff)
          opacity = 0xff;
     
     dfb_window->SetOpacity( dfb_window, opacity );
}

void lite_set_window_background (LiteWindow *window,
                                 DFBColor   *color)
{
     if (color) {
          if (!window->bg.enabled ||
              !DFB_COLOR_EQUAL(window->bg.color, *color))
          {
               window->bg.enabled = DFB_TRUE;
               window->bg.color   = *color;
               
               lite_redraw_box( LITE_BOX(window) );
          }
     }
     else
          window->bg.enabled = DFB_FALSE;
}

void lite_set_window_blend_mode (LiteWindow    *window,
                                 LiteBlendMode  content,
                                 LiteBlendMode  opacity)
{
     DFBCardCapabilities caps;
     DFBWindowOptions    options;

     lite_dfb->GetCardCapabilities (lite_dfb, &caps);

     
     switch (content) {
          case LITE_BLEND_ALWAYS:
          case LITE_BLEND_NEVER:
               window->content_mode = content;
               break;
          default:
               if (caps.blitting_flags & DSBLIT_BLEND_ALPHACHANNEL)
                    window->content_mode = LITE_BLEND_ALWAYS;
               else
                    window->content_mode = LITE_BLEND_NEVER;
               break;
     }
     
     window->window->GetOptions (window->window, &options);

     if (window->content_mode == LITE_BLEND_NEVER)
          options |= DWOP_OPAQUE_REGION;
     else
          options &= ~DWOP_OPAQUE_REGION;
     
     window->window->SetOptions (window->window, options);


     switch (opacity) {
          case LITE_BLEND_ALWAYS:
          case LITE_BLEND_NEVER:
               window->opacity_mode = opacity;
               break;
          default:
               if (caps.blitting_flags & DSBLIT_BLEND_COLORALPHA)
                    window->opacity_mode = LITE_BLEND_ALWAYS;
               else
                    window->opacity_mode = LITE_BLEND_NEVER;
               break;
     }
}

void lite_resize_window (LiteWindow   *window,
                         unsigned int  width,
                         unsigned int  height)
{
     DFBResult         ret;
     unsigned int      nw, nh;
     IDirectFBWindow  *win = window->window;

     /* Calculate new size */
     nw = width;
     nh = height;

     if (theme) {
          nw += theme->frame.w.width  + theme->frame.o.width;
          nh += theme->frame.n.height + theme->frame.s.height;
     }

     /* Set new size */
     ret = win->Resize (win, nw, nh);
     if (ret)
          DirectFBError ("lite_resize_window: Window->Resize", ret);

     /* Remaining stuff happens during DWET_SIZE event in handle_resize */
}

void lite_focus_box (LiteWindow *window, LiteBox *box)
{
     LiteBox *old = window->focused_box;

     if (old == box)
          return;

     if (old) {
          old->is_focused = 0;

          if (old->OnFocusOut)
               old->OnFocusOut (old);
     }

     window->focused_box = box;

     box->is_focused = 1;

     if (box->OnFocusIn)
          box->OnFocusIn (box);
}

int lite_handle_window_event (LiteWindow *window, DFBWindowEvent *event)
{
     if (!window && !(window = find_window_by_id (event->window_id)))
          return 0;

     switch (event->type) {
          case DWET_POSITION:
               flush_motion (window);
               return handle_move (window, event);
          case DWET_SIZE:
               flush_motion (window);
               window->last_resize = *event;
               break;
               //      return handle_resize (window, event);
          case DWET_CLOSE:
               return handle_close (window);
          case DWET_DESTROYED:
               return handle_destroy (window);
          case DWET_ENTER:
               return handle_enter (window, event);
          case DWET_LEAVE:
               return handle_leave (window, event);
          case DWET_LOSTFOCUS:
               return handle_lost_focus (window);
          case DWET_GOTFOCUS:
               return handle_got_focus (window);
          case DWET_MOTION:
               flush_resize (window);
               window->last_motion = *event;
               break;
               //      return handle_motion (window, event);
          case DWET_BUTTONUP:
          case DWET_BUTTONDOWN:
               return handle_button (window, event);
          case DWET_KEYUP:
               return handle_key_up (window, event);
          case DWET_KEYDOWN:
               return handle_key_down (window, event);
          default:
               ;
     }

     return 0;
}

void lite_flush_window_events (LiteWindow *window)
{
     if (window) {
          flush_resize (window);
          flush_motion (window);
     }
     else {
          int n;

          for (n=0; n<n_windows; n++)
               lite_flush_window_events (windows[n]);
     }
}

LiteWindow *lite_find_my_window (LiteBox *box)
{
     while (box->parent)
          box = box->parent;

     if (box->type == LITE_TYPE_WINDOW)
          return LITE_WINDOW (box);

     return NULL;
}

void lite_close_window (LiteWindow *window)
{
     IDirectFBWindow *dfb_window = window->window;

     dfb_window->Close( dfb_window );
}

void lite_destroy_window (LiteWindow *window)
{
     remove_window (window);

     if (window->title)
          free (window->title);

     window->surface->Release (window->surface);
     window->window->Release (window->window);

     lite_destroy_box (LITE_BOX (window));
}

/* file internals */

static void flush_resize (LiteWindow *window)
{
     if (!window->last_resize.type)
          return;

     handle_resize (window, &window->last_resize);

     window->last_resize.type = 0;
}

static void flush_motion (LiteWindow *window)
{
     if (!window->last_motion.type)
          return;

     flush_resize (window);

     handle_motion (window, &window->last_motion);

     window->last_motion.type = 0;
}

static int handle_move (LiteWindow *window, DFBWindowEvent *ev)
{
     if (window->OnMove)
          return window->OnMove (window, ev->x, ev->y);

     return 0;
}

static int handle_resize (LiteWindow *window, DFBWindowEvent *ev)
{
     DFBResult         ret;
     DFBRectangle      rect;
     IDirectFBSurface *subsurface;
     IDirectFBSurface *surface = window->surface;

     /* Get a new sub surface */
     if (theme) {
          rect.x = theme->frame.w.width;
          rect.y = theme->frame.n.height;
          rect.w = ev->w - theme->frame.w.width - theme->frame.o.width;
          rect.h = ev->h - theme->frame.n.height - theme->frame.s.height;
     }
     else {
          rect.x = 0;
          rect.y = 0;
          rect.w = ev->w;
          rect.h = ev->h;
     }

     ret = surface->GetSubSurface (surface, &rect, &subsurface);
     if (ret)
          DirectFBError ("handle_resize: Surface->GetSubSurface", ret);
     else {
          /* Destroy old sub surface */
          window->box.surface->Release (window->box.surface);

          /* Store new sub rect/surface */
          window->box.rect    = rect;
          window->box.surface = subsurface;
     }

     /* Set opaque content region. */
     window->window->SetOpaqueRegion (window->window, rect.x, rect.y,
                                      rect.x + rect.w - 1, rect.y + rect.h - 1);
     
     /* Store new size */
     window->width  = ev->w;
     window->height = ev->h;

     /* Render title bar and borders */
     if (theme) {
          render_title (window);
          render_border (window);
     }

     /* Clear window content */
     lite_draw_box_and_children( LITE_BOX(window), DFB_TRUE );

     /* Flip window surface */
     surface->Flip( surface, NULL, 0 );

     /* Call custom handler */
     if (window->OnResize)
          return window->OnResize (window, rect.w, rect.h);

     return 0;
}

static int handle_close (LiteWindow *window)
{
     if (window->OnClose)
          return window->OnClose (window);

     return 0;
}

static int handle_destroy (LiteWindow *window)
{
     if (window->OnDestroy)
          return window->OnDestroy (window);

     return 0;
}

static int handle_enter (LiteWindow *window, DFBWindowEvent *ev)
{
     if (window->OnEnter)
          window->OnEnter (window, ev->x, ev->y);

     handle_motion (window, ev);

     return 0;
}

static int handle_leave (LiteWindow *window, DFBWindowEvent *ev)
{
     if (window->entered_box) {
          LiteBox *entered = window->entered_box;

          if (entered->OnLeave)
               entered->OnLeave (entered,
                                 ev->x - entered->rect.x,
                                 ev->y - entered->rect.y);

          window->entered_box = NULL;
     }

     if (window->OnLeave)
          return window->OnLeave (window, ev->x, ev->y);

     return 0;
}

static int handle_lost_focus (LiteWindow *window)
{
     window->has_focus = 0;

     if (window->OnFocusOut)
          window->OnFocusOut (window);

     return 0;
}

static int handle_got_focus (LiteWindow *window)
{
     window->has_focus = 1;

     if (window->OnFocusIn)
          window->OnFocusIn (window);

     return 0;
}

static int handle_motion (LiteWindow *window, DFBWindowEvent *ev)
{
     LiteBox *box = LITE_BOX (window);

     if (window->moving) {
          window->window->Move (window->window,
                                ev->cx - window->old_x, ev->cy - window->old_y);

          window->old_x = ev->cx;
          window->old_y = ev->cy;

          return 1;
     }

     if (ev->x >= box->rect.x &&
         ev->y >= box->rect.y &&
         ev->x < (box->rect.x + box->rect.w) &&
         ev->y < (box->rect.y + box->rect.h)) {
          int x = ev->x - box->rect.x;
          int y = ev->y - box->rect.y;

          box = find_child (box, &x, &y);

          if (window->entered_box != box) {
               LiteBox *entered = window->entered_box;

               if (entered && entered->OnLeave)
                    entered->OnLeave (entered, -1, -1); /* FIXME */

               window->entered_box = box;

               if (box->OnEnter)
                    return box->OnEnter (box, x, y);
          }
          else if (box->OnMotion)
               return box->OnMotion (box, x, y);
     }
     else
          handle_leave (window, ev);

     return 0;
}

static int handle_button (LiteWindow *window, DFBWindowEvent *ev)
{
     DFBResult  ret;
     LiteBox   *box = LITE_BOX (window);

     if (window->moving) {
          if (ev->type == DWET_BUTTONUP && ev->button == DIBI_LEFT) {
               window->window->UngrabPointer (window->window);
               window->moving = 0;
          }

          return 1;
     }

     if (ev->x >= box->rect.x &&
         ev->y >= box->rect.y &&
         ev->x < (box->rect.x + box->rect.w) &&
         ev->y < (box->rect.y + box->rect.h)) {
          int x = ev->x - box->rect.x;
          int y = ev->y - box->rect.y;

          box = find_child (box, &x, &y);

          if (ev->type == DWET_BUTTONDOWN && box->OnButtonDown)
               return box->OnButtonDown (box, x, y, ev->button);
          else if (ev->type == DWET_BUTTONUP && box->OnButtonUp)
               return box->OnButtonUp (box, x, y, ev->button);
     }
     else if (ev->type == DWET_BUTTONDOWN) {
          switch (ev->button) {
               case DIBI_LEFT:
                    window->window->RaiseToTop (window->window);
          
                    ret = window->window->GrabPointer (window->window);
                    if (ret) {
                         DirectFBError ("handle_button: window->GrabPointer", ret);
                         return 0;
                    }
          
                    window->moving = 1;
                    window->old_x  = ev->cx;
                    window->old_y  = ev->cy;
          
                    return 1;

               case DIBI_RIGHT:
                    window->window->LowerToBottom (window->window);

                    return 1;

               default:
                    break;
          }
     }

     return 0;
}

static int handle_key_up (LiteWindow *window, DFBWindowEvent *ev)
{
     LiteBox *box = window->focused_box;

     if (ev->key_id == DIKI_TAB && box)
          return 1;

     if (box)
          return box->OnKeyUp ? box->OnKeyUp (box, ev) : 0;

     return 0;
}

static int handle_key_down (LiteWindow *window, DFBWindowEvent *ev)
{
     LiteBox *box = window->focused_box;

     if (ev->key_id == DIKI_TAB)
          return focus_traverse (window);

     if (box)
          return box->OnKeyDown ? box->OnKeyDown (box, ev) : 0;

     return 0;
}

static LiteBox *find_child (LiteBox *box, int *x, int *y)
{
     int i;

     for (i=0; i<box->n_children; i++) {
          LiteBox *child = box->children[i];

          if (*x >= child->rect.x &&
              *y >= child->rect.y &&
              *x < (child->rect.x + child->rect.w) &&
              *y < (child->rect.y + child->rect.h)) {
               *x -= child->rect.x;
               *y -= child->rect.y;

               box = find_child (child, x, y);
               break;
          }
     }

     return box;
}

static LiteWindow *find_window_by_id (DFBWindowID id)
{
     int n;

     for (n=0; n<n_windows; n++)
          if (windows[n]->id == id)
               return windows[n];

     return NULL;
}

static int focus_traverse (LiteWindow *window)
{
     /* FIXME */
     return 0;
}

static void add_window (LiteWindow *window)
{
     n_windows++;
     windows = realloc (windows, sizeof (LiteWindow*) * n_windows);
     windows[n_windows - 1] = window;
}

static void remove_window (LiteWindow *window)
{
     int n;

     for (n=0; n<n_windows; n++)
          if (windows[n] == window)
               break;

     if (n == n_windows) {
          printf ("Window not found!\n");
          return;
     }

     n_windows--;

     for (; n < n_windows; n++)
          windows[n] = windows[n+1];

     windows = realloc (windows, sizeof (LiteWindow*) * n_windows);
}

static void render_title (LiteWindow *window)
{
     int               string_width;
     DFBRegion         region;
     IDirectFBSurface *surface = window->surface;
     IDirectFBFont    *font    = lite_font (theme->title_font);

     if (!window->title)
          return;

     /* The title bar region */
     region.x1 = theme->frame.nw.width;
     region.y1 = 0;
     region.x2 = window->width - theme->frame.no.width - 1;
     region.y2 = theme->frame.n.height - 1;

     surface->SetClip (surface, &region);

     /* Fill title bar background */
     surface->TileBlit (surface,
                        theme->frame.n.surface, NULL, theme->frame.nw.width, 0);

     /* Draw title */
     font->GetStringWidth (font, window->title, -1, &string_width);

     surface->SetColor (surface, 0, 0, 0, 0xff);
     surface->SetFont (surface, font);
     surface->DrawString (surface, window->title, -1,
                          theme->frame.w.width + (window->box.rect.w - string_width) / 2,
                          6, DSTF_TOPLEFT);
}

static void render_border (LiteWindow *window)
{
     DFBRegion         region;
     IDirectFBSurface *surface = window->surface;

     surface->SetClip (surface, NULL);

     /* north west corner */
     surface->Blit (surface, theme->frame.nw.surface, NULL, 0, 0);

     /* north east corner */
     surface->Blit (surface, theme->frame.no.surface, NULL,
                    window->width - theme->frame.no.width, 0);

     /* south west corner */
     surface->Blit (surface, theme->frame.sw.surface, NULL,
                    0, window->height - theme->frame.sw.height);

     /* south east corner */
     surface->Blit (surface, theme->frame.so.surface, NULL,
                    window->width - theme->frame.so.width,
                    window->height - theme->frame.so.height);

     /* south edge */
     region.x1 = theme->frame.sw.width;
     region.y1 = window->height - theme->frame.s.height;
     region.x2 = window->width - theme->frame.so.width - 1;
     region.y2 = window->height - 1;

     surface->SetClip (surface, &region);

     surface->TileBlit (surface,
                        theme->frame.s.surface, NULL, region.x1, region.y1);

     /* west edge */
     region.x1 = 0;
     region.y1 = theme->frame.nw.height;
     region.x2 = theme->frame.w.width - 1;
     region.y2 = window->height - theme->frame.sw.height - 1;

     surface->SetClip (surface, &region);

     surface->TileBlit (surface,
                        theme->frame.w.surface, NULL, region.x1, region.y1);

     /* east edge */
     region.x1 = window->width - theme->frame.o.width;
     region.y1 = theme->frame.no.height;
     region.x2 = window->width - 1;
     region.y2 = window->height - theme->frame.so.height - 1;

     surface->SetClip (surface, &region);

     surface->TileBlit (surface,
                        theme->frame.o.surface, NULL, region.x1, region.y1);
}

static void
draw_window (LiteBox *box, DFBRegion *region, DFBBoolean clear)
{
     LiteWindow *window = LITE_WINDOW(box);

     if (window->bg.enabled) {
          IDirectFBSurface *surface = box->surface;

          surface->SetClip (surface, region);

          surface->Clear (surface,
                          window->bg.color.r,
                          window->bg.color.g,
                          window->bg.color.b,
                          window->bg.color.a);
     }
}

