/* LiTE
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Denis Oliver Kropp <dok@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h> /* free */

#include <pthread.h>
#include <directfb.h>

#include "lite_internal.h"

#include "util.h"

DFBResult lite_util_load_image (const char             *filename,
                                DFBSurfacePixelFormat   pixelformat,
                                IDirectFBSurface      **surface,
                                unsigned int           *width,
                                unsigned int           *height,
                                DFBImageDescription    *desc)
{
     DFBResult               ret;
     DFBSurfaceDescription   dsc;
     IDirectFBSurface       *image;
     IDirectFBImageProvider *provider;

     if (!surface)
          return DFB_INVARG;

     /* Create an image provider for loading the file */
     ret = lite_dfb->CreateImageProvider (lite_dfb, filename, &provider);
     if (ret) {
          fprintf (stderr,
                   "load_image: CreateImageProvider for '%s': %s\n",
                   filename, DirectFBErrorString (ret));
          return ret;
     }

     /* Retrieve a surface description for the image */
     ret = provider->GetSurfaceDescription (provider, &dsc);
     if (ret) {
          fprintf (stderr,
                   "load_image: GetSurfaceDescription for '%s': %s\n",
                   filename, DirectFBErrorString (ret));
          provider->Release (provider);
          return ret;
     }

     /* Use the specified pixelformat if the image's pixelformat is not ARGB */
     if (pixelformat != DSPF_UNKNOWN && dsc.pixelformat != DSPF_ARGB)
          dsc.pixelformat = pixelformat;

     /* Create a surface using the description */
     ret = lite_dfb->CreateSurface (lite_dfb, &dsc, &image);
     if (ret) {
          fprintf (stderr,
                   "load_image: CreateSurface %dx%d: %s\n",
                   dsc.width, dsc.height, DirectFBErrorString (ret));
          provider->Release (provider);
          return ret;
     }

     /* Render the image to the created surface */
     ret = provider->RenderTo (provider, image, NULL);
     if (ret) {
          fprintf (stderr,
                   "load_image: RenderTo for '%s': %s\n",
                   filename, DirectFBErrorString (ret));
          image->Release (image);
          provider->Release (provider);
          return ret;
     }

     /* Return surface */
     *surface = image;

     /* Return width? */
     if (width)
          *width = dsc.width;

     /* Return height? */
     if (height)
          *height  = dsc.height;

     /* Retrieve the image description? */
     if (desc)
          provider->GetImageDescription (provider, desc);

     /* Release the provider */
     provider->Release (provider);

     return DFB_OK;
}

DFBResult lite_util_tile_image (IDirectFBSurface *surface,
                                const char       *filename,
                                int               x,
                                int               y,
                                int               nx,
                                int               ny)
{
     DFBResult         ret;
     unsigned int      width, height;
     IDirectFBSurface *image;

     ret = lite_util_load_image (filename, DSPF_UNKNOWN,
                                 &image, &width, &height, NULL);
     if (ret)
          return ret;

     surface->SetBlittingFlags (surface, DSBLIT_NOFX);

     while (ny--) {
          int i;

          for (i=0; i<nx; i++) {
               surface->Blit (surface, image, NULL, x + i*width, y);
          }

          y += height;
     }

     image->Release (image);

     return DFB_OK;
}

IDirectFBSurface *lite_util_sub_surface (IDirectFBSurface *surface,
                                         int               x,
                                         int               y,
                                         int               width,
                                         int               height)
{
     DFBResult         ret;
     IDirectFBSurface *sub;
     DFBRectangle      rect;

     rect.x = x;
     rect.y = y;
     rect.w = width;
     rect.h = height;

     ret = surface->GetSubSurface (surface, &rect, &sub);
     if (ret) {
          DirectFBError ("lite_util_sub_surface: surface->GetSubSurface", ret);
          return NULL;
     }

     return sub;
}
