/* LiTE
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Denis Oliver Kropp <dok@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "lite_internal.h"

#include "util.h"
#include "font.h"
#include "window.h"
#include "textline.h"


struct _LiteTextLine {
     LiteBox            box;

     LiteFont          *font;
     char              *text;
     unsigned int       cursor_pos;
     int                modified;

     char              *backup;
     
     TextLineEnterFunc  enter;
     void              *enter_data;
     
     TextLineAbortFunc  abort;
     void              *abort_data;
};

static int on_focus_in (LiteBox *box);
static int on_focus_out (LiteBox *box);
static int on_key_down (LiteBox *box, DFBWindowEvent *ev);
static int on_button_down (LiteBox *box, int x, int y,
                           DFBInputDeviceButtonIdentifier button);

static void draw_textline (LiteBox *box, DFBRegion *region, DFBBoolean clear);
static void destroy_textline (LiteBox *box);

LiteTextLine *lite_new_textline (LiteBox *parent, int x, int y, int width, int height)
{
     LiteTextLine *textline;

     textline = calloc (1, sizeof(LiteTextLine));

     textline->box.parent = parent;

     textline->box.rect.x = x;
     textline->box.rect.y = y;
     textline->box.rect.w = width;
     textline->box.rect.h = height;

     if (lite_init_box (LITE_BOX (textline))) {
          free (textline);
          return NULL;
     }

     textline->box.Draw    = draw_textline;
     textline->box.Destroy = destroy_textline;

     textline->box.OnFocusIn    = on_focus_in;
     textline->box.OnFocusOut   = on_focus_out;
     textline->box.OnKeyDown    = on_key_down;
     textline->box.OnButtonDown = on_button_down;

     textline->font = lite_get_font ("default", LITE_FONT_PLAIN,
                                     height*9/10 - 6);
     textline->text = strdup ("");

     textline->box.surface->SetFont (textline->box.surface,
                                     lite_font (textline->font));

     lite_redraw_box (LITE_BOX (textline));

     return textline;
}

void lite_set_textline_text (LiteTextLine      *textline,
                             const char        *text)
{
     if (!strcmp (textline->text, text)) {
          if (!textline->modified)
               return;
     }
     else {
          if (textline->modified)
               free (textline->backup);

          free (textline->text);

          textline->text = strdup (text);
          textline->cursor_pos = strlen (text);
     }

     textline->modified = 0;

     lite_redraw_box (LITE_BOX (textline));
}

void lite_on_textline_enter (LiteTextLine      *textline,
                             TextLineEnterFunc  func,
                             void              *funcdata)
{
     textline->enter      = func;
     textline->enter_data = funcdata;
}

void lite_on_textline_abort (LiteTextLine      *textline,
                             TextLineAbortFunc  func,
                             void              *funcdata)
{
     textline->abort      = func;
     textline->abort_data = funcdata;
}

/* file internals */

static void destroy_textline (LiteBox *box)
{
     LiteTextLine *textline = LITE_TEXTLINE (box);

     if (!textline)
          return;

     if (textline->modified)
          free (textline->backup);
     
     free (textline->text);

     lite_destroy_box (box);
}

static void draw_textline (LiteBox *box, DFBRegion *region, DFBBoolean clear)
{
     IDirectFBSurface *surface  = box->surface;
     LiteTextLine     *textline = LITE_TEXTLINE (box);
     IDirectFBFont    *font     = lite_font (textline->font);
     int               cursor_x = 0;

     surface->SetClip (surface, region);

     font->GetStringWidth (font, textline->text, textline->cursor_pos, &cursor_x);

     /* Draw border */
     surface->SetDrawingFlags (surface, DSDRAW_NOFX);

     if (box->is_focused)
          surface->SetColor (surface, 0xa0, 0xa0, 0xff, 0xff);
     else
          surface->SetColor (surface, 0xe0, 0xe0, 0xe0, 0xff);

     surface->DrawRectangle (surface, 0, 0, box->rect.w, box->rect.h);

     surface->SetColor (surface, 0xc0, 0xc0, 0xc0, 0xff);
     surface->DrawRectangle (surface, 1, 1, box->rect.w - 2, box->rect.h - 2);

     /* Fill the background */
     if (textline->modified)
          surface->SetColor (surface, 0xf0, 0xf0, 0xf0, 0xff);
     else
          surface->SetColor (surface, 0xf0, 0xf0, 0xf0, 0xf0);

     surface->FillRectangle (surface, 2, 2, box->rect.w - 4, box->rect.h - 4);

     /* Draw the text */
     surface->SetColor (surface, 0x30, 0x30, 0x30, 0xff);
     surface->DrawString (surface, textline->text, -1, 5, 2, DSTF_TOPLEFT);

     /* Draw the cursor */
     surface->SetDrawingFlags (surface, DSDRAW_BLEND);

     if (box->is_focused)
          surface->SetColor (surface, 0x40, 0x40, 0x80, 0x80);
     else
          surface->SetColor (surface, 0x80, 0x80, 0x80, 0x80);

     surface->FillRectangle (surface, cursor_x + 5, 4, 1, box->rect.h - 8);
}

static int on_focus_in (LiteBox *box)
{
     lite_redraw_box (box);

     return 0;
}

static int on_focus_out (LiteBox *box)
{
     lite_redraw_box (box);

     return 0;
}

static void set_modified (LiteTextLine *textline)
{
     if (textline->modified)
          return;

     textline->modified = true;

     textline->backup = strdup (textline->text);
}

static void clear_modified (LiteTextLine *textline, bool restore)
{
     if (!textline->modified)
          return;

     textline->modified = false;

     if (restore) {
          free (textline->text);

          textline->text = textline->backup;
     }
     else
          free (textline->backup);

     textline->cursor_pos = 0;
}

static int on_key_down (LiteBox *box, DFBWindowEvent *ev)
{
     LiteTextLine *textline = LITE_TEXTLINE (box);
     int           redraw   = 0;

     switch (ev->key_symbol) {
          case DIKS_ENTER:
               if (textline->modified) {
                    if (textline->enter)
                         textline->enter (textline->text, textline->enter_data);

                    clear_modified (textline, false);
                    
                    redraw = 1;
               }
               break;
          case DIKS_ESCAPE:
               if (textline->abort)
                    textline->abort (textline->abort_data);
               
               if (textline->modified) {
                    clear_modified (textline, true);

                    redraw = 1;
               }
               break;
          case DIKS_CURSOR_LEFT:
               if (textline->cursor_pos > 0) {
                    textline->cursor_pos--;
                    redraw = 1;
               }
               break;
          case DIKS_CURSOR_RIGHT:
               if (textline->cursor_pos < strlen (textline->text)) {
                    textline->cursor_pos++;
                    redraw = 1;
               }
               break;
          case DIKS_HOME:
               if (textline->cursor_pos > 0) {
                    textline->cursor_pos = 0;
                    redraw = 1;
               }
               break;
          case DIKS_END:
               if (textline->cursor_pos < strlen (textline->text)) {
                    textline->cursor_pos = strlen (textline->text);
                    redraw = 1;
               }
               break;
          case DIKS_DELETE:
               if (textline->cursor_pos < strlen (textline->text)) {
                    int len = strlen (textline->text);

                    set_modified (textline);
                    
                    memmove (textline->text + textline->cursor_pos,
                             textline->text + textline->cursor_pos + 1,
                             len - textline->cursor_pos);

                    textline->text = realloc (textline->text, len);

                    redraw = 1;
               }
               break;
          case DIKS_BACKSPACE:
               if (textline->cursor_pos > 0) {
                    int len = strlen (textline->text);

                    set_modified (textline);
                    
                    memmove (textline->text + textline->cursor_pos - 1,
                             textline->text + textline->cursor_pos,
                             len - textline->cursor_pos + 1);

                    textline->text = realloc (textline->text, len);

                    textline->cursor_pos--;

                    redraw = 1;
               }
               break;
          default:
               if (ev->key_symbol >= 32 && ev->key_symbol <= 127) {
                    int len = strlen (textline->text);

                    set_modified (textline);
                    
                    textline->text = realloc (textline->text, len + 2);

                    memmove (textline->text + textline->cursor_pos + 1,
                             textline->text + textline->cursor_pos,
                             len - textline->cursor_pos + 1);

                    textline->text[textline->cursor_pos] = ev->key_symbol;

                    textline->cursor_pos++;

                    redraw = 1;
               }
               else
                    return 0;
     }

     if (redraw)
          lite_redraw_box (box);

     return 1;
}

static int on_button_down (LiteBox *box, int x, int y,
                           DFBInputDeviceButtonIdentifier button)
{
     LiteWindow *window;

     window = lite_find_my_window (box);

     if (window)
          lite_focus_box (window, box);

     return 1;
}
