/* LiTE
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Denis Oliver Kropp <dok@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "lite_internal.h"

#include "font.h"
#include "label.h"


struct _LiteLabel {
     LiteBox            box;

     LiteFont          *font;
     char              *text;
};

static void draw_label (LiteBox *box, DFBRegion *region, DFBBoolean clear);
static void destroy_label (LiteBox *box);

LiteLabel *lite_new_label (LiteBox *parent, int x, int y, int width, int height)
{
     LiteLabel *label;

     label = calloc (1, sizeof(LiteLabel));

     label->box.parent = parent;

     label->box.rect.x = x;
     label->box.rect.y = y;
     label->box.rect.w = width;
     label->box.rect.h = height;

     if (lite_init_box (LITE_BOX (label))) {
          free (label);
          return NULL;
     }

     label->box.Draw    = draw_label;
     label->box.Destroy = destroy_label;

     label->font = lite_get_font ("default", LITE_FONT_PLAIN, height*9/10);
     label->text = strdup ("");

     label->box.surface->SetFont (label->box.surface,
                                  lite_font (label->font));

     return label;
}

void lite_set_label_text (LiteLabel  *label,
                          const char *text)
{
     if (strcmp (label->text, text)) {
          free (label->text);
          label->text = strdup (text);
     }

     lite_redraw_box (LITE_BOX (label));
}

/* file internals */

static void destroy_label (LiteBox *box)
{
     LiteLabel *label = LITE_LABEL (box);

     free (label->text);

     lite_release_font (label->font);

     lite_destroy_box (box);
}

static void draw_label (LiteBox *box, DFBRegion *region, DFBBoolean clear)
{
     IDirectFBSurface *surface = box->surface;
     LiteLabel        *label   = LITE_LABEL (box);

     surface->SetClip (surface, region);

     /* Fill the background */
     if (clear)
          lite_clear_box (box, region);

     /* Draw the text */
     surface->SetColor (surface, 0x30, 0x30, 0x30, 0xff);
     surface->DrawString (surface, label->text, -1,
                          0, 0, DSTF_TOPLEFT);
}

