#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <pthread.h>
#include <directfb.h>

#include "lite_internal.h"

#include "font.h"

#define DEBUG

#ifdef DEBUG
     #define DEBUGMSG(x...) fprintf (stderr, "<<< LiTE Font >>> "x)
#else
     #define DEBUGMSG(x...)
#endif

struct _LiteFont {
     int            refs;
     char          *file;
     int            size;
     IDirectFBFont *font;

     LiteFont      *next;
     LiteFont      *prev;
};

static char *styles[4] = {
     "", "bd", "i", "bi"
};

static LiteFont        *fonts       = NULL;
static pthread_mutex_t  fonts_mutex = PTHREAD_MUTEX_INITIALIZER;

/*
 * Returns existing an font entry from cache after increasing
 * it's reference counter or tries to create a new one otherwise.
 */
static LiteFont *cache_get_entry           (const char *name,
                                            int         size);

static LiteFont *cache_get_entry_from_file (const char *file,
                                            int         size);

/*
 * Decreases the reference counter of a cache entry
 * and removes/destroys it if the counter got zero.
 */
static void cache_release_entry (LiteFont *entry);



LiteFont* lite_get_font (const char *spec, LiteFontStyle style, int size)
{
     int       name_len;
     char     *name, *c;
     LiteFont *font;

     /* translate predefined specs */
     if (!strcasecmp (spec, "default"))
          spec = "arial";
     else if (!strcasecmp (spec, "monospaced"))
          spec = "courier";
     else if (!strcasecmp (spec, "serif"))
          spec = "times";
     else if (!strcasecmp (spec, "sans"))
          spec = "arial";
     else if (!strcasecmp (spec, "sansserif"))
          spec = "arial";

     /* append characters depending on font style */
     name_len = strlen(spec) + 3;
     name = alloca (name_len);
     snprintf (name, name_len, "%s%s", spec, styles[style&3]);

     /* replace spaces by underscores */
     while ((c = strchr (name, ' ')) != NULL)
          *c = '_';

     /* get the font from the cache,
        if it does not exist yet it will be loaded */
     font = cache_get_entry (name, size);

     /* if font loading failed try to get our default font,
        but make sure we aren't already trying that */
     if (!font) {
          if (strcmp (spec, "arial")) {
               char fallback[10];

               snprintf (fallback, 10, "arial%s", styles[style&3]);
               font = cache_get_entry (fallback, size);

               if (!font)
                    fprintf (stderr, "lite_get_font: Could not load fallback "
                             "font '%s' for '"LITEFONTDIR"/%s.ttf'!\n", fallback, name);
          }
          else
               fprintf (stderr, "lite_get_font: "
                        "Could not load '"LITEFONTDIR"/%s.ttf'!\n", name);
     }

     return font;
}

LiteFont* lite_get_font_from_file (const char *file, int size)
{
     LiteFont *font;

     /* get the font from the cache,
        if it does not exist yet it will be loaded */
     font = cache_get_entry_from_file (file, size);

     /* if font loading failed try to get our default font,
        but make sure we aren't already trying that */
     if (!font) {
          font = cache_get_entry ("arial", size);

          if (!font)
               fprintf (stderr, "lite_get_font: Could not load fallback "
                        "font '%s' for '%s'!\n", "arial", file);
     }

     return font;
}

void lite_release_font (LiteFont *font)
{
     cache_release_entry (font);
}

IDirectFBFont *lite_font (LiteFont *font)
{
     return font->font;
}

/*
 * Font cache functions
 */

static LiteFont *cache_get_entry_from_file (const char *file, int size)
{
     LiteFont           *entry;
     DFBResult           ret;
     IDirectFBFont      *font;
     DFBFontDescription  desc;

     /* lock cache */
     pthread_mutex_lock (&fonts_mutex);

     /* lookup existing entry if any */
     for (entry = fonts; entry; entry = entry->next) {
          if (!strcmp (file, entry->file)  &&  size == entry->size) {
               DEBUGMSG( "cache_get_entry: + existing cache entry found "
                         "for '%s' with size %d!\n", file, size );
               entry->refs++;
               pthread_mutex_unlock (&fonts_mutex);
               return entry;
          }
     }

     DEBUGMSG ("cache_get_entry: loading `%s' (%d)...\n", file, size);

     /* try to load the font */
     desc.flags  = DFDESC_HEIGHT;
     desc.height = size;

     ret = lite_dfb->CreateFont (lite_dfb, file, &desc, &font);
     if (ret) {
          DEBUGMSG ("cache_get_entry: font loading failed, using fallback!\n");
          //      DirectFBError ("cache_get_entry: font loading failed", ret);
          pthread_mutex_unlock (&fonts_mutex);
          return NULL;
     }

     /* create a new entry for it */
     entry = calloc (1, sizeof(LiteFont));

     entry->refs = 1;
     entry->file = strdup( file );
     entry->size = size;
     entry->font = font;

     /* insert into cache */
     if (fonts) {
          fonts->prev = entry;
          entry->next = fonts;
     }
     fonts = entry;

     /* unlock cache */
     pthread_mutex_unlock (&fonts_mutex);

     DEBUGMSG ("cache_get_entry: * new cache entry "
               "added for '%s' with size %d\n", file, size);

     return entry;
}

static LiteFont *cache_get_entry (const char *name, int size)
{
     int  len = strlen(LITEFONTDIR) + 1 + strlen(name) + 4 + 1;
     char filename[len];

     /* build the name of the font file to load */
     snprintf (filename, len, LITEFONTDIR"/%s.ttf", name);

     return cache_get_entry_from_file (filename, size);
}

static void cache_release_entry (LiteFont *entry)
{
     /* lock cache */
     pthread_mutex_lock (&fonts_mutex);

     /* decrease reference counter and return if it's not zero */
     if (--entry->refs) {
          pthread_mutex_unlock (&fonts_mutex);
          return;
     }

     DEBUGMSG ("cache_release_entry: - destroying "
               "cache entry '%s' (%d)\n", entry->file, entry->size);

     /* remove entry from cache */
     if (entry->next)
          entry->next->prev = entry->prev;

     if (entry->prev)
          entry->prev->next = entry->next;
     else
          fonts = entry->next;

     /* unlock as early as possible */
     pthread_mutex_unlock (&fonts_mutex);

     /* deinitialize */
     entry->font->Release (entry->font);
     free (entry->file);
     free (entry);
}



void _lite_font_init()
{
     fonts = NULL;
}
