/* LiTE
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Denis Oliver Kropp <dok@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>

#include "lite_internal.h"

#include "util.h"
#include "button.h"


struct _LiteButton {
     LiteBox           box;

     int               enabled;
     ButtonState       state;
     IDirectFBSurface *imgsurface[4];

     ButtonPressFunc   press;
     void             *press_data;
};

static int on_enter       (LiteBox *box, int x, int y);
static int on_leave       (LiteBox *box, int x, int y);
static int on_motion      (LiteBox *box, int x, int y);
static int on_button_down (LiteBox *box, int x, int y,
                           DFBInputDeviceButtonIdentifier button);
static int on_button_up   (LiteBox *box, int x, int y,
                           DFBInputDeviceButtonIdentifier button);

static void draw_button (LiteBox *box, DFBRegion *region, DFBBoolean clear);
static void destroy_button (LiteBox *box);


LiteButton *lite_new_button (LiteBox *parent, int x, int y, int width, int height)
{
     LiteButton *button;

     button = calloc (1, sizeof(LiteButton));

     button->box.parent = parent;

     button->box.rect.x = x;
     button->box.rect.y = y;
     button->box.rect.w = width;
     button->box.rect.h = height;

     if (lite_init_box (LITE_BOX (button))) {
          free (button);
          return NULL;
     }

     button->box.Draw    = draw_button;
     button->box.Destroy = destroy_button;

     button->box.OnEnter      = on_enter;
     button->box.OnLeave      = on_leave;
     button->box.OnMotion     = on_motion;
     button->box.OnButtonDown = on_button_down;
     button->box.OnButtonUp   = on_button_up;

     button->enabled = 1;
     button->state   = BS_NORMAL;

     return button;
}

void lite_enable_button (LiteButton *button, int enabled)
{
     if (!button)
          return;

     if ((button->enabled && !enabled) ||
         (!button->enabled && enabled)) {
          button->enabled = enabled;

          lite_redraw_box (LITE_BOX (button));
     }
}

int lite_set_button_state (LiteButton *button, ButtonState state)
{
     if (!button)
          return -1;

     switch (state) {
          case BS_NORMAL:
          case BS_PRESSED:
          case BS_HILITE:
          case BS_DISABLED:
               break;
          default:
               return -1;
     }

     if (button->state == state)
          return 0;

     button->state = state;

     if (button->enabled) {
          lite_redraw_box (LITE_BOX (button));
     }

     return 0;
}

int lite_set_button_image (LiteButton *button, ButtonState state, char *filename)
{
     if (!filename || !button)
          return -1;

     switch (state) {
          case BS_NORMAL:
          case BS_PRESSED:
          case BS_HILITE:
          case BS_DISABLED:
               break;
          default:
               return -1;
     }

     lite_util_load_image (filename, DSPF_UNKNOWN,
                           &button->imgsurface[state], NULL, NULL, NULL);

     if (state == button->state || (state == BS_DISABLED && !button->enabled)) {
          lite_redraw_box (LITE_BOX (button));
     }

     return 0;
}

int lite_on_button_press (LiteButton *button, ButtonPressFunc func, void *funcdata)
{
     button->press      = func;
     button->press_data = funcdata;

     return 0;
}


/* file internals */

static void destroy_button (LiteBox *box)
{
     LiteButton *button = LITE_BUTTON (box);

     if (!button)
          return;

     if (button->imgsurface[BS_NORMAL])
          button->imgsurface[BS_NORMAL]->Release (button->imgsurface[BS_NORMAL]);

     if (button->imgsurface[BS_PRESSED])
          button->imgsurface[BS_PRESSED]->Release (button->imgsurface[BS_PRESSED]);

     if (button->imgsurface[BS_HILITE])
          button->imgsurface[BS_HILITE]->Release (button->imgsurface[BS_HILITE]);

     if (button->imgsurface[BS_DISABLED])
          button->imgsurface[BS_DISABLED]->Release (button->imgsurface[BS_DISABLED]);

     lite_destroy_box (box);
}

static int on_enter (LiteBox *box, int x, int y)
{
     LiteButton *buttonbox = LITE_BUTTON (box);

     lite_set_button_state (buttonbox, BS_HILITE);

     return 1;
}

static int on_leave (LiteBox *box, int x, int y)
{
     LiteButton *buttonbox = LITE_BUTTON (box);

     lite_set_button_state (buttonbox, BS_NORMAL);

     return 1;
}

static int on_motion (LiteBox *box, int x, int y)
{
     return 1;
}

static int on_button_down (LiteBox *box, int x, int y,
                           DFBInputDeviceButtonIdentifier button)
{
     LiteButton *buttonbox = LITE_BUTTON (box);

     lite_set_button_state (buttonbox, BS_PRESSED);

     return 1;
}

static int on_button_up   (LiteBox *box, int x, int y,
                           DFBInputDeviceButtonIdentifier button)
{
     LiteButton *buttonbox = LITE_BUTTON (box);

     lite_set_button_state (buttonbox, BS_HILITE);

     if (buttonbox->enabled && buttonbox->press)
          buttonbox->press (buttonbox, buttonbox->press_data);

     return 1;
}

static void draw_button (LiteBox *box, DFBRegion *region, DFBBoolean clear)
{
     ButtonState state;
     LiteButton *button = LITE_BUTTON (box);

     if (clear)
          lite_clear_box (box, region);

     box->surface->SetClip (box->surface, NULL);
     box->surface->SetBlittingFlags (box->surface, DSBLIT_BLEND_ALPHACHANNEL);

     if (button->enabled)
          state = button->state;
     else
          state = BS_DISABLED;

     if (button->imgsurface[state])
          box->surface->Blit (box->surface, button->imgsurface[state], NULL, 0, 0);
}
