/* LiTE
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Denis Oliver Kropp <dok@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <pthread.h>
#include <directfb.h>

#include "util.h"

#include "box.h"

static void append_child (LiteBox *box, LiteBox *child);

LiteBox *lite_new_box (LiteBox *parent, int x, int y, int width, int height)
{
     LiteBox          *box;

     if (!parent)
          return NULL;

     box = calloc (1, sizeof (LiteBox));

     box->parent = parent;

     box->rect.x = x;
     box->rect.y = y;
     box->rect.w = width;
     box->rect.h = height;

     if (lite_init_box (box)) {
          free (box);
          return NULL;
     }

     return box;
}

void lite_draw_box_and_children (LiteBox *box, DFBBoolean clear)
{
     int i;

     /* draw box */
     if (box->Draw)
          box->Draw( box, NULL, clear );
     else if (clear)
          lite_clear_box( box, NULL );
     
     /* draw children */
     for (i=0; i<box->n_children; i++)
          lite_draw_box_and_children( box->children[i], DFB_FALSE );
}

void lite_redraw_box (LiteBox *box)
{
     lite_draw_box_and_children( box, DFB_TRUE );

     box->surface->Flip( box->surface, NULL, 0 );
}

void lite_destroy_box (LiteBox *box)
{
     int i;

     /* destroy children */
     for (i=0; i<box->n_children; i++)
          box->children[i]->Destroy (box->children[i]);

     box->surface->Release (box->surface);

     free (box->children);
     free (box);
}

/* lite internals */

int lite_init_box (LiteBox *box)
{
     DFBResult ret;

     /* Get sub surface */
     ret = box->parent->surface->GetSubSurface (box->parent->surface,
                                                &box->rect, &box->surface);
     if (ret) {
          DirectFBError ("lite_new_box: Surface->GetSubSurface", ret);
          return -1;
     }

     box->Destroy = lite_destroy_box;

     append_child (box->parent, box);

     return 0;
}

void lite_clear_box (LiteBox *box, DFBRegion *region)
{
     if (box->parent) {
          DFBRegion reg;

          if (region) {
               reg = *region;

               reg.x1 += box->rect.x;
               reg.x2 += box->rect.x;
               reg.y1 += box->rect.y;
               reg.y2 += box->rect.y;
          }
          else {
               reg.x1 = box->rect.x;
               reg.y1 = box->rect.y;
               reg.x2 = box->rect.x + box->rect.w - 1;
               reg.y2 = box->rect.y + box->rect.h - 1;
          }

          if (box->parent->Draw)
               box->parent->Draw (box->parent, &reg, DFB_TRUE);
          else
               lite_clear_box (box->parent, &reg);
     }
     else {
          fprintf (stderr, "lite_clear_box() called with no parent!\n");
     }
}

/* file internals */

static void append_child (LiteBox *box, LiteBox *child)
{
     box->n_children++;

     box->children = realloc (box->children,
                              sizeof (LiteBox*) * box->n_children);

     box->children[box->n_children-1] = child;
}
