/* LiTE
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Denis Oliver Kropp <dok@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>

#include "lite_internal.h"

#include "util.h"

#include "animation.h"

struct _LiteAnimation {
     LiteBox           box;

     int               stretch;

     int               still_frame;
     int               current;
     int               timeout;
     long              last_time;

     IDirectFBSurface *image;
     int               frame_width;
     int               frame_height;
     int               frames;
     int               frames_h;
     int               frames_v;
};

static void draw_animation (LiteBox *box, DFBRegion *region, DFBBoolean clear);
static void destroy_animation (LiteBox *box);

LiteAnimation *lite_new_animation (LiteBox *parent, int x, int y, int width, int height)
{
     LiteAnimation *animation;

     animation = calloc (1, sizeof(LiteAnimation));

     animation->box.parent = parent;

     animation->box.rect.x = x;
     animation->box.rect.y = y;
     animation->box.rect.w = width;
     animation->box.rect.h = height;

     if (lite_init_box (LITE_BOX (animation))) {
          free (animation);
          return NULL;
     }

     animation->box.Draw    = draw_animation;
     animation->box.Destroy = destroy_animation;

     return animation;
}

int lite_load_animation (LiteAnimation *animation,
                         const char    *filename,
                         int            still_frame,
                         int            frame_width,
                         int            frame_height)
{
     int               frames, frames_v, frames_h;
     int               image_width, image_height;
     IDirectFBSurface *image;

     if (!animation || !filename || frame_width < 1 || frame_height < 1)
          return -1;

     if (lite_util_load_image (filename, DSPF_UNKNOWN, &image,
                               &image_width, &image_height, NULL))
          return -1;

     if ((image_width % frame_width) || (image_height % frame_height)) {
          fprintf (stderr, "lite_load_animation: image width/height "
                   "not a multiple of frame width/height!\n");
          image->Release (image);
          return -1;
     }

     frames_h = image_width / frame_width;
     frames_v = image_height / frame_height;
     frames   = frames_h * frames_v;

     if (still_frame >= frames) {
          fprintf (stderr, "lite_load_animation: still frame out of bounds!\n");
          image->Release (image);
          return -1;
     }

     lite_stop_animation (animation);

     if (animation->image)
          animation->image->Release (animation->image);

     animation->image        = image;
     animation->still_frame  = still_frame;
     animation->current      = -1;
     animation->frame_width  = frame_width;
     animation->frame_height = frame_height;
     animation->frames_h     = frames_h;
     animation->frames_v     = frames_v;
     animation->frames       = frames;

     if (frame_width != animation->box.rect.w  ||  frame_height != animation->box.rect.h)
          animation->stretch = 1;

     lite_stop_animation (animation);

     return 0;
}

void lite_start_animation (LiteAnimation *animation, unsigned int ms_timeout)
{
     animation->current = (animation->still_frame < 0) ? 0 : animation->still_frame;
     animation->timeout = ms_timeout ? ms_timeout : 1;

     lite_redraw_box (LITE_BOX (animation));
}

int lite_update_animation (LiteAnimation *animation)
{
     struct timeval tv;
     long           new_time, diff;

     if (!animation->timeout)
          return 0;

     gettimeofday (&tv, NULL);

     new_time = tv.tv_sec * 1000 + tv.tv_usec / 1000;
     diff     = new_time - animation->last_time;

     if (diff >= animation->timeout) {
          int advance = diff / animation->timeout;

          animation->current += advance;
          animation->current %= animation->frames;

          animation->last_time += advance * animation->timeout;

          lite_redraw_box (LITE_BOX (animation));

          return 1;
     }

     return 0;
}

void lite_stop_animation (LiteAnimation *animation)
{
     animation->timeout = 0;

     if (animation->still_frame >= 0  &&
         animation->current != animation->still_frame) {
          animation->current = animation->still_frame;

          lite_redraw_box (LITE_BOX (animation));
     }
}

int lite_animation_running (LiteAnimation *animation)
{
     return animation->timeout > 0;
}

/* internals */

static void destroy_animation (LiteBox *box)
{
     LiteAnimation *animation = LITE_ANIMATION (box);

     if (animation->image)
          animation->image->Release (animation->image);

     lite_destroy_box (LITE_BOX (animation));
}

static void draw_animation (LiteBox *box, DFBRegion *region, DFBBoolean clear)
{
     DFBRectangle      rect;
     IDirectFBSurface *surface = box->surface;
     LiteAnimation    *animation = LITE_ANIMATION (box);

     if (clear)
          lite_clear_box (box, region);

     rect.w = animation->frame_width;
     rect.h = animation->frame_height;
     rect.x = (animation->current % animation->frames_h) * rect.w;
     rect.y = (animation->current / animation->frames_h) * rect.h;

     surface->SetBlittingFlags (surface, DSBLIT_BLEND_ALPHACHANNEL);
     surface->SetClip (surface, region);

     if (animation->stretch)
          surface->StretchBlit (surface, animation->image, &rect, NULL);
     else
          surface->Blit (surface, animation->image, &rect, 0, 0);
}
