#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>

#include <esd.h>

#include "defs.h"
#include "pcm.h"
#include "rc.h"

#define ESD_CLIENT_NAME "gnuboy"

struct pcm pcm;

static char *esd_server;
static int stereo = 1;
static int samplerate = 44100;
static int sound = 1;

static int esd_fd = -1;
static int esd_play_fd = -1;

rcvar_t pcm_exports[] =
{
	RCV_BOOL("sound", &sound),
	RCV_INT("stereo", &stereo),
	RCV_INT("samplerate", &samplerate),
	RCV_STRING("esd_server", &esd_server),
	RCV_END
};


void pcm_init()
{
	esd_format_t esd_fmt;
	int fl;

	if (!sound)
	{
		pcm.hz = 11025;
		pcm.len = 4096;
		pcm.buf = malloc(pcm.len);
		pcm.pos = 0;
		return;
	}

	esd_fd = esd_open_sound(esd_server);
	if (esd_fd < 0) {
		fprintf(stderr, "cannot open esound server, error %s\n", strerror(errno));
		return;
	}

	
    	esd_fmt = ESD_STREAM | ESD_PLAY;
	if(stereo) {
		esd_fmt |= ESD_STEREO;
	} else {
		esd_fmt |= ESD_MONO;
	}
		
	esd_fmt |= ESD_BITS8;

	esd_play_fd = esd_play_stream_fallback(esd_fmt, samplerate,
					   esd_server, ESD_CLIENT_NAME);
	if (esd_play_fd < 0) {
		fprintf(stderr, "cannot open stream to esound server, error %s\n", strerror(errno));
		return;
	}

	/* enable non-blocking i/o on the socket connection to the esd server */
	if ((fl = fcntl(esd_play_fd, F_GETFL)) >= 0)
		fcntl(esd_play_fd, F_SETFL, O_NDELAY|fl);

	pcm.stereo = stereo;
	pcm.hz = samplerate;
	pcm.len = 4096;
	pcm.buf = malloc(pcm.len);
}

void pcm_close()
{
	if (pcm.buf) free(pcm.buf);
	memset(&pcm, 0, sizeof pcm);

	if (esd_play_fd >= 0) {
		esd_close(esd_play_fd);
		esd_play_fd = -1;
	}

	if (esd_fd >= 0) {
		esd_close(esd_fd);
		esd_fd = -1;
	}
}

int pcm_submit()
{
	if (esd_play_fd < 0)
	{
		pcm.pos = 0;
		return 0;
	}
	if (pcm.buf) write(esd_play_fd, pcm.buf, pcm.pos);
	pcm.pos = 0;
	return 1;
}






