/*
   (c) Copyright 2000-2002  convergence integrated media GmbH.
   (c) Copyright 2002       convergence GmbH.
   
   All rights reserved.

   Written by Denis Oliver Kropp <dok@directfb.org>,
              Andreas Hundt <andi@fischlustig.de> and
              Sven Neumann <sven@convergence.de>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <directfb.h>

#include <core/coredefs.h>
#include <core/coretypes.h>

#include <core/gfxcard.h>
#include <core/layers.h>
#include <core/palette.h>
#include <core/windows.h>

#include <misc/util.h>
#include <gfx/util.h>
#include <gfx/convert.h>


static void repaint_stack_for_window( CoreWindowStack     *stack, 
                                      DFBRegion           *region,
                                      DFBSurfaceFlipFlags  flags,
                                      int                  window );

static void repaint_stack( CoreWindowStack     *stack,
                           DFBRegion           *region,
                           DFBSurfaceFlipFlags  flags );

/******************************************************************************/

void
dfb_window_repaint( CoreWindow          *window,
                    DFBRegion           *region,
                    DFBSurfaceFlipFlags  flags,
                    bool                 force_complete,
                    bool                 force_invisible )
{
     DFBRegion        reg;
     CoreWindowStack *stack = window->stack;
     DisplayLayer *layer = dfb_layer_at( stack->layer_id );

     if (!force_invisible && !VISIBLE_WINDOW(window))
          return;

     if (dfb_windowstack_lock( stack ))
          return;

     if (region) {
          reg.x1 = region->x1 + window->x;
          reg.y1 = region->y1 + window->y;
          reg.x2 = region->x2 + window->x;
          reg.y2 = region->y2 + window->y;
     }
     else {
          reg.x1 = window->x;
          reg.y1 = window->y;
          reg.x2 = window->x + window->width  - 1;
          reg.y2 = window->y + window->height - 1;
     }

     if (stack->hw_mode) {
          dfb_layer_update_region( layer, &reg, flags );
          dfb_windowstack_unlock( stack );
          return;
     }

     if (force_complete)
          repaint_stack( stack, &reg, flags );
     else
          repaint_stack_for_window( stack, &reg, flags,
                                    dfb_windowstack_get_window_index(window) );

     dfb_windowstack_unlock( stack );
}

void
dfb_windowstack_repaint_all( CoreWindowStack *stack )
{
     DFBRegion region;
     
     DFB_ASSERT( stack != NULL );
     
     if (stack->hw_mode)
          return;

     if (dfb_windowstack_lock( stack ))
          return;

     region.x1 = 0;
     region.y1 = 0;
     region.x2 = stack->width  - 1;
     region.y2 = stack->height - 1;
     
     repaint_stack( stack, &region, 0 );
          
     dfb_windowstack_unlock( stack );
}

void
dfb_windowstack_sync_buffers( CoreWindowStack *stack )
{
     DisplayLayer *layer;
     CoreSurface  *surface;

     DFB_ASSERT( stack != NULL );

     if (stack->hw_mode)
          return;
     
     dfb_windowstack_lock( stack );

     layer = dfb_layer_at( stack->layer_id );
     surface = dfb_layer_surface( layer );

     if (surface->caps & (DSCAPS_FLIPPING | DSCAPS_TRIPLE))
          dfb_gfx_copy(surface, surface, NULL);

     dfb_windowstack_unlock( stack );
}

/******************************************************************************/

static void
alpharegion_update(CoreWindow *window, CardState *state, int level,
                   int x1, int y1, int x2, int y2, int func)
{
     DFBRegion region = { x1, y1, x2, y2 };
     
     while (level >= 0) {
          DFBRectangle *arect = &window->alphawin[level]->rect;
          if (dfb_region_intersect(&region, arect->x, arect->y,
                                   arect->x + arect->w - 1,
                                   arect->y + arect->h - 1))
               break;
          level--;
     }

     if (level >= 0) {     
          if (region.x1 != x1)
               alpharegion_update( window, state, level-1, x1, region.y1, region.x1-1, region.y2, func );
          if (region.y1 != y1)
               alpharegion_update( window, state, level-1, x1, y1, x2, region.y1-1, func );
          if (region.x2 != x2)
               alpharegion_update( window, state, level-1, region.x2+1, region.y1, x2, region.y2, func );
          if (region.y2 != y2)
               alpharegion_update( window, state, level-1, x1, region.y2+1, x2, y2, func );
     }

     if (func == 2) { // colorkey       
       switch (window->alphawin[level]->ck_mode) {
       case COLOR_KEY_TRANSPARENT_MODE:
         state->color.a = 0x00;
         state->color.r = 0x00;
         state->color.g = 0x00;
         state->color.b = 0x00;
         break;
       case COLOR_KEY_SOLID_COLOR_MODE:
         state->color.a = 0xff;
         if (level >= 0) {
           state->color.r = window->alphawin[level]->r;
           state->color.g = window->alphawin[level]->g;
           state->color.b = window->alphawin[level]->b;         
         }
         else {
           state->color.a = 0x00;
           state->color.r = 0x00;
           state->color.g = 0x00;
           state->color.b = 0x00;
         }
         break;
       case COLOR_KEY_ALPHA_BLEND_MODE:         
         if (level >= 0) {
           // Calculate colors using fixed point arithmetics
           int alpha = state->color.a = window->alphawin[level]->alpha;
           int r = (((window->color_key >> 0x10) & 0xff) * (alpha + 1)) >> 8;
           int g = (((window->color_key >> 0x08) & 0xff) * (alpha + 1)) >> 8;
           int b = (((window->color_key << 0x00) & 0xff) * (alpha + 1)) >> 8;           
           state->color.a = alpha;
           state->color.r = r;
           state->color.g = g;
           state->color.b = b;         
         }
         else {
           state->color.a = 0xff;
           state->color.r = 0x00;
           state->color.g = 0x00;
           state->color.b = 0x00;
         }
         break;
       }

          DFBRectangle rect;
       
          state->modified |= SMF_COLOR;
          rect.x = region.x1;
          rect.y = region.y1;
          rect.w = region.x2 - region.x1 + 1;
          rect.h = region.y2 - region.y1 + 1;
          dfb_gfxcard_fillrectangle( &rect, state );
     }
     else if (func == 1) { // overlay
          if (level == 0 || window->alphawin[level]->alpha == 0x00) {
               DFBRectangle srect = { region.x1 - window->overlayregion.x1,
                                      region.y1 - window->overlayregion.y1,
                                      region.x2 - region.x1 + 1,
                                      region.y2 - region.y1 + 1 };
               dfb_gfxcard_blit( &srect, region.x1, region.y1, state );
          }
     }
     else { // blend dest graphics
          if (level >= 0 && window->alphawin[level]->ck_mode != COLOR_KEY_ALPHA_BLEND_MODE) {
               state->color.a = window->alphawin[level]->alpha;
          }
          else {
               state->color.a = 0xff;
          }

          if (level == 0 && (window->options & DWOP_COLORKEYING)) {
              state->color.a = 0xff;
          }

          DFBRectangle rect;

          state->modified |= SMF_COLOR;
          rect.x = region.x1;
          rect.y = region.y1;
          rect.w = region.x2 - region.x1 + 1;
          rect.h = region.y2 - region.y1 + 1;
          dfb_gfxcard_fillrectangle(&rect, state);
     }
}


static __u32
select_color_key(CoreWindow *window)
{
     int i;
     DFBBoolean found = DFB_TRUE;
     __u32 ck = (window->color_key + 1) & 0x00ffffff;

     DFB_ASSERT(window->num_alphawin);
     
     while (found) {
          found = DFB_FALSE;
          for (i = 0; i < window->num_alphawin; i++) {
               DFBSecretAlphaWindow* win = window->alphawin[i];
               if (win->ck_mode == COLOR_KEY_SOLID_COLOR_MODE && PIXEL_RGB32(win->r, win->g, win->b) == ck) {
                    ck++;
                    ck &= 0x00ffffff;
                    found = DFB_TRUE;
               }
          }
          if (ck == 0) {
              found = DFB_TRUE;
              ck++;
          }
     }
     return ck;
}

static void
draw_window( CoreWindow *window, CardState *state,
             DFBRegion *region, bool alpha_channel )
{
     DFBRectangle            srect;
     DFBSurfaceBlittingFlags flags = DSBLIT_NOFX;

     DFB_ASSERT( window != NULL );
     DFB_ASSERT( state != NULL );
     DFB_ASSERT( region != NULL );

     srect.x = region->x1 - window->x;
     srect.y = region->y1 - window->y;
     srect.w = region->x2 - region->x1 + 1;
     srect.h = region->y2 - region->y1 + 1;

     __u32			    ck = 0;
     DFBRegion               cregion = *region;
     DFBBoolean                useck = DFB_FALSE;

     if (window->caps & DWCAPS_ALPHAWIN) {
          if ((window->options & DWOP_COLORKEYING) && window->num_alphawin) {
              DFBRectangle* vrect = &window->alphawin[0]->rect;
              if (dfb_region_intersect(&cregion, vrect->x, vrect->y,
                                       vrect->x + vrect->w - 1,
                                       vrect->y + vrect->h - 1)) {
                  CoreSurface *olddest = state->destination;
                  DFBRectangle rect = { cregion.x1, cregion.y1,
                                        cregion.x2 - cregion.x1 + 1,
                                        cregion.y2 - cregion.y1 + 1 };
                  
                  useck = DFB_TRUE;
                  ck = select_color_key(window);
             
                  /* Copy background to scratch surface */
             
                  state->source = state->destination;
                  state->destination = window->surface1;
                  state->blittingflags = DSBLIT_NOFX;

                  state->modified |= SMF_SOURCE | SMF_DESTINATION | SMF_BLITTING_FLAGS;

                  /* Need to copy from the back buffer! Is this safe? */
                  SurfaceBuffer *tmp = state->source->front_buffer;
                  state->source->front_buffer = state->source->back_buffer;
         
                  dfb_gfxcard_blit(&rect, cregion.x1, cregion.y1, state);
               
                  olddest->front_buffer = tmp;

                  /* Fill with inverse destination color key */

                  state->color.a = 0xff;
                  state->color.r = (ck >> 16) & 0xff;
                  state->color.g = (ck >> 8) & 0xff;
                  state->color.b = ck & 0xff;
                  state->dst_colorkey = 0xff000000 | window->color_key;
                  state->drawingflags = DSDRAW_DST_COLORKEY;
                  state->modified |= SMF_COLOR | SMF_DST_COLORKEY | SMF_DRAWING_FLAGS;

                  dfb_gfxcard_fillrectangle(&rect, state);

                  /* Restore dest */
         
                  state->destination = olddest;
                  state->modified |= SMF_DESTINATION;
              }
          }

          /* Blend destination graphics */
          state->drawingflags = DSDRAW_BLEND;
          state->src_blend = DSBF_ZERO;
          state->dst_blend = DSBF_SRCALPHA; 
          state->modified |= SMF_SRC_BLEND | SMF_DST_BLEND | SMF_DRAWING_FLAGS;

          alpharegion_update(window, state, window->num_alphawin - 1,
                             region->x1, region->y1, region->x2, region->y2, 0);

          /* Set alpha values for video (hacked to set 0xff000000 write mask) */

          state->src_blend = DSBF_ONE;
          state->dst_blend = DSBF_ZERO;
          state->modified |= SMF_SRC_BLEND | SMF_DST_BLEND;

          alpharegion_update(window, state, window->num_alphawin - 1,
                             region->x1, region->y1, region->x2, region->y2, 0);

          /* Restore blend modes */

          state->src_blend = DSBF_SRCALPHA;
          state->dst_blend = DSBF_INVSRCALPHA;
          state->modified |= SMF_SRC_BLEND | SMF_DST_BLEND;

          if (useck) {
               CoreSurface *olddest = state->destination;
               DFBRectangle rect = { cregion.x1, cregion.y1,
                                     cregion.x2 - cregion.x1 + 1,
                                     cregion.y2 - cregion.y1 + 1 };

               /* Draw alpha and solid color value with destination color key */

               state->destination = window->surface1;
               state->dst_colorkey = window->color_key;
               state->drawingflags = DSDRAW_DST_COLORKEY;
               state->modified |= SMF_DST_COLORKEY | SMF_DESTINATION | SMF_DRAWING_FLAGS;

               alpharegion_update(window, state, window->num_alphawin - 1,
                                  cregion.x1, cregion.y1, cregion.x2, cregion.y2, 2);

               /* Blit scratch surface with inverse source color key. */

               state->source        = window->surface1;
               state->blittingflags = DSBLIT_SRC_COLORKEY;
               state->src_colorkey  = 0xff000000 | ck;
               state->destination   = olddest;
               state->modified     |= SMF_SOURCE | SMF_DESTINATION | SMF_BLITTING_FLAGS | SMF_SRC_COLORKEY;
               
               dfb_gfxcard_blit(&rect, cregion.x1, cregion.y1, state);
          }
          
          /* Clip overlay against video window (alpha win 0) and alpha windows */

          DFBRegion oregion = *region;
          if (window->overlay_active &&
              window->num_alphawin &&
              dfb_region_intersect(&oregion,
                                   window->overlayregion.x1, window->overlayregion.y1,
                                   window->overlayregion.x2, window->overlayregion.y2)) {
               DFBRectangle* vrect = &window->alphawin[0]->rect;
               if (dfb_region_intersect(&oregion, vrect->x, vrect->y,
                                        vrect->x + vrect->w - 1,
                                        vrect->y + vrect->h - 1)) {
                    state->blittingflags = DSBLIT_BLEND_ALPHACHANNEL;
                    state->source = window->surface;
                    state->modified |= SMF_BLITTING_FLAGS | SMF_SOURCE;
                    
                    alpharegion_update(window, state, window->num_alphawin - 1,
                                       oregion.x1, oregion.y1, oregion.x2, oregion.y2, 1);
               }
          }

          /* Clear area for the PIP window */

          if (window->pip_active) {
              DFBRegion* pregion = &window->pip_region;
              DFBRegion iregion = *region;

              if (dfb_region_intersect(&iregion, pregion->x1, pregion->y1,
                                       pregion->x2, pregion->y2)) {
                  DFBRectangle rect = { iregion.x1, iregion.y1,
                                        iregion.x2 - iregion.x1 + 1,
                                        iregion.y2 - iregion.y1 + 1 };

                  state->color.a = 0x00;
                  state->color.r = 0x00;
                  state->color.g = 0x00;
                  state->color.b = 0x00;
                  state->modified |= SMF_COLOR;

                  state->src_blend = DSBF_ZERO;
                  state->dst_blend = DSBF_SRCALPHA;
                  state->modified |= SMF_SRC_BLEND | SMF_DST_BLEND;

                  dfb_gfxcard_fillrectangle(&rect, state);
              }
          }

          state->source = NULL;
          state->modified |= SMF_SOURCE;

          return;
     }

     if (alpha_channel && (window->options & DWOP_ALPHACHANNEL))
          flags |= DSBLIT_BLEND_ALPHACHANNEL;

     if (window->opacity != 0xFF) {
          flags |= DSBLIT_BLEND_COLORALPHA;

          if (state->color.a != window->opacity) {
               state->color.a = window->opacity;
               state->modified |= SMF_COLOR;
          }
     }

     if (window->options & DWOP_COLORKEYING) {
          flags |= DSBLIT_SRC_COLORKEY;

          if (state->src_colorkey != window->color_key) {
               state->src_colorkey = window->color_key;
               state->modified |= SMF_SRC_COLORKEY;
          }
     }

     if (window->surface->caps & DSCAPS_INTERLACED)
          flags |= DSBLIT_DEINTERLACE;

     if (state->blittingflags != flags) {
          state->blittingflags  = flags;
          state->modified      |= SMF_BLITTING_FLAGS;
     }

     state->source    = window->surface;
     state->modified |= SMF_SOURCE;

     dfb_gfxcard_blit( &srect, region->x1, region->y1, state );

     state->source    = NULL;
     state->modified |= SMF_SOURCE;
}

static void
draw_background( CoreWindowStack *stack, CardState *state, DFBRegion *region )
{
     DFB_ASSERT( stack != NULL );
     DFB_ASSERT( state != NULL );
     DFB_ASSERT( region != NULL );

     switch (stack->bg.mode) {
          case DLBM_COLOR: {
                    CoreSurface *dest  = state->destination;
                    DFBColor    *color = &stack->bg.color;
                    DFBRectangle rect  = { region->x1, region->y1,
                         region->x2 - region->x1 + 1,
                         region->y2 - region->y1 + 1};

                    state->color = *color;

                    if (DFB_PIXELFORMAT_IS_INDEXED( dest->format ))
                         state->color_index = dfb_palette_search( dest->palette,
                                                                  color->r,
                                                                  color->g,
                                                                  color->b,
                                                                  color->a );

                    state->modified |= SMF_COLOR;

                    dfb_gfxcard_fillrectangle( &rect, state );
                    break;
               }
          case DLBM_IMAGE: {
                    DFBRectangle rect = { region->x1, region->y1,
                         region->x2 - region->x1 + 1,
                         region->y2 - region->y1 + 1};

                    DFB_ASSERT( stack->bg.image != NULL );

                    if (state->blittingflags != DSBLIT_NOFX) {
                         state->blittingflags  = DSBLIT_NOFX;
                         state->modified      |= SMF_BLITTING_FLAGS;
                    }

                    state->source    = stack->bg.image;
                    state->modified |= SMF_SOURCE;

                    dfb_gfxcard_blit( &rect, region->x1, region->y1, state );

                    state->source    = NULL;
                    state->modified |= SMF_SOURCE;
                    break;
               }
          case DLBM_TILE: {
                    DFBRectangle rect = { 0, 0,
                         stack->bg.image->width,
                         stack->bg.image->height};

                    DFBRegion orig_clip = state->clip;

                    DFB_ASSERT( stack->bg.image != NULL );

                    if (state->blittingflags != DSBLIT_NOFX) {
                         state->blittingflags  = DSBLIT_NOFX;
                         state->modified      |= SMF_BLITTING_FLAGS;
                    }

                    state->source    = stack->bg.image;
                    state->clip.x1   = region->x1;
                    state->clip.y1   = region->y1;
                    state->clip.x2   = region->x2;
                    state->clip.y2   = region->y2;
                    state->modified |= SMF_SOURCE | SMF_CLIP;

                    dfb_gfxcard_tileblit( &rect,
                                          (region->x1 / rect.w) * rect.w,
                                          (region->y1 / rect.h) * rect.h,
                                          (region->x2 / rect.w + 1) * rect.w,
                                          (region->y2 / rect.h + 1) * rect.h,
                                          state );

                    state->source    = NULL;
                    state->clip      = orig_clip;
                    state->modified |= SMF_SOURCE | SMF_CLIP;
                    break;
               }
          case DLBM_DONTCARE:
               break;
          default:
               BUG( "unknown background mode" );
               break;
     }
}

static void
update_region( CoreWindowStack *stack,
               CardState       *state,
               int              start,
               int              x1,
               int              y1,
               int              x2,
               int              y2 )
{
     int       i      = start;
     DFBRegion region = { x1, y1, x2, y2};

     /* check for empty region */
     DFB_ASSERT (x1 <= x2  &&  y1 <= y2);

     while (i >= 0) {
          if (VISIBLE_WINDOW(stack->windows[i])) {
               int       wx2    = stack->windows[i]->x +
                                  stack->windows[i]->width - 1;
               int       wy2    = stack->windows[i]->y +
                                  stack->windows[i]->height - 1;

               if (dfb_region_intersect( &region, stack->windows[i]->x,
                                         stack->windows[i]->y, wx2, wy2 ))
                    break;
          }

          i--;
     }

     if (i >= 0) {
          CoreWindow *window = stack->windows[i];

          if ((window->options & DWOP_ALPHACHANNEL) &&
              (window->options & DWOP_OPAQUE_REGION)) {
               DFBRegion opaque = region;

               if (!dfb_region_intersect( &opaque,
                                          window->x + window->opaque.x1,
                                          window->y + window->opaque.y1,
                                          window->x + window->opaque.x2,
                                          window->y + window->opaque.y2 )) {
                    update_region( stack, state, i-1, x1, y1, x2, y2 );

                    draw_window( window, state, &region, true );
               }
               else {
                    if ((window->opacity < 0xff) ||
                        (window->options & DWOP_COLORKEYING)) {
                         /* draw everything below */
                         update_region( stack, state, i-1, x1, y1, x2, y2 );
                    }
                    else {
                         /* left */
                         if (opaque.x1 != x1)
                              update_region( stack, state, i-1, x1, opaque.y1, opaque.x1-1, opaque.y2 );

                         /* upper */
                         if (opaque.y1 != y1)
                              update_region( stack, state, i-1, x1, y1, x2, opaque.y1-1 );

                         /* right */
                         if (opaque.x2 != x2)
                              update_region( stack, state, i-1, opaque.x2+1, opaque.y1, x2, opaque.y2 );

                         /* lower */
                         if (opaque.y2 != y2)
                              update_region( stack, state, i-1, x1, opaque.y2+1, x2, y2 );
                    }

                    /* left */
                    if (opaque.x1 != region.x1) {
                         DFBRegion r = { region.x1, opaque.y1,
                              opaque.x1 - 1, opaque.y2};
                         draw_window( window, state, &r, true );
                    }

                    /* upper */
                    if (opaque.y1 != region.y1) {
                         DFBRegion r = { region.x1, region.y1,
                              region.x2, opaque.y1 - 1};
                         draw_window( window, state, &r, true );
                    }

                    /* right */
                    if (opaque.x2 != region.x2) {
                         DFBRegion r = { opaque.x2 + 1, opaque.y1,
                              region.x2, opaque.y2};
                         draw_window( window, state, &r, true );
                    }

                    /* lower */
                    if (opaque.y2 != region.y2) {
                         DFBRegion r = { region.x1, opaque.y2 + 1,
                              region.x2, region.y2};
                         draw_window( window, state, &r, true );
                    }

                    /* inner */
                    draw_window( window, state, &opaque, false );
               }
          }
          else {
               if ((window->opacity < 0xff) ||
                   (window->options & (DWOP_COLORKEYING | DWOP_ALPHACHANNEL))) {
                    /* draw everything below */
                    update_region( stack, state, i-1, x1, y1, x2, y2 );
               }
               else {
                    /* left */
                    if (region.x1 != x1)
                         update_region( stack, state, i-1, x1, region.y1, region.x1-1, region.y2 );

                    /* upper */
                    if (region.y1 != y1)
                         update_region( stack, state, i-1, x1, y1, x2, region.y1-1 );

                    /* right */
                    if (region.x2 != x2)
                         update_region( stack, state, i-1, region.x2+1, region.y1, x2, region.y2 );

                    /* lower */
                    if (region.y2 != y2)
                         update_region( stack, state, i-1, x1, region.y2+1, x2, y2 );
               }

               draw_window( window, state, &region, true );
          }
     }
     else
          draw_background( stack, state, &region );
}

static void
repaint_stack( CoreWindowStack     *stack,
               DFBRegion           *region,
               DFBSurfaceFlipFlags  flags )
{
     DisplayLayer *layer   = dfb_layer_at( stack->layer_id );
     CoreSurface  *surface = dfb_layer_surface( layer );
     CardState    *state   = dfb_layer_state( layer );

     if (!surface)
          return;
     
     if (!dfb_region_intersect( region, 0, 0,
                                surface->width - 1, surface->height - 1 ))
          return;

     if (dfb_layer_lease( layer ))
          return;

     state->destination = surface;
     state->clip        = *region;
     state->modified   |= SMF_DESTINATION | SMF_CLIP;

     update_region( stack, state, stack->num_windows - 1,
                    region->x1, region->y1, region->x2, region->y2 );

     if (surface->caps & (DSCAPS_FLIPPING | DSCAPS_TRIPLE)) {
          if (region->x1 == 0 &&
              region->y1 == 0 &&
              region->x2 == surface->width - 1 &&
              region->y2 == surface->height - 1) {
               dfb_layer_flip_buffers( layer, flags );
          }
          else {
               DFBRectangle rect = { region->x1, region->y1,
                    region->x2 - region->x1 + 1,
                    region->y2 - region->y1 + 1};

               if ((flags & DSFLIP_WAITFORSYNC) == DSFLIP_WAITFORSYNC)
                    dfb_layer_wait_vsync( layer );

               dfb_back_to_front_copy( surface, &rect );
               dfb_layer_update_region( layer, region, flags );

               if ((flags & DSFLIP_WAITFORSYNC) == DSFLIP_WAIT)
                    dfb_layer_wait_vsync( layer );
          }
     }
     else
          dfb_layer_update_region( layer, region, flags );

     dfb_layer_release( layer, false );

     state->destination = NULL;
}

/*
     recurseve procedure to call repaint
     skipping opaque windows that are above the window
     that changed
*/
static void
wind_of_change( CoreWindowStack     *stack,
                DFBRegion           *region,
                DFBSurfaceFlipFlags  flags,
                int                  current,
                int                  changed )
{
     DFB_ASSERT(current>=changed);

     /*
          got to the window that changed, redraw.
     */
     if (current == changed)
          repaint_stack( stack, region, flags );
     else {
          CoreWindow *window = stack->windows[current];
          DFBRegion opaque;

          /*
               can skip opaque region
          */
          if ((
              //can skip all opaque window?
              (window->opacity == 0xff) &&
              !(window->options & (DWOP_COLORKEYING | DWOP_ALPHACHANNEL)) &&
              (opaque=*region,dfb_region_intersect( &opaque,
                                                    window->x, window->y,
                                                    window->x + window->width - 1,
                                                    window->y + window->height -1 ) ) 
              )||(
                 //can skip opaque region?
                 (window->options & DWOP_ALPHACHANNEL) &&
                 (window->options & DWOP_OPAQUE_REGION) &&
                 (window->opacity == 0xff) &&
                 !(window->options & DWOP_COLORKEYING) &&
                 (opaque=*region,dfb_region_intersect( &opaque,
                                                       window->x + window->opaque.x1,
                                                       window->y + window->opaque.y1,
                                                       window->x + window->opaque.x2,
                                                       window->y + window->opaque.y2 )) 
                 )) {
               /* left */
               if (opaque.x1 != region->x1) {
                    DFBRegion update = { region->x1, opaque.y1, opaque.x1-1, opaque.y2};
                    wind_of_change( stack, &update, flags, current-1, changed );
               }
               /* upper */
               if (opaque.y1 != region->y1) {
                    DFBRegion update = { region->x1, region->y1, region->x2, opaque.y1-1};
                    wind_of_change( stack, &update, flags, current-1, changed );
               }
               /* right */
               if (opaque.x2 != region->x2) {
                    DFBRegion update = { opaque.x2+1, opaque.y1, region->x2, opaque.y2};
                    wind_of_change( stack, &update, flags, current-1, changed );
               }
               /* lower */
               if (opaque.y2 != region->y2) {
                    DFBRegion update = { region->x1, opaque.y2+1, region->x2, region->y2};
                    wind_of_change( stack, &update, flags, current-1, changed );
               }
          }
          /*
               pass through
          */
          else
               wind_of_change( stack, region, flags, current-1, changed );
     }
}

static void
repaint_stack_for_window( CoreWindowStack     *stack,
                          DFBRegion           *region,
                          DFBSurfaceFlipFlags  flags,
                          int                  window )
{
     if (stack->num_windows && window >= 0) {
          DFB_ASSERT(window < stack->num_windows);

          wind_of_change( stack, region, flags, stack->num_windows - 1, window ); 
     }
     else
          repaint_stack( stack, region, flags );     
}

