/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.sling.sitemap.impl;

import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.apache.sling.api.resource.path.PathSet;
import org.apache.sling.commons.scheduler.Scheduler;
import org.apache.sling.event.jobs.Job;
import org.apache.sling.event.jobs.JobManager;
import org.apache.sling.serviceusermapping.ServiceUserMapped;
import org.apache.sling.sitemap.SitemapGeneratorManager;
import org.jetbrains.annotations.Nullable;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.metatype.annotations.AttributeDefinition;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.stream.Collectors;

import static org.apache.sling.sitemap.SitemapUtil.findSitemapRoots;

@Component(
        service = {SitemapScheduler.class, Runnable.class},
        configurationPolicy = ConfigurationPolicy.REQUIRE,
        property = {
                Scheduler.PROPERTY_SCHEDULER_CONCURRENT + ":Boolean=false",
                Scheduler.PROPERTY_SCHEDULER_RUN_ON + "=" + Scheduler.VALUE_RUN_ON_SINGLE,
                Scheduler.PROPERTY_SCHEDULER_THREAD_POOL + "=" + SitemapScheduler.THREADPOOL_NAME
        }
)
@Designate(ocd = SitemapScheduler.Configuration.class, factory = true)
public class SitemapScheduler implements Runnable {

    @ObjectClassDefinition(name = "Apache Sling Sitemap - Scheduler")
    @interface Configuration {

        @AttributeDefinition(name = "Name", description = "The name of the scheduler configuration")
        String scheduler_name();

        @AttributeDefinition(name = "Schedule", description = "A cron expression defining the schedule at which the " +
                "sitemap generation jobs will be scheduled.")
        String scheduler_expression();

        @AttributeDefinition(name = "Include Generators", description = "A list of full qualified class names of " +
                "SitemapGenerator implementations. If set only the listed SitemapGenerators will be called. If left " +
                "empty all will be called.")
        String[] includeGenerators() default {};

        @AttributeDefinition(name = "Exclude Generators", description = "A list of full qualified class names of " +
                "SitemapGenerator implementations. If set the listed SitemapGenerators will not be called. If left " +
                "empty all will be called.")
        String[] excludeGenerators() default {};

        @AttributeDefinition(name = "Names", description = "A list of names. If set only sitemaps for the given " +
                "names will be generated by. If left empty all will be generated.")
        String[] names() default {};

        @AttributeDefinition(name = "Search Path", description = "The path under which sitemap roots should be " +
                "searched for")
        String searchPath() default "/content";

        @AttributeDefinition(name = "Include Paths", description = "A list of paths that should be included by the scheduler. "
            + "If left empty, all sitemap roots in the configured search path will be included. Absolute paths and glob patterns "
            + "are supported.")
        String[] includePaths() default {};

        @AttributeDefinition(name = "Exclude Paths", description = "A list of paths that should be excluded by the scheduler. "
            + "If left empty, no sitemap roots in the configured search path will be excluded. Absolute paths and glob patterns "
            + "are supported.")
        String[] excludePaths() default {};
    }

    public static final String THREADPOOL_NAME = "org-apache-sling-sitemap";

    private Logger log;
    private static final Map<String, Object> AUTH = Collections.singletonMap(ResourceResolverFactory.SUBSERVICE,
            "sitemap-reader");

    @Reference
    private JobManager jobManager;
    @Reference
    private ResourceResolverFactory resourceResolverFactory;
    @Reference
    private SitemapGeneratorManager generatorManager;
    @Reference(target = "(subServiceName=sitemap-reader)")
    private ServiceUserMapped serviceUserMapped;

    private Set<String> names;
    private Set<String> includeGenerators;
    private Set<String> excludeGenerators;
    private String searchPath;
    private PathSet includePaths;
    private PathSet excludePaths;

    @Activate
    protected void activate(Configuration configuration) {
        log = LoggerFactory.getLogger(SitemapScheduler.class.getName() + '~' + configuration.scheduler_name());
        includeGenerators = asSet(configuration.includeGenerators());
        excludeGenerators = asSet(configuration.excludeGenerators());
        names = asSet(configuration.names());
        searchPath = configuration.searchPath();
        if (configuration.includePaths().length > 0) {
            includePaths = PathSet.fromStringCollection(Arrays.asList(configuration.includePaths()));
        }
        if (configuration.excludePaths().length > 0) {
            excludePaths = PathSet.fromStringCollection(Arrays.asList(configuration.excludePaths()));
        }
    }

    @Override
    public void run() {
        schedule(null);
    }

    public void schedule(@Nullable Collection<String> includeNames) {
        try (ResourceResolver resolver = resourceResolverFactory.getServiceResourceResolver(AUTH)) {
            Iterator<Resource> sitemapRoots = findSitemapRoots(resolver, searchPath);
            while (sitemapRoots.hasNext()) {
                schedule(sitemapRoots.next(), includeNames);
            }
        } catch (LoginException ex) {
            log.warn("Failed start sitemap jobs: {}", ex.getMessage(), ex);
        }
    }

    public void schedule(Resource sitemapRoot, @Nullable Collection<String> includeNames) {
        if (isExcluded(sitemapRoot)) {
            return;
        }

        Set<String> configuredNames = getApplicableNames(sitemapRoot);

        if (includeNames != null) {
            configuredNames.retainAll(includeNames);
        }

        for (String applicableName : configuredNames) {
            addJob(sitemapRoot.getPath(), applicableName);
        }
    }

    /**
     * Returns the names for the given sitemap root this {@link SitemapScheduler} is applicable to. This depends on the
     * configured generators. If no generators were configured the names of all are returned. If some are configured
     * the names provided only by those where the class name matches are returned.
     *
     * @param sitemapRoot
     * @return
     */
    public Set<String> getApplicableNames(Resource sitemapRoot) {
        if (isExcluded(sitemapRoot)) {
            return Collections.emptySet();
        }

        Set<String> onDemandNames = generatorManager.getOnDemandNames(sitemapRoot);
        Set<String> toSchedule = generatorManager.getGenerators(sitemapRoot).entrySet().stream()
                .filter(entry -> includeGenerators == null
                        || includeGenerators.contains(entry.getValue().getClass().getName()))
                .filter(entry -> excludeGenerators == null
                        || !excludeGenerators.contains(entry.getValue().getClass().getName()))
                .filter(entry -> !onDemandNames.contains(entry.getKey()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toSet());

        // limit to  the configured names
        if (names != null) {
            toSchedule.retainAll(names);
        }

        return toSchedule;
    }

    protected boolean isExcluded(Resource sitemapRoot) {
        // verify that the sitemapRoot is in the schedulers search path
        if (!sitemapRoot.getPath().equals(searchPath) && !sitemapRoot.getPath().startsWith(searchPath + "/")) {
            log.debug("Exclude sitemap root outside of the configured search path '{}': {}", searchPath, sitemapRoot.getPath());
            return true;
        }

        // verify if the sitemapRoot is included
        if (includePaths != null && includePaths.matches(sitemapRoot.getPath()) == null) {
            log.debug("Sitemap root is not included: {}", sitemapRoot.getPath());
            return true;
        }

        // verify if the sitemapRoot is not excluded
        if (excludePaths != null && excludePaths.matches(sitemapRoot.getPath()) != null) {
            log.debug("Sitemap root is explicitly excluded: {}", sitemapRoot.getPath());
            return true;
        }

        return false;
    }

    protected void addJob(String sitemapRoot, String applicableName) {
        Map<String, Object> jobProperties = new HashMap<>();
        jobProperties.put(SitemapGeneratorExecutor.JOB_PROPERTY_SITEMAP_NAME, applicableName);
        jobProperties.put(SitemapGeneratorExecutor.JOB_PROPERTY_SITEMAP_ROOT, sitemapRoot);
        Job job = jobManager.addJob(SitemapGeneratorExecutor.JOB_TOPIC, jobProperties);
        log.debug("Added job {}", job.getId());
    }

    @Nullable
    private static Set<String> asSet(@Nullable String[] configuration) {
        if (configuration == null || configuration.length == 0) {
            return null;
        }

        Set<String> result = Arrays.stream(configuration)
                .filter(Objects::nonNull)
                .filter(entry -> !"".equals(entry.trim()))
                .collect(Collectors.toSet());

        return result.isEmpty() ? null : result;
    }
}
