/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.client.java.metrics;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import org.apache.rocketmq.client.apis.consumer.PushConsumer;
import org.apache.rocketmq.client.apis.consumer.SimpleConsumer;
import org.apache.rocketmq.client.java.hook.Attribute;
import org.apache.rocketmq.client.java.hook.AttributeKey;
import org.apache.rocketmq.client.java.hook.MessageHookPoints;
import org.apache.rocketmq.client.java.hook.MessageHookPointsStatus;
import org.apache.rocketmq.client.java.hook.MessageInterceptor;
import org.apache.rocketmq.client.java.hook.MessageInterceptorContext;
import org.apache.rocketmq.client.java.impl.Client;
import org.apache.rocketmq.client.java.message.GeneralMessage;
import org.apache.rocketmq.client.java.metrics.ClientMeterManager;
import org.apache.rocketmq.client.java.metrics.HistogramEnum;
import org.apache.rocketmq.client.java.metrics.InvocationStatus;
import org.apache.rocketmq.client.java.metrics.MetricLabels;
import org.apache.rocketmq.shaded.com.google.common.base.Stopwatch;
import org.apache.rocketmq.shaded.io.opentelemetry.api.common.Attributes;
import org.apache.rocketmq.shaded.org.slf4j.Logger;
import org.apache.rocketmq.shaded.org.slf4j.LoggerFactory;

public class MessageMeterInterceptor
implements MessageInterceptor {
    static final AttributeKey<Stopwatch> SEND_STOPWATCH_KEY = AttributeKey.create("send_stopwatch");
    static final AttributeKey<Stopwatch> CONSUME_STOPWATCH_KEY = AttributeKey.create("consume_stopwatch");
    private static final Logger log = LoggerFactory.getLogger(MessageMeterInterceptor.class);
    private final Client client;
    private final ClientMeterManager meterManager;

    public MessageMeterInterceptor(Client client, ClientMeterManager meterManager) {
        this.client = client;
        this.meterManager = meterManager;
    }

    private void doBeforeSendMessage(MessageInterceptorContext context) {
        context.putAttribute(SEND_STOPWATCH_KEY, Attribute.create(Stopwatch.createStarted()));
    }

    private void doAfterSendMessage(MessageInterceptorContext context, List<GeneralMessage> messages) {
        Attribute<Stopwatch> stopwatchAttr = context.getAttribute(SEND_STOPWATCH_KEY);
        if (null == stopwatchAttr) {
            return;
        }
        for (GeneralMessage message : messages) {
            Duration duration = stopwatchAttr.get().elapsed();
            InvocationStatus status = MessageHookPointsStatus.OK.equals((Object)context.getStatus()) ? InvocationStatus.SUCCESS : InvocationStatus.FAILURE;
            Attributes attributes = Attributes.builder().put(MetricLabels.TOPIC, message.getTopic()).put(MetricLabels.CLIENT_ID, this.client.getClientId().toString()).put(MetricLabels.INVOCATION_STATUS, status.getName()).build();
            this.meterManager.record(HistogramEnum.SEND_COST_TIME, attributes, duration.toMillis());
        }
    }

    private void doAfterReceiveMessage(List<GeneralMessage> messages) {
        if (messages.isEmpty()) {
            return;
        }
        String consumerGroup = null;
        if (this.client instanceof PushConsumer) {
            consumerGroup = ((PushConsumer)((Object)this.client)).getConsumerGroup();
        }
        if (this.client instanceof SimpleConsumer) {
            consumerGroup = ((SimpleConsumer)((Object)this.client)).getConsumerGroup();
        }
        if (null == consumerGroup) {
            log.error("[Bug] consumerGroup is not recognized, clientId={}", (Object)this.client.getClientId());
            return;
        }
        GeneralMessage message = messages.iterator().next();
        Optional<Long> optionalTransportDeliveryTimestamp = message.getTransportDeliveryTimestamp();
        if (!optionalTransportDeliveryTimestamp.isPresent()) {
            return;
        }
        long transportDeliveryTimestamp = optionalTransportDeliveryTimestamp.get();
        long currentTimeMillis = System.currentTimeMillis();
        long latency = currentTimeMillis - transportDeliveryTimestamp;
        if (0L > latency) {
            log.debug("latency is negative, latency={}ms, currentTimeMillis={}, transportDeliveryTimestamp={}", latency, currentTimeMillis, transportDeliveryTimestamp);
            return;
        }
        Attributes attributes = Attributes.builder().put(MetricLabels.TOPIC, message.getTopic()).put(MetricLabels.CONSUMER_GROUP, consumerGroup).put(MetricLabels.CLIENT_ID, this.client.getClientId().toString()).build();
        this.meterManager.record(HistogramEnum.DELIVERY_LATENCY, attributes, latency);
    }

    private void doBeforeConsumeMessage(MessageInterceptorContext context, List<GeneralMessage> messages) {
        if (messages.isEmpty()) {
            return;
        }
        String consumerGroup = null;
        if (this.client instanceof PushConsumer) {
            consumerGroup = ((PushConsumer)((Object)this.client)).getConsumerGroup();
        }
        if (null == consumerGroup) {
            log.error("[Bug] consumerGroup is not recognized, clientId={}", (Object)this.client.getClientId());
            return;
        }
        GeneralMessage message = messages.iterator().next();
        Optional<Long> optionalDecodeTimestamp = message.getDecodeTimestamp();
        if (!optionalDecodeTimestamp.isPresent()) {
            return;
        }
        long decodeTimestamp = optionalDecodeTimestamp.get();
        Attributes attributes = Attributes.builder().put(MetricLabels.TOPIC, message.getTopic()).put(MetricLabels.CONSUMER_GROUP, consumerGroup).put(MetricLabels.CLIENT_ID, this.client.getClientId().toString()).build();
        long latency = System.currentTimeMillis() - decodeTimestamp;
        this.meterManager.record(HistogramEnum.AWAIT_TIME, attributes, latency);
        context.putAttribute(CONSUME_STOPWATCH_KEY, Attribute.create(Stopwatch.createStarted()));
    }

    private void doAfterConsumeMessage(MessageInterceptorContext context, List<GeneralMessage> messages) {
        if (!(this.client instanceof PushConsumer)) {
            log.error("[Bug] current client is not push consumer, clientId={}", (Object)this.client.getClientId());
            return;
        }
        Attribute<Stopwatch> stopwatchAttr = context.getAttribute(CONSUME_STOPWATCH_KEY);
        if (null == stopwatchAttr) {
            return;
        }
        PushConsumer pushConsumer = (PushConsumer)((Object)this.client);
        MessageHookPointsStatus status = context.getStatus();
        for (GeneralMessage message : messages) {
            InvocationStatus invocationStatus = MessageHookPointsStatus.OK.equals((Object)status) ? InvocationStatus.SUCCESS : InvocationStatus.FAILURE;
            Attributes attributes = Attributes.builder().put(MetricLabels.TOPIC, message.getTopic()).put(MetricLabels.CONSUMER_GROUP, pushConsumer.getConsumerGroup()).put(MetricLabels.CLIENT_ID, this.client.getClientId().toString()).put(MetricLabels.INVOCATION_STATUS, invocationStatus.getName()).build();
            Duration duration = stopwatchAttr.get().elapsed();
            this.meterManager.record(HistogramEnum.PROCESS_TIME, attributes, duration.toMillis());
        }
    }

    @Override
    public void doBefore(MessageInterceptorContext context, List<GeneralMessage> messages) {
        if (!this.meterManager.isEnabled()) {
            return;
        }
        MessageHookPoints hookPoints = context.getMessageHookPoints();
        switch (hookPoints) {
            case SEND: {
                this.doBeforeSendMessage(context);
                break;
            }
            case CONSUME: {
                this.doBeforeConsumeMessage(context, messages);
                break;
            }
        }
    }

    @Override
    public void doAfter(MessageInterceptorContext context, List<GeneralMessage> messages) {
        if (!this.meterManager.isEnabled()) {
            return;
        }
        MessageHookPoints hookPoints = context.getMessageHookPoints();
        switch (hookPoints) {
            case SEND: {
                this.doAfterSendMessage(context, messages);
                break;
            }
            case RECEIVE: {
                this.doAfterReceiveMessage(messages);
                break;
            }
            case CONSUME: {
                this.doAfterConsumeMessage(context, messages);
                break;
            }
        }
    }
}

