/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.portals.applications.webcontent2.proxy.RequestContext;
import org.apache.portals.applications.webcontent2.proxy.util.RequestUtils;
import org.apache.portals.applications.webcontent2.rewriter.Sink;

/**
 * {@link javax.servlet.http.HttpServletRequest} / {@link javax.servlet.http.HttpServletResponse}
 * abstraction needed for reverse proxy processing.
 */
public class ServletRequestContext implements RequestContext
{

    /**
     * Base relative path of the request before the request path info.
     * Typically, it is either the concatenation of the request context path and
     * the servlet path, or the concatenation of the request context path and
     * the filter mapping path prefix.
     */
    private String requestBasePath;

    /**
     * Underlying {@link javax.servlet.http.HttpServletRequest}
     */
    private final HttpServletRequest request;

    /**
     * Underlying {@link javax.servlet.http.HttpServletResponse}
     */
    private final HttpServletResponse response;

    /**
     * Underlying {@link Sink} used during reverse proxy processing.
     */
    private Sink sink;

    /**
     * Constructs a <code>ServletRequestContext</code>
     * with the given {@link javax.servlet.http.HttpServletRequest}
     * and {@link javax.servlet.http.HttpServletResponse}.
     * @param request
     * @param response
     */
    public ServletRequestContext(final HttpServletRequest request, final HttpServletResponse response)
    {
        this.request = request;
        this.response = response;
    }

    /**
     * Returns the underlying {@link javax.servlet.http.HttpServletRequest}.
     * @return
     */
    public HttpServletRequest getServletRequest()
    {
        return request;
    }

    /**
     * Returns the underlying {@link javax.servlet.http.HttpServletResponse}.
     * @return
     */
    public HttpServletResponse getServletResponse()
    {
        return response;
    }

    /**
     * {@inheritDoc}
     */
    public boolean isSecure()
    {
        return request.isSecure();
    }

    /**
     * {@inheritDoc}
     */
    public String getScheme()
    {
        return request.getScheme();
    }

    /**
     * {@inheritDoc}
     */
    public String getServerName()
    {
        return request.getServerName();
    }

    /**
     * {@inheritDoc}
     */
    public int getServerPort()
    {
        return request.getServerPort();
    }

    /**
     * {@inheritDoc}
     */
    public String getMethod()
    {
        return request.getMethod();
    }

    /**
     * {@inheritDoc}
     */
    public String getRequestBasePath()
    {
        if (requestBasePath != null)
        {
            return requestBasePath;
        }

        StringBuilder sb = new StringBuilder(20);
        sb.append(RequestUtils.getContextPath(request));

        String servletPath = RequestUtils.getServletPath(request);

        if (servletPath != null)
        {
            sb.append(servletPath);
        }

        return sb.toString();
    }

    public void setRequestBasePath(String requestBasePath)
    {
        this.requestBasePath = requestBasePath;
    }

    /**
     * {@inheritDoc}
     */
    public String getPathInfo()
    {
        if (requestBasePath != null)
        {
            String requestURI = RequestUtils.getRequestURI(request);

            if (requestURI.startsWith(requestBasePath))
            {
                return requestURI.substring(requestBasePath.length());
            }
        }

        return RequestUtils.getPathInfo(request);
    }

    /**
     * {@inheritDoc}
     */
    public Object getAttribute(final String name)
    {
        return request.getAttribute(name);
    }

    /**
     * {@inheritDoc}
     */
    public String getQueryString()
    {
        return request.getQueryString();
    }

    /**
     * {@inheritDoc}
     */
    public String getHeader(String name)
    {
        return request.getHeader(name);
    }

    /**
     * {@inheritDoc}
     */
    public InputStream getInputStream() throws IOException
    {
        return request.getInputStream();
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("rawtypes")
    public Enumeration getHeaderNames()
    {
        return request.getHeaderNames();
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("rawtypes")
    public Enumeration getHeaders(String name)
    {
        return request.getHeaders(name);
    }

    /**
     * {@inheritDoc}
     */
    public void sendRedirect(String location) throws IOException
    {
        response.sendRedirect(location);
    }

    /**
     * {@inheritDoc}
     */
    public void setHeader(String name, String value)
    {
        response.setHeader(name, value);
    }

    /**
     * {@inheritDoc}
     */
    public void setIntHeader(String name, int value)
    {
        response.setIntHeader(name, value);
    }

    /**
     * {@inheritDoc}
     */
    public void addCookie(Cookie cookie)
    {
        response.addCookie(cookie);
    }

    /**
     * {@inheritDoc}
     */
    public void setStatus(int sc)
    {
        response.setStatus(sc);
    }

    /**
     * {@inheritDoc}
     */
    public Sink getSink()
    {
        if (sink == null)
        {
            sink = new HttpServletResponseSink(response);
        }

        return sink;
    }

}
