/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.portals.applications.webcontent2.rewriter.impl;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;

import org.apache.portals.applications.webcontent2.rewriter.Sink;

/**
 * Acts as an holder for a content rewriting {@link Sink} in the form of byte or character stream.
 */
public class StreamSink implements Sink
{

    /**
     * Underlying {@link java.io.OutputStream} of this sink.
     */
    private OutputStream output;

    /**
     * Character encoding to be used to create a {@link java.io.Writer}
     * from the underlying byte output stream.
     */
    private String encoding;

    /**
     * Underlying {@link java.io.Writer} of this sink.
     */
    private Writer writer;

    /**
     * Construct a StreamSink for a byte stream.
     * @param output
     */
    public StreamSink(OutputStream output)
    {
        this(output, null);
    }

    /**
     * Construct a StreamSink for a byte stream with the specified <code>encoding</code>.
     * The <code>encoding</code> is used when {@link #getWriter()} is invoked to create a
     * writer from the underlying byte stream.
     * @param output
     * @param encoding
     */
    public StreamSink(OutputStream output, String encoding)
    {
        if (output == null)
        {
            throw new IllegalArgumentException("output must not be null.");
        }

        this.output = output;
        this.encoding = encoding;
    }

    /**
     * Construct a StreamSink for a character stream.
     * @param writer
     */
    public StreamSink(Writer writer)
    {
        if (writer == null)
        {
            throw new IllegalArgumentException("writer must not be null.");
        }

        this.writer = writer;
    }

    /**
     * {@inheritDoc}
     * <p>
     * Either this method or {@link #getWriter} may be called to write the body, not both.
     * </p>
     */
    public OutputStream getOutputStream() throws IOException
    {
        if (writer != null)
        {
            throw new IllegalStateException("getWriter() has already been called on this sink.");
        }

        return output;
    }

    /**
     * {@inheritDoc}
     * <p>
     * Either this method or {@link #getOutputStream} may be called to write the body, not both.
     * </p>
     */
    public Writer getWriter() throws IOException
    {
        if (output != null)
        {
            throw new IllegalStateException("getOutputStream() has already been called on this sink.");
        }

        if (writer == null)
        {
            if (encoding == null)
            {
                writer = new OutputStreamWriter(output);
            }
            else
            {
                writer = new OutputStreamWriter(output, encoding);
            }
        }

        return writer;
    }
}
