/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.util;

import javax.servlet.http.HttpServletRequest;

import org.apache.portals.applications.webcontent2.proxy.RequestContext;

/**
 * Utility to read the context path, request URI, servlet path, path info, etc.
 * from an {@link javax.servlet.http.HttpServletRequest}.
 * This utility does care both situations: whether or not the current request was dispatched.
 * So, with this utility, you don't have to read servlet request attributes manually.
 * <p>
 * <em>Note:</em> Please see the Java Servlet Specification for the detail on the related request attributes.
 * </p>
 */
public class RequestUtils
{

    private RequestUtils()
    {
    }

    /**
     * Finds and returns the servlet context path in the following order.
     * <ol>
     * <li>returns <code>javax.servlet.include.context_path</code> request attribute if found.</li>
     * <li>returns <code>javax.servlet.forward.context_path</code> request attribute if found.</li>
     * <li>returns {@link javax.servlet.http.HttpServletRequest#getContextPath().</li>
     * </ol>
     * 
     * @param request
     * @return
     */
    public static String getContextPath(HttpServletRequest request)
    {
        String contextPath = (String) request.getAttribute("javax.servlet.include.context_path");

        if (contextPath == null)
        {
            contextPath = (String) request.getAttribute("javax.servlet.forward.context_path");
        }

        if (contextPath != null)
        {
            return contextPath;
        }

        return request.getContextPath();
    }

    /**
     * Finds and returns the request URI in the following order.
     * <ol>
     * <li>returns <code>javax.servlet.include.request_uri</code> request attribute if found.</li>
     * <li>returns <code>javax.servlet.forward.request_uri</code> request attribute if found.</li>
     * <li>returns {@link javax.servlet.http.HttpServletRequest#getRequestURI().</li>
     * </ol>
     * @param request
     * @return
     */
    public static String getRequestURI(HttpServletRequest request)
    {
        String requestURI = (String) request.getAttribute("javax.servlet.include.request_uri");

        if (requestURI == null)
        {
            requestURI = (String) request.getAttribute("javax.servlet.forward.request_uri");
        }

        if (requestURI != null)
        {
            return requestURI;
        }

        return request.getRequestURI();
    }

    /**
     * Finds and returns the request URI in the following order.
     * <ol>
     * <li>returns <code>javax.servlet.include.servlet_path</code> request attribute if found.</li>
     * <li>returns <code>javax.servlet.forward.servlet_path</code> request attribute if found.</li>
     * <li>returns {@link javax.servlet.http.HttpServletRequest#getServletPath().</li>
     * </ol>
     * @param request
     * @return
     */
    public static String getServletPath(HttpServletRequest request)
    {
        String servletPath = (String) request.getAttribute("javax.servlet.include.servlet_path");

        if (servletPath == null)
        {
            servletPath = (String) request.getAttribute("javax.servlet.forward.servlet_path");
        }

        if (servletPath != null)
        {
            return servletPath;
        }

        return request.getServletPath();
    }

    /**
     * Finds and returns the request URI in the following order.
     * <ol>
     * <li>returns <code>javax.servlet.include.path_info</code> request attribute if found.</li>
     * <li>returns <code>javax.servlet.forward.path_info</code> request attribute if found.</li>
     * <li>returns {@link javax.servlet.http.HttpServletRequest#getPathInfo().</li>
     * </ol>
     * @param request
     * @return
     */
    public static String getPathInfo(HttpServletRequest request)
    {
        String pathInfo = (String) request.getAttribute("javax.servlet.include.path_info");

        if (pathInfo == null)
        {
            pathInfo = (String) request.getAttribute("javax.servlet.forward.path_info");
        }

        if (pathInfo != null)
        {
            return pathInfo;
        }

        return request.getPathInfo();
    }

    /**
     * Returns true if the current request was dispatched by either
     * {@link javax.servlet.RequestDispatcher#include(javax.servlet.ServletRequest, javax.servlet.ServletResponse)}
     * or {@link javax.servlet.RequestDispatcher#forward(javax.servlet.ServletRequest, javax.servlet.ServletResponse)}.
     * Otherwise, it should return false.
     * @param requestContext
     * @return
     */
    public static boolean isDispatched(RequestContext requestContext)
    {
        return (requestContext.getAttribute("javax.servlet.include.context_path") != null 
                        || requestContext.getAttribute("javax.servlet.forward.context_path") != null);
    }
}
