/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.command;

import java.io.IOException;

import org.apache.commons.chain.Command;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.protocol.HttpContext;
import org.apache.portals.applications.webcontent2.proxy.HttpClientContextBuilder;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;


/**
 * {@link Command} responsible for executing the internal {@link HttpClient}
 * with the {@link HttpRequestBase} instance.
 * <p>
 * If {@link #getHttpClientContextBuilder()} returns a non-null instance, then
 * this command executes the internal {@link HttpClient}
 * with the {@link HttpRequestBase} instance and a {@link HttpContext} built by
 * {@link #getHttpClientContextBuilder()}.
 * </p>
 */
public class ExecuteHttpClientCommand extends AbstractProxyCommand
{

    /**
     * Internal {@link HttpClientContextBuilder}.
     */
    private HttpClientContextBuilder httpClientContextBuilder;

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean executeInternal(final ProxyContext context) throws ReverseProxyException, IOException
    {
        HttpRequestBase httpRequest = context.getHttpRequest();

        HttpClient httpClient = context.getHttpClient();
        HttpResponse httpResponse = null;

        if (getHttpClientContextBuilder() == null)
        {
            httpResponse = httpClient.execute(httpRequest);
        }
        else
        {
            httpResponse = httpClient.execute(httpRequest, getHttpClientContextBuilder().build());
        }

        context.setHttpResponse(httpResponse);

        return false;
    }

    /**
     * Returns the internal {@link HttpClientContextBuilder}.
     * @return
     */
    public HttpClientContextBuilder getHttpClientContextBuilder()
    {
        return httpClientContextBuilder;
    }

    /**
     * Sets the internal {@link HttpClientContextBuilder}.
     * @param httpClientContextBuilder
     */
    public void setHttpClientContextBuilder(HttpClientContextBuilder httpClientContextBuilder)
    {
        this.httpClientContextBuilder = httpClientContextBuilder;
    }
}
