/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.command;

import java.io.IOException;

import org.apache.commons.chain.Command;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * {@link Command} responsible for cleaning up all the resources
 * (such as <code>HttpRequestBase</code>, <code>HttpResponse</code>, <code>HttpEntity</code> and <code>HttpClient</code>)
 * used during reverse proxy processing in the previous steps.
 */
public class CleanupCommand extends AbstractProxyCommand
{

    private static Logger log = LoggerFactory.getLogger(CleanupCommand.class);

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean executeInternal(final ProxyContext context) throws ReverseProxyException, IOException
    {
        HttpRequestBase httpRequest = context.getHttpRequest();

        if (httpRequest != null)
        {
            try
            {
                httpRequest.abort();
            }
            catch (Exception e)
            {
                if (log.isDebugEnabled())
                {
                    log.warn("Failed to abort http request.", e);
                }
                else
                {
                    log.warn("Failed to abort http request. " + e);
                }
            }
        }

        HttpResponse httpResponse = context.getHttpResponse();

        if (httpResponse != null)
        {
            HttpEntity httpEntity = httpResponse.getEntity();

            if (httpEntity != null)
            {
                try
                {
                    EntityUtils.consume(httpEntity);
                }
                catch (Exception e)
                {
                    if (log.isDebugEnabled())
                    {
                        log.warn("Failed to consume http entity.", e);
                    }
                    else
                    {
                        log.warn("Failed to consume http entity. " + e);
                    }
                }
            }
        }

        HttpClient httpClient = context.getHttpClient();

        if (httpClient instanceof CloseableHttpClient)
        {
            try
            {
                ((CloseableHttpClient) httpClient).close();
            }
            catch (Exception e)
            {
                if (log.isDebugEnabled())
                {
                    log.warn("Failed to close http client.", e);
                }
                else
                {
                    log.warn("Failed to close http client. " + e);
                }
            }
        }

        return false;
    }
}
