/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.rewriter.htmlcleaner;

import java.util.ArrayList;
import java.util.List;

import org.htmlcleaner.TagNode;
import org.htmlcleaner.conditional.ITagNodeCondition;

/**
 * Simple AND node condition checker implementation for HtmlCleaner based content rewriting process.
 * This class is for combining existing {@link ITagNodeCondition} checkers and for checking if
 * all the {@link ITagNodeCondition} checker instances return true.
 * <p>
 * This class can basically be given multiple {@link ITagNodeCondition} checker instances.
 * This class iterates the multiple {@link ITagNodeCondition} checker instances
 * and invokes {@link ITagNodeCondition#satisfy(TagNode)} on each.
 * If and only if every call on {@link ITagNodeCondition#satisfy(TagNode)} returns true,
 * then the {@link #satisfy(TagNode)} method of this class returns true.
 * Otherwise, it will return false.
 * </p>
 */
public class AndTagNodeCondition implements ITagNodeCondition
{

    private List<ITagNodeCondition> conditionsList;

    /**
     * Zero-argument default constructor.
     */
    public AndTagNodeCondition()
    {
    }

    /**
     * Adds a {@link ITagNodeCondition} checker instance.
     * @param condition
     */
    public void addTagNodeCondition(ITagNodeCondition condition)
    {
        if (conditionsList == null)
        {
            conditionsList = new ArrayList<ITagNodeCondition>();
        }

        conditionsList.add(condition);
    }

    /**
     * {@inheritDoc}
     * <p>
     * This method iterates the multiple {@link ITagNodeCondition} checker instances
     * and invokes {@link ITagNodeCondition#satisfy(TagNode)} on each.
     * If and only if every call on {@link ITagNodeCondition#satisfy(TagNode)} returns true,
     * then the {@link #satisfy(TagNode)} method of this class returns true.
     * Otherwise, it will return false.
     * </p>
     * <p>
     * Also, if there is no {@link ITagNodeCondition} checker instances, then it returns false.
     * </p>
     */
    public boolean satisfy(TagNode tagNode)
    {
        if (conditionsList == null || conditionsList.isEmpty())
        {
            return false;
        }

        for (ITagNodeCondition condition : conditionsList)
        {
            if (!condition.satisfy(tagNode))
            {
                return false;
            }
        }

        return true;
    }
}
