/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent.proxy;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.TestCase;

import org.apache.portals.applications.webcontent.proxy.impl.DefaultHttpReverseProxyPathMapperImpl;
import org.apache.portals.applications.webcontent.proxy.impl.DefaultHttpReverseProxyPathMapperProviderImpl;
import org.apache.portals.applications.webcontent.rewriter.RewriterController;
import org.apache.portals.applications.webcontent.rewriter.rules.Ruleset;

/**
 * TestProxyPassMappings
 * 
 * @version $Id: TestProxyPathMappings.java 822629 2009-10-07 09:19:36Z woonsan $
 */
public class TestProxyPathMappings extends TestCase
{
    private DefaultHttpReverseProxyPathMapperProviderImpl pathMapperProvider;
    
    @Override
    public void setUp()
    {
        Map<HttpReverseProxyPathMapper, RewriterController> rewriterControllerMap = new HashMap<HttpReverseProxyPathMapper, RewriterController>();
        Map<HttpReverseProxyPathMapper, Ruleset> rewriterRulesetMap = new HashMap<HttpReverseProxyPathMapper, Ruleset>();
        List<HttpReverseProxyPathMapper> proxyPathMappers = new ArrayList<HttpReverseProxyPathMapper>();
        proxyPathMappers.add(new DefaultHttpReverseProxyPathMapperImpl("localhost", "/localhost/", "http://www.localhost.com/"));
        proxyPathMappers.add(new DefaultHttpReverseProxyPathMapperImpl("localhost1", "/localhost/path1/", "http://www.localhost.com/path1/"));
        proxyPathMappers.add(new DefaultHttpReverseProxyPathMapperImpl("localhost2", "/localhost/path1/path2/", "http://www.localhost.com/path1/path2/"));
        proxyPathMappers.add(new DefaultHttpReverseProxyPathMapperImpl("apache0", "/apache/", "http://apache.org/"));
        proxyPathMappers.add(new DefaultHttpReverseProxyPathMapperImpl("apache", "/*_apache/", "http://$1.apache.org/"));
        proxyPathMappers.add(new DefaultHttpReverseProxyPathMapperImpl("secure_apache", "/secure/*_apache/", "https://$1.apache.org/"));
        proxyPathMappers.add(new DefaultHttpReverseProxyPathMapperImpl("google", "/*.google.*/", "http://$1.google.$2/"));
        pathMapperProvider = new DefaultHttpReverseProxyPathMapperProviderImpl(proxyPathMappers, rewriterControllerMap, rewriterRulesetMap);
        pathMapperProvider.setMaxMatchingPathPartCount(3);
    }
    
    public void testProxyPathMappings() throws Exception
    {
        HttpReverseProxyPathMapper mapper = null;

        mapper = pathMapperProvider.findMapper("/localhost/index.html");
        assertEquals("localhost", mapper.getName());
        mapper = pathMapperProvider.findMapperByRemoteURL("http://www.localhost.com/index.html");
        assertEquals("localhost", mapper.getName());
        
        mapper = pathMapperProvider.findMapper("/www_apache/index.html");
        assertEquals("apache:/www_apache/", mapper.getName());
        mapper = pathMapperProvider.findMapperByRemoteURL("http://www.apache.org/index.html");
        assertEquals("apache:/www_apache/", mapper.getName());
        assertEquals("http://www.apache.org/index.html", mapper.getRemoteURL("/www_apache/index.html"));
        assertEquals("/www_apache/index.html", mapper.getLocalPath("http://www.apache.org/index.html"));
        
        mapper = pathMapperProvider.findMapper("/apache/index.html");
        assertEquals("apache0", mapper.getName());
        assertEquals("http://apache.org/index.html", mapper.getRemoteURL("/apache/index.html"));
        
        mapper = pathMapperProvider.findMapper("/portals_apache/index.html");
        assertEquals("apache:/portals_apache/", mapper.getName());
        mapper = pathMapperProvider.findMapperByRemoteURL("http://portals.apache.org/index.html");
        assertEquals("apache:/portals_apache/", mapper.getName());
        assertEquals("http://portals.apache.org/index.html", mapper.getRemoteURL("/portals_apache/index.html"));
        assertEquals("/portals_apache/index.html", mapper.getLocalPath("http://portals.apache.org/index.html"));

        mapper = pathMapperProvider.findMapper("/secure/www_apache/index.html");
        assertEquals("secure_apache:/secure/www_apache/", mapper.getName());
        mapper = pathMapperProvider.findMapperByRemoteURL("https://blogs.apache.org/index.html");
        assertEquals("secure_apache:/secure/blogs_apache/", mapper.getName());
        assertEquals("https://blogs.apache.org/index.html", mapper.getRemoteURL("/secure/blogs_apache/index.html"));
        assertEquals("/secure/blogs_apache/index.html", mapper.getLocalPath("https://blogs.apache.org/index.html"));
    }
    
    public void testProxyURLMappings() throws Exception
    {
        HttpReverseProxyPathMapper mapper = pathMapperProvider.findMapperByRemoteURL("http://www.apache.org/index.html");
        assertEquals("/www_apache/", mapper.getLocalBasePath());
        
        mapper = pathMapperProvider.findMapperByRemoteURL("http://portals.apache.org/index.html");
        assertEquals("/portals_apache/", mapper.getLocalBasePath());
        
        mapper = pathMapperProvider.findMapperByRemoteURL("https://www.apache.org/index.html");
        assertEquals("/secure/www_apache/", mapper.getLocalBasePath());
        
        mapper = pathMapperProvider.findMapperByRemoteURL("https://blogs.apache.org/index.html");
        assertEquals("/secure/blogs_apache/", mapper.getLocalBasePath());
        
        mapper = pathMapperProvider.findMapperByRemoteURL("http://www.localhost.com/path1/path2/index.html");
        assertEquals("/localhost/path1/path2/", mapper.getLocalBasePath());
        
        mapper = pathMapperProvider.findMapperByRemoteURL("http://www.localhost.com/path1/index.html");
        assertEquals("/localhost/path1/", mapper.getLocalBasePath());
        
        mapper = pathMapperProvider.findMapperByRemoteURL("http://www.localhost.com/index.html");
        assertEquals("/localhost/", mapper.getLocalBasePath());
        
        mapper = pathMapperProvider.findMapperByRemoteURL("http://www.localhost.com/path2/index.html");
        assertEquals("/localhost/", mapper.getLocalBasePath());
        
        mapper = pathMapperProvider.findMapperByRemoteURL("http://www.localhost.com");
        assertEquals("/localhost/", mapper.getLocalBasePath());
        
        mapper = pathMapperProvider.findMapperByRemoteURL("http://www.google.com/");
        assertEquals("/www.google.com/", mapper.getLocalBasePath());
        
        mapper = pathMapperProvider.findMapperByRemoteURL("http://images.google.com/query?q=1/");
        assertEquals("/images.google.com/", mapper.getLocalBasePath());
    }
    
}
