/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.logging.tomcat;

import java.io.File;

import org.apache.catalina.Lifecycle;
import org.apache.catalina.LifecycleEvent;
import org.apache.catalina.LifecycleListener;

/**
 * Tomcat Catalina server Listener implementation used to initialize
 * logging system properties before individual web applications are
 * initialized and started.
 * 
 * Set the logDir attribute for the Listener tag that initializes this
 * listener in ${CATALINA_BASE}/conf/server.xml or define the
 * org.apache.portals.logdir system property using CATALINA_OPTS on startup:
 * 
 * > export CATALINA_OPTS=-Dorg.apache.portals.logdir=/var/log/portal
 * 
 * @author <a href="mailto:rwatler@apache.org">Randy Watler</a>
 * @version $Id:$
 */
public class LoggingPropertiesServerListener implements LifecycleListener
{
    public static final String CATALINA_BASE_PROPERTY_NAME = "catalina.base";
    public static final String CATALINA_HOME_PROPERTY_NAME = "catalina.home";
    public static final String CATALINA_LOGS_DIRECTORY_NAME = "logs";

    public static final String LOG_DIR_PROPERTY_NAME = "org.apache.portals.logdir";
    
    private String logDir;
    
    /* (non-Javadoc)
     * @see org.apache.catalina.LifecycleListener#lifecycleEvent(org.apache.catalina.LifecycleEvent)
     */
    public void lifecycleEvent(LifecycleEvent event)
    {
        if (Lifecycle.BEFORE_START_EVENT.equals(event.getType()))
        {
            // check for overridden logging directory from server environment
            String logDirOverride = System.getProperty(LOG_DIR_PROPERTY_NAME);
            if (logDirOverride == null)
            {
                // compute default logging directory if required
                if (logDir == null)
                {
                    // locate tomcat catalina base/home
                    String catalinaBase = System.getProperty(CATALINA_BASE_PROPERTY_NAME);
                    if (catalinaBase == null)
                    {
                        catalinaBase = System.getProperty(CATALINA_HOME_PROPERTY_NAME);
                    }
                    if (catalinaBase != null)
                    {
                        File catalinaLogsDirectoryFile = new File(catalinaBase+File.separator+CATALINA_LOGS_DIRECTORY_NAME);
                        if (catalinaLogsDirectoryFile.isDirectory())
                        {
                            logDir = catalinaLogsDirectoryFile.getAbsolutePath();
                        }
                    }
                }
                // set logging directory property if available
                if (logDir != null)
                {
                    // validate logging directory; create if necessary
                    File logDirFile = new File(logDir);
                    if (!logDirFile.isDirectory() && !logDirFile.exists())
                    {
                        logDirFile.mkdirs();
                    }
                    // set system property if logging directory exists
                    if (logDirFile.isDirectory())
                    {
                        System.out.println("Setting "+LOG_DIR_PROPERTY_NAME+" = "+logDir);
                        System.setProperty(LOG_DIR_PROPERTY_NAME, logDir);
                    }
                    else
                    {
                        System.err.println("Unable to set "+LOG_DIR_PROPERTY_NAME+": logDir attribute "+logDir+" does not exist and cannot be created or is not a directory.");
                    }
                }
                else
                {
                    System.err.println("Unable to set "+LOG_DIR_PROPERTY_NAME+": logDir attribute not set and "+CATALINA_BASE_PROPERTY_NAME+" or "+CATALINA_HOME_PROPERTY_NAME+" system properties not available or "+CATALINA_LOGS_DIRECTORY_NAME+" directory missing.");
                }
            }
            else
            {
                logDir = logDirOverride;
                System.out.println("Overridden "+LOG_DIR_PROPERTY_NAME+" = "+logDir);                
            }
        }
    }

    /**
     * Get logging directory attribute.
     * 
     * @return logging directory
     */
    public String getLogDir()
    {
        return logDir;
    }

    /**
     * Set logging directory attribute.
     * 
     * @param logDir logging directory
     */
    public void setLogDir(String logDir)
    {
        this.logDir = logDir;
    }
}
