/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.orchestra.conversation.spring;

import java.io.Serializable;

import org.aopalliance.aop.Advice;
import org.springframework.aop.Pointcut;
import org.springframework.aop.PointcutAdvisor;

/**
 * Define a trivial Advisor object that always applies to the class passed to it.
 * <p>
 * Spring requires Advisor objects to be returned rather than Advices. Advisors can implement various interfaces to
 * control which classes or methods they get applied to, but here the OrchestraAdvisorBeanPostProcessor only returns an
 * instance of this for classes that it really *should* apply to, and the advices always apply to all methods, so there
 * is really nothing for this Advisor to do.
 * <p>
 * TODO: maybe it would be nice to allow an orchestra scope object to hold Advisors as well as just Advices, so that
 * users can configure specific code to run only for specific methods of orchestra beans.
 * <p>
 * NB: In Spring2.5, this class can simply implement Advisor, and it will be applied to all methods. However in
 * Spring2.0, class DefaultAdvisorChainFactory only accepts PointcutAdvisor or IntroductionAdvisor, and silently ignores
 * Advisor classes that are not of those types. So here for Spring2.x compatibility we need to make this class a
 * PointcutAdvisor with a dummy pointcut that matches all methods...
 */
class SimpleAdvisor implements PointcutAdvisor, Serializable
{
    private Advice advice;

    public SimpleAdvisor(Advice advice)
    {
        this.advice = advice;
    }

    public Advice getAdvice()
    {
        return advice;
    }

    public boolean isPerInstance()
    {
        return false;
    }

    public Pointcut getPointcut()
    {
        return Pointcut.TRUE;
    }
}
