/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.jmap.http;

import com.github.fge.lambdas.Throwing;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.CharMatcher;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpResponseStatus;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.stream.Stream;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.james.jmap.Endpoint;
import org.apache.james.jmap.JMAPRoute;
import org.apache.james.jmap.JMAPRoutes;
import org.apache.james.jmap.draft.api.SimpleTokenFactory;
import org.apache.james.jmap.draft.exceptions.BadRequestException;
import org.apache.james.jmap.draft.exceptions.BlobNotFoundException;
import org.apache.james.jmap.draft.exceptions.InternalErrorException;
import org.apache.james.jmap.draft.methods.BlobManager;
import org.apache.james.jmap.draft.model.AttachmentAccessToken;
import org.apache.james.jmap.draft.model.Blob;
import org.apache.james.jmap.draft.model.BlobId;
import org.apache.james.jmap.draft.utils.DownloadPath;
import org.apache.james.jmap.exceptions.UnauthorizedException;
import org.apache.james.jmap.http.Authenticator;
import org.apache.james.jmap.http.LoggingHelper;
import org.apache.james.mailbox.MailboxSession;
import org.apache.james.mailbox.exception.MailboxException;
import org.apache.james.mailbox.model.ContentType;
import org.apache.james.metrics.api.MetricFactory;
import org.apache.james.mime4j.codec.EncoderUtil;
import org.apache.james.util.ReactorUtils;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;
import reactor.netty.http.server.HttpServerRequest;
import reactor.netty.http.server.HttpServerResponse;

public class DownloadRoutes
implements JMAPRoutes {
    private static final Logger LOGGER = LoggerFactory.getLogger(DownloadRoutes.class);
    static final String BLOB_ID_PATH_PARAM = "blobId";
    private static final String NAME_PATH_PARAM = "name";
    private static final String DOWNLOAD_FROM_ID = String.format("%s/{%s}", "/download", "blobId");
    private static final String DOWNLOAD_FROM_ID_AND_NAME = String.format("%s/{%s}/{%s}", "/download", "blobId", "name");
    private static final int BUFFER_SIZE = 16384;
    private final BlobManager blobManager;
    private final SimpleTokenFactory simpleTokenFactory;
    private final MetricFactory metricFactory;
    private final Authenticator authenticator;

    @Inject
    @VisibleForTesting
    DownloadRoutes(BlobManager blobManager, SimpleTokenFactory simpleTokenFactory, MetricFactory metricFactory, @Named(value="DRAFT") Authenticator authenticator) {
        this.blobManager = blobManager;
        this.simpleTokenFactory = simpleTokenFactory;
        this.metricFactory = metricFactory;
        this.authenticator = authenticator;
    }

    public Stream<JMAPRoute> routes() {
        return Stream.of(JMAPRoute.builder().endpoint(new Endpoint(HttpMethod.POST, DOWNLOAD_FROM_ID)).action(this::postFromId).corsHeaders(), JMAPRoute.builder().endpoint(new Endpoint(HttpMethod.GET, DOWNLOAD_FROM_ID)).action(this::getFromId).corsHeaders(), JMAPRoute.builder().endpoint(new Endpoint(HttpMethod.POST, DOWNLOAD_FROM_ID_AND_NAME)).action(this::postFromIdAndName).corsHeaders(), JMAPRoute.builder().endpoint(new Endpoint(HttpMethod.GET, DOWNLOAD_FROM_ID_AND_NAME)).action(this::getFromIdAndName).corsHeaders(), JMAPRoute.builder().endpoint(new Endpoint(HttpMethod.OPTIONS, DOWNLOAD_FROM_ID)).action(CORS_CONTROL).noCorsHeaders(), JMAPRoute.builder().endpoint(new Endpoint(HttpMethod.OPTIONS, DOWNLOAD_FROM_ID_AND_NAME)).action(CORS_CONTROL).noCorsHeaders());
    }

    private Mono<Void> postFromId(HttpServerRequest request, HttpServerResponse response) {
        String blobId = request.param((CharSequence)BLOB_ID_PATH_PARAM);
        DownloadPath downloadPath = DownloadPath.ofBlobId(blobId);
        return this.post(request, response, downloadPath);
    }

    private Mono<Void> postFromIdAndName(HttpServerRequest request, HttpServerResponse response) {
        String blobId = request.param((CharSequence)BLOB_ID_PATH_PARAM);
        String name = request.param((CharSequence)NAME_PATH_PARAM);
        DownloadPath downloadPath = DownloadPath.of(blobId, name);
        return this.post(request, response, downloadPath);
    }

    private Mono<Void> post(HttpServerRequest request, HttpServerResponse response, DownloadPath downloadPath) {
        return this.authenticator.authenticate(request).flatMap(session -> Mono.from((Publisher)this.metricFactory.decoratePublisherWithTimerMetric("JMAP-download-post", this.respondAttachmentAccessToken((MailboxSession)session, downloadPath, response))).subscriberContext(LoggingHelper.jmapAuthContext(session))).onErrorResume(UnauthorizedException.class, e -> this.handleAuthenticationFailure(response, LOGGER, (Throwable)e)).doOnEach(ReactorUtils.logOnError(e -> LOGGER.error("Unexpected error", e))).onErrorResume(e -> this.handleInternalError(response, LOGGER, (Throwable)e)).subscriberContext(LoggingHelper.jmapContext(request)).subscriberContext(LoggingHelper.jmapAction("download-post")).subscribeOn(Schedulers.elastic());
    }

    private Mono<Void> getFromId(HttpServerRequest request, HttpServerResponse response) {
        String blobId = request.param((CharSequence)BLOB_ID_PATH_PARAM);
        DownloadPath downloadPath = DownloadPath.ofBlobId(blobId);
        return this.get(request, response, downloadPath);
    }

    private Mono<Void> getFromIdAndName(HttpServerRequest request, HttpServerResponse response) {
        String blobId = request.param((CharSequence)BLOB_ID_PATH_PARAM);
        try {
            String name = URLDecoder.decode(request.param((CharSequence)NAME_PATH_PARAM), StandardCharsets.UTF_8.toString());
            DownloadPath downloadPath = DownloadPath.of(blobId, name);
            return this.get(request, response, downloadPath);
        }
        catch (UnsupportedEncodingException e) {
            throw new BadRequestException("Wrong url encoding", e);
        }
    }

    private Mono<Void> get(HttpServerRequest request, HttpServerResponse response, DownloadPath downloadPath) {
        return this.authenticator.authenticate(request).flatMap(session -> Mono.from((Publisher)this.metricFactory.decoratePublisherWithTimerMetric("JMAP-download-get", this.download((MailboxSession)session, downloadPath, response))).subscriberContext(LoggingHelper.jmapAuthContext(session))).onErrorResume(UnauthorizedException.class, e -> this.handleAuthenticationFailure(response, LOGGER, (Throwable)e)).doOnEach(ReactorUtils.logOnError(e -> LOGGER.error("Unexpected error", e))).onErrorResume(IllegalArgumentException.class, e -> this.handleBadRequest(response, LOGGER, (Throwable)e)).onErrorResume(e -> this.handleInternalError(response, LOGGER, (Throwable)e)).subscriberContext(LoggingHelper.jmapContext(request)).subscriberContext(LoggingHelper.jmapAction("download-get")).subscribeOn(Schedulers.elastic());
    }

    private Mono<Void> respondAttachmentAccessToken(MailboxSession mailboxSession, DownloadPath downloadPath, HttpServerResponse resp) {
        String blobId = downloadPath.getBlobId();
        try {
            if (!this.attachmentExists(mailboxSession, blobId)) {
                return resp.status(HttpResponseStatus.NOT_FOUND).send();
            }
            AttachmentAccessToken attachmentAccessToken = this.simpleTokenFactory.generateAttachmentAccessToken(mailboxSession.getUser().asString(), blobId);
            byte[] bytes = attachmentAccessToken.serialize().getBytes(StandardCharsets.UTF_8);
            return resp.header((CharSequence)HttpHeaderNames.CONTENT_TYPE, (CharSequence)"text/plain").status(HttpResponseStatus.OK).header((CharSequence)HttpHeaderNames.CONTENT_LENGTH, (CharSequence)Integer.toString(bytes.length)).sendByteArray((Publisher)Mono.just((Object)bytes)).then();
        }
        catch (MailboxException e) {
            throw new InternalErrorException("Error while asking attachment access token", e);
        }
    }

    private boolean attachmentExists(MailboxSession mailboxSession, String blobId) throws MailboxException {
        try {
            this.blobManager.retrieve(BlobId.of(blobId), mailboxSession);
            return true;
        }
        catch (BlobNotFoundException e) {
            return false;
        }
    }

    @VisibleForTesting
    Mono<Void> download(MailboxSession mailboxSession, DownloadPath downloadPath, HttpServerResponse response) {
        String blobId = downloadPath.getBlobId();
        try {
            Blob blob = this.blobManager.retrieve(BlobId.of(blobId), mailboxSession);
            return Mono.usingWhen((Publisher)Mono.fromCallable(blob::getStream), stream -> this.downloadBlob(downloadPath.getName(), response, blob.getSize(), blob.getContentType(), (InputStream)stream), stream -> Mono.fromRunnable((Runnable)Throwing.runnable(stream::close).sneakyThrow()));
        }
        catch (BlobNotFoundException e) {
            LOGGER.info("Attachment '{}' not found", (Object)blobId, (Object)e);
            return response.status(HttpResponseStatus.NOT_FOUND).send();
        }
        catch (MailboxException e) {
            throw new InternalErrorException("Error while downloading", e);
        }
    }

    private Mono<Void> downloadBlob(Optional<String> optionalName, HttpServerResponse response, long blobSize, ContentType blobContentType, InputStream stream) {
        return this.addContentDispositionHeader(optionalName, response).header((CharSequence)"Content-Length", (CharSequence)String.valueOf(blobSize)).header((CharSequence)HttpHeaderNames.CONTENT_TYPE, (CharSequence)blobContentType.asString()).status(HttpResponseStatus.OK).send((Publisher)ReactorUtils.toChunks((InputStream)stream, (int)16384).map(Unpooled::wrappedBuffer).subscribeOn(Schedulers.elastic())).then();
    }

    private HttpServerResponse addContentDispositionHeader(Optional<String> optionalName, HttpServerResponse resp) {
        return optionalName.map(name -> this.addContentDispositionHeaderRegardingEncoding((String)name, resp)).orElse(resp);
    }

    private HttpServerResponse addContentDispositionHeaderRegardingEncoding(String name, HttpServerResponse resp) {
        if (CharMatcher.ascii().matchesAllOf((CharSequence)name)) {
            return resp.header((CharSequence)"Content-Disposition", (CharSequence)("attachment; filename=\"" + name + "\""));
        }
        return resp.header((CharSequence)"Content-Disposition", (CharSequence)("attachment; filename*=\"" + EncoderUtil.encodeEncodedWord((String)name, (EncoderUtil.Usage)EncoderUtil.Usage.TEXT_TOKEN) + "\""));
    }
}

