/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.raftlog.segmented;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.metrics.SegmentedRaftLogMetrics;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.raftlog.LogEntryHeader;
import org.apache.ratis.server.raftlog.LogProtoUtils;
import org.apache.ratis.server.raftlog.RaftLogIOException;
import org.apache.ratis.server.raftlog.segmented.LogSegmentStartEnd;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLogFormat;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLogInputStream;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.ratis.thirdparty.com.google.common.cache.CacheLoader;
import org.apache.ratis.thirdparty.com.google.protobuf.CodedOutputStream;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.SizeInBytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class LogSegment {
    static final Logger LOG = LoggerFactory.getLogger(LogSegment.class);
    private volatile boolean isOpen;
    private long totalFileSize = SegmentedRaftLogFormat.getHeaderLength();
    private AtomicLong totalCacheSize = new AtomicLong(0L);
    private final long startIndex;
    private volatile long endIndex;
    private final RaftStorage storage;
    private final SizeInBytes maxOpSize;
    private final LogEntryLoader cacheLoader;
    private final AtomicInteger loadingTimes = new AtomicInteger();
    private final List<LogRecord> records = new ArrayList<LogRecord>();
    private final Map<TermIndex, RaftProtos.LogEntryProto> entryCache = new ConcurrentHashMap<TermIndex, RaftProtos.LogEntryProto>();
    static final Comparator<Object> SEGMENT_TO_INDEX_COMPARATOR = (o1, o2) -> {
        if (o1 instanceof LogSegment && o2 instanceof Long) {
            return ((LogSegment)o1).compareTo((Long)o2);
        }
        if (o1 instanceof Long && o2 instanceof LogSegment) {
            return Integer.compare(0, ((LogSegment)o2).compareTo((Long)o1));
        }
        throw new IllegalStateException("Unexpected objects to compare(" + o1 + "," + o2 + ")");
    };

    static long getEntrySize(RaftProtos.LogEntryProto entry, Op op) {
        RaftProtos.LogEntryProto e = entry;
        if (op == Op.CHECK_SEGMENT_FILE_FULL) {
            e = LogProtoUtils.removeStateMachineData(entry);
        } else if (op == Op.LOAD_SEGMENT_FILE || op == Op.WRITE_CACHE_WITH_STATE_MACHINE_CACHE) {
            Preconditions.assertTrue((entry == LogProtoUtils.removeStateMachineData(entry) ? 1 : 0) != 0, () -> "Unexpected LogEntryProto with StateMachine data: op=" + (Object)((Object)op) + ", entry=" + entry);
        } else {
            Preconditions.assertTrue((op == Op.WRITE_CACHE_WITHOUT_STATE_MACHINE_CACHE || op == Op.REMOVE_CACHE ? 1 : 0) != 0, () -> "Unexpected op " + (Object)((Object)op) + ", entry=" + entry);
        }
        int serialized = e.getSerializedSize();
        return (long)(serialized + CodedOutputStream.computeUInt32SizeNoTag((int)serialized)) + 4L;
    }

    static LogSegment newOpenSegment(RaftStorage storage, long start, SizeInBytes maxOpSize, SegmentedRaftLogMetrics raftLogMetrics) {
        Preconditions.assertTrue((start >= 0L ? 1 : 0) != 0);
        return new LogSegment(storage, true, start, start - 1L, maxOpSize, raftLogMetrics);
    }

    @VisibleForTesting
    static LogSegment newCloseSegment(RaftStorage storage, long start, long end, SizeInBytes maxOpSize, SegmentedRaftLogMetrics raftLogMetrics) {
        Preconditions.assertTrue((start >= 0L && end >= start ? 1 : 0) != 0);
        return new LogSegment(storage, false, start, end, maxOpSize, raftLogMetrics);
    }

    static LogSegment newLogSegment(RaftStorage storage, LogSegmentStartEnd startEnd, SizeInBytes maxOpSize, SegmentedRaftLogMetrics metrics) {
        return startEnd.isOpen() ? LogSegment.newOpenSegment(storage, startEnd.getStartIndex(), maxOpSize, metrics) : LogSegment.newCloseSegment(storage, startEnd.getStartIndex(), startEnd.getEndIndex(), maxOpSize, metrics);
    }

    public static int readSegmentFile(File file, LogSegmentStartEnd startEnd, SizeInBytes maxOpSize, RaftServerConfigKeys.Log.CorruptionPolicy corruptionPolicy, SegmentedRaftLogMetrics raftLogMetrics, Consumer<RaftProtos.LogEntryProto> entryConsumer) throws IOException {
        int count = 0;
        try (SegmentedRaftLogInputStream in = new SegmentedRaftLogInputStream(file, startEnd, maxOpSize, raftLogMetrics);){
            RaftProtos.LogEntryProto next;
            RaftProtos.LogEntryProto prev = null;
            while ((next = in.nextEntry()) != null) {
                if (prev != null) {
                    Preconditions.assertTrue((next.getIndex() == prev.getIndex() + 1L ? 1 : 0) != 0, (String)"gap between entry %s and entry %s", (Object[])new Object[]{prev, next});
                }
                if (entryConsumer != null) {
                    entryConsumer.accept(next);
                }
                ++count;
                prev = next;
            }
        }
        catch (IOException ioe) {
            switch (corruptionPolicy) {
                case EXCEPTION: {
                    throw ioe;
                }
                case WARN_AND_RETURN: {
                    LOG.warn("Failed to read segment file {} ({}): only {} entries read successfully", new Object[]{file, startEnd, count, ioe});
                    break;
                }
                default: {
                    throw new IllegalStateException("Unexpected enum value: " + corruptionPolicy + ", class=" + RaftServerConfigKeys.Log.CorruptionPolicy.class);
                }
            }
        }
        return count;
    }

    static LogSegment loadSegment(RaftStorage storage, File file, LogSegmentStartEnd startEnd, SizeInBytes maxOpSize, boolean keepEntryInCache, Consumer<RaftProtos.LogEntryProto> logConsumer, SegmentedRaftLogMetrics raftLogMetrics) throws IOException {
        boolean corrupted;
        LogSegment segment = LogSegment.newLogSegment(storage, startEnd, maxOpSize, raftLogMetrics);
        RaftServerConfigKeys.Log.CorruptionPolicy corruptionPolicy = RaftServerConfigKeys.Log.CorruptionPolicy.get((Object)storage, RaftStorage::getLogCorruptionPolicy);
        boolean isOpen = startEnd.isOpen();
        int entryCount = LogSegment.readSegmentFile(file, startEnd, maxOpSize, corruptionPolicy, raftLogMetrics, entry -> {
            segment.append(keepEntryInCache || isOpen, (RaftProtos.LogEntryProto)entry, Op.LOAD_SEGMENT_FILE);
            if (logConsumer != null) {
                logConsumer.accept((RaftProtos.LogEntryProto)entry);
            }
        });
        LOG.info("Successfully read {} entries from segment file {}", (Object)entryCount, (Object)file);
        long start = startEnd.getStartIndex();
        long end = isOpen ? segment.getEndIndex() : startEnd.getEndIndex();
        int expectedEntryCount = Math.toIntExact(end - start + 1L);
        boolean bl = corrupted = entryCount != expectedEntryCount;
        if (corrupted) {
            LOG.warn("Segment file is corrupted: expected to have {} entries but only {} entries read successfully", (Object)expectedEntryCount, (Object)entryCount);
        }
        if (entryCount == 0) {
            Path deleted = FileUtils.deleteFile((File)file);
            LOG.info("Deleted RaftLog segment since entry count is zero: startEnd={}, path={}", (Object)startEnd, (Object)deleted);
            return null;
        }
        if (file.length() > segment.getTotalFileSize()) {
            FileUtils.truncateFile((File)file, (long)segment.getTotalFileSize());
        }
        try {
            segment.assertSegment(start, entryCount, corrupted, end);
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to read segment file " + file, e);
        }
        return segment;
    }

    private void assertSegment(long expectedStart, int expectedEntryCount, boolean corrupted, long expectedEnd) {
        Preconditions.assertSame((long)expectedStart, (long)this.getStartIndex(), (String)"Segment start index");
        Preconditions.assertSame((int)expectedEntryCount, (int)this.records.size(), (String)"Number of records");
        long expectedLastIndex = expectedStart + (long)expectedEntryCount - 1L;
        Preconditions.assertSame((long)expectedLastIndex, (long)this.getEndIndex(), (String)"Segment end index");
        LogRecord last = this.getLastRecord();
        if (last != null) {
            Preconditions.assertSame((long)expectedLastIndex, (long)last.getTermIndex().getIndex(), (String)"Index at the last record");
            Preconditions.assertSame((long)expectedStart, (long)this.records.get(0).getTermIndex().getIndex(), (String)"Index at the first record");
        }
        if (!corrupted) {
            Preconditions.assertSame((long)expectedEnd, (long)expectedLastIndex, (String)"End/last Index");
        }
    }

    File getFile() {
        return LogSegmentStartEnd.valueOf(this.startIndex, this.endIndex, this.isOpen).getFile(this.storage);
    }

    private LogSegment(RaftStorage storage, boolean isOpen, long start, long end, SizeInBytes maxOpSize, SegmentedRaftLogMetrics raftLogMetrics) {
        this.storage = storage;
        this.isOpen = isOpen;
        this.startIndex = start;
        this.endIndex = end;
        this.maxOpSize = maxOpSize;
        this.cacheLoader = new LogEntryLoader(raftLogMetrics);
    }

    long getStartIndex() {
        return this.startIndex;
    }

    long getEndIndex() {
        return this.endIndex;
    }

    boolean isOpen() {
        return this.isOpen;
    }

    int numOfEntries() {
        return Math.toIntExact(this.endIndex - this.startIndex + 1L);
    }

    RaftServerConfigKeys.Log.CorruptionPolicy getLogCorruptionPolicy() {
        return RaftServerConfigKeys.Log.CorruptionPolicy.get((Object)this.storage, RaftStorage::getLogCorruptionPolicy);
    }

    void appendToOpenSegment(RaftProtos.LogEntryProto entry, Op op) {
        Preconditions.assertTrue((boolean)this.isOpen(), (String)"The log segment %s is not open for append", (Object[])new Object[]{this});
        this.append(true, entry, op);
    }

    private void append(boolean keepEntryInCache, RaftProtos.LogEntryProto entry, Op op) {
        LogRecord currentLast;
        Objects.requireNonNull(entry, "entry == null");
        if (this.records.isEmpty()) {
            Preconditions.assertTrue((entry.getIndex() == this.startIndex ? 1 : 0) != 0, (String)"gap between start index %s and first entry to append %s", (Object[])new Object[]{this.startIndex, entry.getIndex()});
        }
        if ((currentLast = this.getLastRecord()) != null) {
            Preconditions.assertTrue((entry.getIndex() == currentLast.getTermIndex().getIndex() + 1L ? 1 : 0) != 0, (String)"gap between entries %s and %s", (Object[])new Object[]{entry.getIndex(), currentLast.getTermIndex().getIndex()});
        }
        LogRecord record = new LogRecord(this.totalFileSize, entry);
        if (keepEntryInCache) {
            this.putEntryCache(record.getTermIndex(), entry, op);
        }
        this.records.add(record);
        this.totalFileSize += LogSegment.getEntrySize(entry, op);
        this.endIndex = entry.getIndex();
    }

    RaftProtos.LogEntryProto getEntryFromCache(TermIndex ti) {
        return this.entryCache.get(ti);
    }

    synchronized RaftProtos.LogEntryProto loadCache(LogRecord record) throws RaftLogIOException {
        RaftProtos.LogEntryProto entry = this.entryCache.get(record.getTermIndex());
        if (entry != null) {
            return entry;
        }
        try {
            return this.cacheLoader.load(record);
        }
        catch (RaftLogIOException e) {
            throw e;
        }
        catch (Exception e) {
            throw new RaftLogIOException("Failed to loadCache for log entry " + record, (Throwable)e);
        }
    }

    LogRecord getLogRecord(long index) {
        if (index >= this.startIndex && index <= this.endIndex) {
            return this.records.get(Math.toIntExact(index - this.startIndex));
        }
        return null;
    }

    private LogRecord getLastRecord() {
        return this.records.isEmpty() ? null : this.records.get(this.records.size() - 1);
    }

    TermIndex getLastTermIndex() {
        LogRecord last = this.getLastRecord();
        return last == null ? null : last.getTermIndex();
    }

    long getTotalFileSize() {
        return this.totalFileSize;
    }

    long getTotalCacheSize() {
        return this.totalCacheSize.get();
    }

    synchronized void truncate(long fromIndex) {
        Preconditions.assertTrue((fromIndex >= this.startIndex && fromIndex <= this.endIndex ? 1 : 0) != 0);
        for (long index = this.endIndex; index >= fromIndex; --index) {
            LogRecord removed = this.records.remove(Math.toIntExact(index - this.startIndex));
            this.removeEntryCache(removed.getTermIndex(), Op.REMOVE_CACHE);
            this.totalFileSize = removed.offset;
        }
        this.isOpen = false;
        this.endIndex = fromIndex - 1L;
    }

    void close() {
        Preconditions.assertTrue((boolean)this.isOpen());
        this.isOpen = false;
    }

    public String toString() {
        return this.isOpen() ? "log_inprogress_" + this.startIndex : "log-" + this.startIndex + "_" + this.endIndex;
    }

    private int compareTo(Long l) {
        return l >= this.getStartIndex() && l <= this.getEndIndex() ? 0 : (this.getEndIndex() < l ? -1 : 1);
    }

    synchronized void clear() {
        this.records.clear();
        this.evictCache();
        this.endIndex = this.startIndex - 1L;
    }

    int getLoadingTimes() {
        return this.loadingTimes.get();
    }

    void evictCache() {
        this.entryCache.clear();
        this.totalCacheSize.set(0L);
    }

    void putEntryCache(TermIndex key, RaftProtos.LogEntryProto value, Op op) {
        RaftProtos.LogEntryProto previous = this.entryCache.put(key, value);
        long previousSize = 0L;
        if (previous != null) {
            previousSize = LogSegment.getEntrySize(value, Op.REMOVE_CACHE);
        }
        this.totalCacheSize.getAndAdd(LogSegment.getEntrySize(value, op) - previousSize);
    }

    void removeEntryCache(TermIndex key, Op op) {
        RaftProtos.LogEntryProto value = this.entryCache.remove(key);
        if (value != null) {
            this.totalCacheSize.getAndAdd(-LogSegment.getEntrySize(value, op));
        }
    }

    boolean hasCache() {
        return this.isOpen || !this.entryCache.isEmpty();
    }

    boolean containsIndex(long index) {
        return this.startIndex <= index && this.endIndex >= index;
    }

    boolean hasEntries() {
        return this.numOfEntries() > 0;
    }

    static enum Op {
        LOAD_SEGMENT_FILE,
        REMOVE_CACHE,
        CHECK_SEGMENT_FILE_FULL,
        WRITE_CACHE_WITH_STATE_MACHINE_CACHE,
        WRITE_CACHE_WITHOUT_STATE_MACHINE_CACHE;

    }

    static class LogRecord {
        private final long offset;
        private final LogEntryHeader logEntryHeader;

        LogRecord(long offset, RaftProtos.LogEntryProto entry) {
            this.offset = offset;
            this.logEntryHeader = LogEntryHeader.valueOf((RaftProtos.LogEntryProto)entry);
        }

        LogEntryHeader getLogEntryHeader() {
            return this.logEntryHeader;
        }

        TermIndex getTermIndex() {
            return this.getLogEntryHeader().getTermIndex();
        }

        long getOffset() {
            return this.offset;
        }
    }

    class LogEntryLoader
    extends CacheLoader<LogRecord, RaftProtos.LogEntryProto> {
        private final SegmentedRaftLogMetrics raftLogMetrics;

        LogEntryLoader(SegmentedRaftLogMetrics raftLogMetrics) {
            this.raftLogMetrics = raftLogMetrics;
        }

        public RaftProtos.LogEntryProto load(LogRecord key) throws IOException {
            File file = LogSegment.this.getFile();
            AtomicReference toReturn = new AtomicReference();
            LogSegmentStartEnd startEnd = LogSegmentStartEnd.valueOf(LogSegment.this.startIndex, LogSegment.this.endIndex, LogSegment.this.isOpen);
            LogSegment.readSegmentFile(file, startEnd, LogSegment.this.maxOpSize, LogSegment.this.getLogCorruptionPolicy(), this.raftLogMetrics, entry -> {
                TermIndex ti = TermIndex.valueOf((RaftProtos.LogEntryProto)entry);
                LogSegment.this.putEntryCache(ti, (RaftProtos.LogEntryProto)entry, Op.LOAD_SEGMENT_FILE);
                if (ti.equals(key.getTermIndex())) {
                    toReturn.set(entry);
                }
            });
            LogSegment.this.loadingTimes.incrementAndGet();
            RaftProtos.LogEntryProto proto = (RaftProtos.LogEntryProto)toReturn.get();
            if (proto == null) {
                throw new RaftLogIOException("Failed to load log entry " + key);
            }
            return proto;
        }
    }
}

