"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SwfTextEditorController = exports.SwfTextEditorOperation = void 0;
var api_1 = require("@kie-tools-core/editor/dist/api");
var operating_system_1 = require("@kie-tools-core/operating-system");
var api_2 = require("@kie-tools/serverless-workflow-language-service/dist/api");
var editor_1 = require("@kie-tools/serverless-workflow-language-service/dist/editor");
var monaco_editor_1 = require("monaco-editor");
var json_1 = require("./augmentation/language/json");
var yaml_1 = require("./augmentation/language/yaml");
(0, json_1.initJsonSchemaDiagnostics)();
(0, yaml_1.initYamlSchemaDiagnostics)();
var SwfTextEditorOperation;
(function (SwfTextEditorOperation) {
    SwfTextEditorOperation[SwfTextEditorOperation["UNDO"] = 0] = "UNDO";
    SwfTextEditorOperation[SwfTextEditorOperation["REDO"] = 1] = "REDO";
    SwfTextEditorOperation[SwfTextEditorOperation["EDIT"] = 2] = "EDIT";
})(SwfTextEditorOperation = exports.SwfTextEditorOperation || (exports.SwfTextEditorOperation = {}));
var SwfTextEditorController = (function () {
    function SwfTextEditorController(content, onContentChange, language, operatingSystem, isReadOnly, setValidationErrors, onSelectionChanged) {
        var _this = this;
        this.onContentChange = onContentChange;
        this.language = language;
        this.operatingSystem = operatingSystem;
        this.isReadOnly = isReadOnly;
        this.setValidationErrors = setValidationErrors;
        this.onSelectionChanged = onSelectionChanged;
        this.getValidationMarkers = function () {
            return monaco_editor_1.editor.getModelMarkers({});
        };
        this.model = monaco_editor_1.editor.createModel(content, this.language);
        this.model.onDidChangeContent(function (event) {
            var _a;
            if (!event.isUndoing && !event.isRedoing) {
                (_a = _this.editor) === null || _a === void 0 ? void 0 : _a.pushUndoStop();
                onContentChange({ content: _this.model.getValue(), operation: SwfTextEditorOperation.EDIT });
            }
        });
        monaco_editor_1.editor.onDidChangeMarkers(function () {
            _this.setValidationErrors(_this.getValidationMarkers());
        });
        monaco_editor_1.editor.onDidCreateEditor(function (codeEditor) {
            codeEditor.onDidChangeCursorSelection(function (event) { return _this.handleDidChangeCursorSelection(event); });
        });
    }
    SwfTextEditorController.prototype.redo = function () {
        var _a, _b;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.focus();
        (_b = this.editor) === null || _b === void 0 ? void 0 : _b.trigger("editor", "redo", null);
    };
    SwfTextEditorController.prototype.undo = function () {
        var _a, _b;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.focus();
        (_b = this.editor) === null || _b === void 0 ? void 0 : _b.trigger("editor", "undo", null);
    };
    SwfTextEditorController.prototype.setTheme = function (theme) {
        monaco_editor_1.editor.setTheme(this.getMonacoThemeByEditorTheme(theme));
    };
    SwfTextEditorController.prototype.show = function (container, theme) {
        var _this = this;
        if (!container) {
            throw new Error("We need a container to show the editor!");
        }
        this.editor = monaco_editor_1.editor.create(container, {
            model: this.model,
            language: this.language,
            scrollBeyondLastLine: false,
            automaticLayout: true,
            fontSize: 12,
            theme: this.getMonacoThemeByEditorTheme(theme),
            readOnly: this.isReadOnly,
        });
        this.editor.addCommand(monaco_editor_1.KeyMod.CtrlCmd | monaco_editor_1.KeyCode.KeyZ, function () {
            _this.onContentChange({ content: _this.model.getValue(), operation: SwfTextEditorOperation.UNDO });
        });
        this.editor.addCommand(monaco_editor_1.KeyMod.CtrlCmd | monaco_editor_1.KeyMod.Shift | monaco_editor_1.KeyCode.KeyZ, function () {
            _this.onContentChange({ content: _this.model.getValue(), operation: SwfTextEditorOperation.REDO });
        });
        if (this.operatingSystem !== operating_system_1.OperatingSystem.MACOS) {
            this.editor.addCommand(monaco_editor_1.KeyMod.CtrlCmd | monaco_editor_1.KeyCode.KeyY, function () {
                _this.onContentChange({ content: _this.model.getValue(), operation: SwfTextEditorOperation.REDO });
            });
        }
        return this.editor;
    };
    SwfTextEditorController.prototype.getContent = function () {
        var _a, _b;
        return ((_b = (_a = this.editor) === null || _a === void 0 ? void 0 : _a.getModel()) === null || _b === void 0 ? void 0 : _b.getValue()) || "";
    };
    SwfTextEditorController.prototype.forceRedraw = function () {
        var _a;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.render(true);
    };
    SwfTextEditorController.prototype.moveCursorToNode = function (nodeName) {
        var _a, _b;
        if (!this.editor || !nodeName) {
            return;
        }
        var getStateNameOffsetArgs = {
            content: this.getContent(),
            stateName: nodeName,
        };
        var targetOffset = this.language === api_2.FileLanguage.JSON
            ? (0, editor_1.getJsonStateNameOffset)(getStateNameOffsetArgs)
            : (0, editor_1.getYamlStateNameOffset)(getStateNameOffsetArgs);
        if (!targetOffset) {
            return;
        }
        var targetPosition = (_b = (_a = this.editor) === null || _a === void 0 ? void 0 : _a.getModel()) === null || _b === void 0 ? void 0 : _b.getPositionAt(targetOffset);
        if (!targetPosition) {
            return;
        }
        this.moveCursorToPosition(targetPosition);
    };
    SwfTextEditorController.prototype.moveCursorToPosition = function (position) {
        var _a, _b, _c;
        if (!this.editor) {
            return;
        }
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.revealLineInCenter(position.lineNumber);
        (_b = this.editor) === null || _b === void 0 ? void 0 : _b.setPosition(position);
        (_c = this.editor) === null || _c === void 0 ? void 0 : _c.focus();
    };
    SwfTextEditorController.prototype.handleDidChangeCursorSelection = function (event) {
        var _a, _b;
        var selection = event.selection;
        if (event.reason !== monaco_editor_1.editor.CursorChangeReason.Explicit ||
            !monaco_editor_1.Position.equals(selection.getStartPosition(), selection.getEndPosition())) {
            return;
        }
        var offset = (_b = (_a = this.editor) === null || _a === void 0 ? void 0 : _a.getModel()) === null || _b === void 0 ? void 0 : _b.getOffsetAt(selection.getStartPosition());
        if (!offset) {
            return;
        }
        var getStateNameFromOffsetArgs = { content: this.getContent(), offset: offset };
        var nodeName = this.language === api_2.FileLanguage.JSON
            ? (0, editor_1.getJsonStateNameFromOffset)(getStateNameFromOffsetArgs)
            : (0, editor_1.getYamlStateNameFromOffset)(getStateNameFromOffsetArgs);
        if (!nodeName) {
            return;
        }
        this.onSelectionChanged(nodeName);
    };
    SwfTextEditorController.prototype.getMonacoThemeByEditorTheme = function (theme) {
        switch (theme) {
            case api_1.EditorTheme.DARK:
                return "vs-dark";
            case api_1.EditorTheme.HIGH_CONTRAST:
                return "hc-black";
            default:
                return "vs";
        }
    };
    SwfTextEditorController.prototype.dispose = function () {
        var _a;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.dispose();
    };
    return SwfTextEditorController;
}());
exports.SwfTextEditorController = SwfTextEditorController;
//# sourceMappingURL=SwfTextEditorController.js.map