/*
 * RPC server.cc
 * Server RPC demo for wxWindows class library
 * Usage:
 *   server [<port number>]
 *
 * When the server is running, fire up the client. The client's RPC menu item
 * can be used to send an arbitrary call to the server.
 * Also, selecting the Advise menu item on the server tests out sending an advise
 * message to the client, which then responds with an RPC back to the server, popping
 * up a message box. This scenario could be common where a server is acting as both
 * the provider of information AND a user interface, so that menu items on the server
 * (possibly put there by the client) tell the client what info. to get from the
 * server. An example should make this a bit clearer.
 *
 * Say the server, e.g. a diagram editor, only implements a
 * general set of functions, such as move shape, change shape colour.
 * We can make it *look* like the server is a special-purpose program by simply
 * adding to the server the capability to have user-definable menus, with menu events
 * sent to an interested client. The client then implements the specific functions
 * associated with menu selections, with visual feedback by calling back to the server
 * interface.
 */

#include <afxwin.h>
#include "wx.h"
#include "read.h"
#include "prorpc.h"
#include "server.h"
#include <stdarg.h>

MyFrame *frame = NULL;

// This statement initialises the whole application
MyApp myApp;

char ipc_buffer[4000];
myRPCConnection *the_connection = NULL;

rpcCallTable call_table;

// The `main program' equivalent, creating the windows and returning the
// main frame
wxFrame *MyApp::OnInit(void)
{
  // Create the main frame window
  frame = new MyFrame(NULL, "Server", 0, 0, 400, 300);

  // Give it an icon
  wxIcon *icon = new wxIcon("conn_icn");
  frame->SetIcon(icon);

  // Make a menubar
  wxMenu *file_menu = new wxMenu;

  file_menu->Append(SERVER_ADVISE, "Advise");
  file_menu->Append(SERVER_QUIT, "Quit");

  wxMenuBar *menu_bar = new wxMenuBar;

  menu_bar->Append(file_menu, "File");

  // Associate the menu bar with the frame
  frame->SetMenuBar(menu_bar);

  // Initialize IPC
  wxIPCInitialize();

  char *server_name = "4242";
  if (argc > 1)
    server_name = argv[1];

  // Set up call table
  call_table.AddCall("hello", ServerHello, PrologString, 0);
  call_table.AddCall("test", ServerTest, PrologString, 0);
  call_table.AddCall("popup", ServerPopup, PrologString, 0);

  // Create a new server
  myRPCServer *server = new myRPCServer;
  server->Create(server_name);

  // Essential - return the main frame window
  return frame;
}

// Define my frame constructor
MyFrame::MyFrame(wxFrame *frame, char *title, int x, int y, int w, int h):
  wxFrame(frame, title, x, y, w, h)
{
}

// Intercept menu commands
void MyFrame::OnMenuCommand(int id)
{
  switch (id)
  {
    case SERVER_ADVISE:
    {
      if (the_connection)
      {
        the_connection->Advise("test", "some data to advise the client with");
      }
      break;
    }
    case SERVER_QUIT:
    {
      OnClose();
      delete this;
      break;
    }
  }
}

// Define the behaviour for the frame closing
// - must delete all frames except for the main one.
Bool MyFrame::OnClose(void)
{
  return TRUE;
}

/*
 * IPC stuff
 *
 */

void IPCButton(wxButton& button, wxEvent& event)
{
#ifdef wx_x
  IPCDialogBox *dialog = (IPCDialogBox *)button.GetParent();
  char *message = "A message from the server!";
#endif
}

void IPCButtonQuit(wxButton& button, wxEvent& event)
{
  IPCDialogBox *dialog = (IPCDialogBox *)button.GetParent();
  dialog->connection->Disconnect();
  delete dialog->connection;
  delete dialog;
}

IPCDialogBox::IPCDialogBox(wxFrame *Parent, char *Title, Bool Modal, 
                         int x, int y, int width, int height, myRPCConnection *the_connection):
                    wxDialogBox(Parent, Title, Modal, x, y, width, height)
{
  connection = the_connection;
  wxButton *button2 = new wxButton(this, (wxFunction)IPCButtonQuit, "Quit this connection");
  Fit();
  Show(TRUE);
}

myRPCServer::myRPCServer(void)
{
}

wxConnection *myRPCServer::OnAcceptConnection(char *topic)
{
  if (strcmp(topic, "RPC") == 0)
  {
    myRPCConnection *connection = new myRPCConnection(ipc_buffer, 4000);
    connection->RegisterCallTable(&call_table);
    return connection;
  }
  else
    return NULL;
}

myRPCConnection::myRPCConnection(char *buf, int size):rpcConnection(buf, size)
{
  IPCDialogBox *box = new IPCDialogBox(frame, "Connection", FALSE, 100, 100, 100, 100, this);
  the_connection = this;
}

PrologExpr *ServerHello(wxConnection *connection, char *functor, wxList *args)
{
  return wxMakeCall("return", wxMakeString("hello to you too!"), 0);
}

PrologExpr *ServerTest(wxConnection *connection, char *functor, wxList *args)
{
  return wxMakeCall("return", wxMakeString("Test worked!"), 0);
}

// Popup a window with the given message
PrologExpr *ServerPopup(wxConnection *connection, char *functor, wxList *args)
{
  PrologExpr *arg1 = (PrologExpr *)args->Nth(1)->Data();
  char *s = arg1->StringValue();
  wxMessageBox(s, "Message from client");
  return wxMakeCall("return", wxMakeString("ok"), 0);
}

