/*
 * File:	shapes.h
 * Purpose:	
 * Author:	Julian Smart
 * Created:	1993
 * Updated:	
 * Copyright:	(c) 1993, AIAI, University of Edinburgh
 */

/* sccsid[] = "%W% %G%" */

#ifndef hy_shapesh
#define hy_shapesh

#ifndef DEFAULT_MOUSE_TOLERANCE
#define DEFAULT_MOUSE_TOLERANCE 3
#endif

// Key identifiers
#define KEY_SHIFT 1
#define KEY_CTRL  2

// Arrow styles
#define ARROW_NONE         0
#define ARROW_END          1
#define ARROW_BOTH         2
#define ARROW_MIDDLE       3
#define ARROW_START        4

#define CONTROL_POINT_SIZE       6

// Control point types
#define CONTROL_POINT_VERTICAL   1
#define CONTROL_POINT_HORIZONTAL 2
#define CONTROL_POINT_DIAGONAL   3

// Types of formatting: can be combined in a bit list
#define FORMAT_NONE           0
                                // Left justification
#define FORMAT_CENTRE_HORIZ   1
                                // Centre horizontally
#define FORMAT_CENTRE_VERT    2
                                // Centre vertically
#define FORMAT_SIZE_TO_CONTENTS 4
                                // Resize shape to contents

// Shadow mode
#define SHADOW_NONE           0
#define SHADOW_LEFT           1
#define SHADOW_RIGHT          2

/*
 * Declare types
 *
 */

#define SHAPE_BASIC           wxTYPE_USER + 1
#define SHAPE_RECTANGLE       wxTYPE_USER + 2
#define SHAPE_CONTROL_POINT   wxTYPE_USER + 3

#define OP_CLICK_LEFT  1
#define OP_CLICK_RIGHT 2
#define OP_DRAG_LEFT 4
#define OP_DRAG_RIGHT 8

#define OP_ALL (OP_CLICK_LEFT | OP_CLICK_RIGHT | OP_DRAG_LEFT | OP_DRAG_RIGHT)

class ShapeCanvas;
class ControlPoint;
class Shape: public wxObject
{
 private:
  wxObject *ClientData;
 protected:
 public:
  Bool formatted;
  float xpos, ypos;
  wxPen *pen;
  wxBrush *brush;
  wxFont *font;
  wxColour *text_colour;
  char *textColourName;
  ShapeCanvas *canvas;
  wxDC *dc;
  char *textString;
  wxList control_points;
  Bool visible;
  Bool disable_label;
  long id;
  Bool selected;
  int sensitivity;
  Bool draggable;
  int formatMode;
  float textMarginX;    // Gap between text and border
  float textMarginY;

  Shape(ShapeCanvas *can = NULL);
  virtual ~Shape(void);
  virtual void GetBoundingBoxMax(float *width, float *height);
  virtual void GetBoundingBoxMin(float *width, float *height) = 0;
  inline ShapeCanvas *GetCanvas(void) { return canvas; }
  void SetCanvas(ShapeCanvas *the_canvas);
  virtual void AddToCanvas(ShapeCanvas *the_canvas);
  inline float GetX(void) { return xpos; }
  inline float GetY(void) { return ypos; }

  virtual void OnDraw(void);
  virtual void OnDrawContents(void);
  virtual void OnErase(void);
  virtual void OnEraseContents(void);
  virtual void OnHighlight(void);
  virtual void OnLeftClick(float x, float y, int keys = 0);
  virtual void OnRightClick(float x, float y, int keys = 0);
  virtual void OnSize(float x, float y);
  virtual void OnMove(float x, float y, float old_x, float old_y, Bool display = TRUE);

  virtual void OnDragLeft(Bool draw, float x, float y, int keys=0);
  virtual void OnBeginDragLeft(float x, float y, int keys=0);
  virtual void OnEndDragLeft(float x, float y, int keys=0);
  virtual void OnDragRight(Bool draw, float x, float y, int keys=0);
  virtual void OnBeginDragRight(float x, float y, int keys=0);
  virtual void OnEndDragRight(float x, float y, int keys=0);
  virtual void OnDrawOutline(void);
  virtual void OnDrawControlPoints(void);
  virtual void OnEraseControlPoints(void);

  virtual void OnBeginSize(float w, float h) { }
  virtual void OnEndSize(float w, float h) { }

  virtual void MakeControlPoints(void);
  virtual void DeleteControlPoints(void);
  virtual void ResetControlPoints(void);

  virtual void Select(Bool select = TRUE);
  virtual Bool Selected(void);
  void SetDraggable(Bool drag, Bool recursive = FALSE);
  virtual Bool HitTest(float x, float y, float *distance);

  void SetPen(wxPen *pen);
  void SetBrush(wxBrush *brush);
  void SetClientData(wxObject *client_data);
  wxObject *GetClientData(void);

  virtual void Show(Bool show);
  virtual void Move(float x1, float y1, Bool display = TRUE);
  virtual void Erase(void);
  virtual void EraseContents(void);
  virtual void Draw(void);
  virtual void Flash(void);
  virtual void DrawContents(void);  // E.g. for drawing text label
  virtual void SetSize(float x, float y, Bool recursive = TRUE);
  void Attach(ShapeCanvas *can);
  void Detach(void);
  virtual void SetDC(wxDC *the_dc);

  inline wxPen *GetPen(void) { return pen; }
  inline wxBrush *GetBrush(void) { return brush; }

  virtual void FormatText(char *s, int regionId = 0);
  virtual inline void SetFormatMode(int mode, int regionId = 0) { formatMode = mode; };
  virtual void SetFont(wxFont *font, int regionId = 0);
  virtual inline wxFont *GetFont(int regionId = 0) { return font; }

  virtual Bool Draggable(void) { return TRUE; }
};

class RectangleShape: public Shape
{
 protected:
 public:
  float width;
  float height;

  RectangleShape(float w, float h);
  void GetBoundingBoxMin(float *w, float *h);
  void OnDraw(void);
  void SetSize(float x, float y, Bool recursive = TRUE);
};

class ControlPoint: public RectangleShape
{
 public:
  int type;
  float xoffset;
  float yoffset;
  Shape *canvas_object;
  wxCursor *old_cursor;

  ControlPoint(ShapeCanvas *the_canvas, Shape *object, float size, float the_xoffset, float the_yoffset, int the_type);
  void OnDraw(void);
  void OnDrawContents(void);
  void OnDragLeft(Bool draw, float x, float y, int keys=0);
  void OnBeginDragLeft(float x, float y, int keys=0);
  void OnEndDragLeft(float x, float y, int keys=0);
};

// Drag states
#define NoDragging             0
#define StartDraggingLeft      1
#define ContinueDraggingLeft   2
#define StartDraggingRight     3
#define ContinueDraggingRight  4

class ShapeCanvas: public wxCanvas
{
 public:
  Bool quick_edit_mode;
  Bool snap_to_grid;
  float grid_spacing;
  int mouseTolerance;

  wxList *object_list;
  int DragState;
  float old_drag_x, old_drag_y;     // Previous drag coordinates
  float first_drag_x, first_drag_y; // INITIAL drag coordinates
  float OutlineStartX;
  float OutlineStartY;

  Shape *DraggedObject;

  ShapeCanvas(wxFrame *frame, int x = -1, int y = -1, int width = -1, int height = -1,
               int style = wxBORDER | wxRETAINED);
  ~ShapeCanvas(void);

  void DrawOutline(float x1, float y1, float x2, float y2);

  virtual void OnPaint(void);
  // This sends higher-level events to images or canvas
  virtual void OnEvent(wxMouseEvent& event);

  virtual void OnLeftClick(float x, float y, int keys = 0);
  virtual void OnRightClick(float x, float y, int keys = 0);

  virtual void OnDragLeft(Bool draw, float x, float y, int keys=0); // Erase if draw false
  virtual void OnBeginDragLeft(float x, float y, int keys=0);
  virtual void OnEndDragLeft(float x, float y, int keys=0);

  virtual void OnDragRight(Bool draw, float x, float y, int keys=0); // Erase if draw false
  virtual void OnBeginDragRight(float x, float y, int keys=0);
  virtual void OnEndDragRight(float x, float y, int keys=0);

  virtual void Redraw(void); // Called automatically by default OnPaint handler
  virtual void Clear(void);

  // Add object to end of object list
  virtual void AddObject(Shape *object);

  // Add object to front of object list
  virtual void InsertObject(Shape *object);

  void SetSnapToGrid(Bool snap);
  void SetGridSpacing(float spacing);
  inline float GetGridSpacing(void) { return grid_spacing; }
  inline Bool GetSnapToGrid(void) { return snap_to_grid; }
  void Snap(float *x, float *y);

  inline void SetQuickEditMode(Bool qem) { quick_edit_mode = qem; }
  inline Bool GetQuickEditMode(void) { return quick_edit_mode; }

  virtual void RemoveObject(Shape *object);
  virtual void RemoveAllObjects(void);
  virtual void ShowAll(Bool show);
  virtual Shape *FindObject(float x, float y);  // Find object for mouse click

  inline void SetMouseTolerance(int tol) { mouseTolerance = tol; }
  inline wxList *GetObjectList(void) { return object_list; }
};

extern wxFont *normal_font;
extern wxFont *italic_font;
extern wxFont *swiss_font_4;
extern wxFont *swiss_font_6;
extern wxFont *swiss_font_8;
extern wxFont *swiss_font_10;
extern wxFont *swiss_font_12;
extern wxFont *swiss_font_14;
extern wxFont *swiss_font_18;
extern wxFont *swiss_font_24;
extern wxPen *red_pen;
extern wxPen *cyan_pen;
extern wxPen *green_pen;
extern wxPen *black_pen;

extern wxBrush *blue_brush;
extern wxBrush *green_brush;
extern wxBrush *white_brush;
extern wxBrush *black_brush;
extern wxBrush *cyan_brush;
extern wxBrush *red_brush;

extern wxPen *white_background_pen;
extern wxBrush *transparent_brush;
extern wxPen *transparent_pen;
extern wxBrush *white_background_brush;
extern wxPen *black_foreground_pen;
extern wxPen *black_dashed_pen;

extern wxCursor *GraphicsBullseyeCursor;

void GraphicsInitialize(void);
void GraphicsCleanUp(void);
#endif // hy_shapesh
