/*
 * File:	shapes.cc
 * Purpose:	
 * Author:	Julian Smart
 * Created:	1993
 * Updated:	
 * Copyright:	(c) 1993, AIAI, University of Edinburgh
 */

static const char sccsid[] = "%W% %G%";
 
#include <wx.h>
#include <iostream.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>

#include "shapes.h"

#define CONTROL_POINT_SIZE       6
#define CONTROL_POINT_VERTICAL   1
#define CONTROL_POINT_HORIZONTAL 2
#define CONTROL_POINT_DIAGONAL   3

wxFont *normal_font;
wxFont *swiss_font_4;
wxFont *swiss_font_6;
wxFont *swiss_font_8;
wxFont *swiss_font_10;
wxFont *swiss_font_12;
wxFont *swiss_font_14;
wxFont *swiss_font_18;
wxFont *swiss_font_24;
wxFont *italic_font;
wxPen *red_pen;
wxPen *cyan_pen;
wxPen *green_pen;
wxPen *black_pen;
wxBrush *blue_brush;
wxBrush *green_brush;
wxBrush *white_brush;
wxBrush *black_brush;
wxBrush *cyan_brush;
wxBrush *red_brush;
wxPen *white_background_pen;
wxPen *transparent_pen;
wxBrush *transparent_brush;
wxBrush *white_background_brush;
wxPen *black_foreground_pen;
wxPen *black_dashed_pen;
wxCursor *GraphicsBullseyeCursor = NULL;

Shape::Shape(ShapeCanvas *can)
{
  __type = SHAPE_BASIC;
  id = 0;
  textString = NULL;
  formatted = FALSE;
  dc = NULL;
  canvas = can;
  xpos = 0.0; ypos = 0.0;
  pen = black_pen;
  brush = white_brush;
  font = normal_font;
  text_colour = wxBLACK;
  textColourName = copystring("BLACK");
  visible = FALSE;
  ClientData = NULL;
  selected = FALSE;
  disable_label = FALSE;
  sensitivity = OP_ALL;
  draggable = TRUE;
  formatMode = FORMAT_NONE;
  textMarginX = 2.0;
  textMarginY = 2.0;
}

Shape::~Shape(void)
{
  if (textString) delete[] textString;
  if (textColourName) delete[] textColourName;

  if (canvas)
    canvas->RemoveObject(this);
}

void Shape::SetClientData(wxObject *client_data)
{
  ClientData = client_data;
}

wxObject *Shape::GetClientData(void)
{
  return ClientData;
}

void Shape::SetDC(wxDC *the_dc)
{
  dc = the_dc;
}

void Shape::SetDraggable(Bool drag, Bool recursive)
{ 
  draggable = drag;
  if (draggable)
    sensitivity |= OP_DRAG_LEFT;
  else
    if (sensitivity & OP_DRAG_LEFT)
      sensitivity = sensitivity - OP_DRAG_LEFT;
}

void Shape::SetCanvas(ShapeCanvas *the_canvas)
{
  canvas = the_canvas;
}

void Shape::AddToCanvas(ShapeCanvas *the_canvas)
{
  the_canvas->AddObject(this);
}

Bool Shape::HitTest(float x, float y, float *distance)
{
  float width = 0.0, height = 0.0;
  GetBoundingBoxMin(&width, &height);
  if (fabs(width) < 4.0) width = 4.0;
  if (fabs(height) < 4.0) height = 4.0;

  width += (float)4.0; height += (float)4.0; // Allowance for inaccurate mousing

  float left = (float)(xpos - (width/2.0));
  float top = (float)(ypos - (height/2.0));
  float right = (float)(xpos + (width/2.0));
  float bottom = (float)(ypos + (height/2.0));

  if (x >= left && x <= right && y >= top && y <= bottom)
  {
    float nearest = 999999.0;
    float l = (float)sqrt(((GetX() - x) * (GetX() - x)) +
                 ((GetY() - y) * (GetY() - y)));

    *distance = l;
    return TRUE;
  }
  else return FALSE;
}

// Format a text string according to the bounding box, add
// strings to text list
void Shape::FormatText(char *s, int i)
{
  if (textString) delete[] textString;
  textString = copystring(s);
  formatted = TRUE;
}

void Shape::SetPen(wxPen *the_pen)
{
  pen = the_pen;
}

void Shape::SetBrush(wxBrush *the_brush)
{
  brush = the_brush;
}

/*
 * Region functions
 *
 */
void Shape::SetFont(wxFont *the_font, int regionId)
{
  font = the_font;
}

void Shape::OnDraw(void)
{
}

void Shape::OnDrawContents(void)
{
  float width, height;
  GetBoundingBoxMin(&width, &height);
  if (dc)
  {
    if (font) dc->SetFont(font);
    if (pen) dc->SetPen(pen);

    if (text_colour) dc->SetTextForeground(text_colour);
    dc->SetBackgroundMode(wxTRANSPARENT);

    if (formatted && textString)
    {
      dc->SetClippingRegion(
                    (float)(xpos - width/2.0), (float)(ypos - height/2.0),
                    (float)width, (float)height);

      dc->DrawText(textString, (float)(textMarginX + xpos - (width/2.0)),
                               (float)(textMarginY + ypos - (height/2.0)));
      dc->DestroyClippingRegion();
    }
  }
}

void Shape::DrawContents(void)
{
  OnDrawContents();
}

void Shape::OnSize(float x, float y)
{
}

void Shape::OnMove(float x, float y, float old_x, float old_y, Bool display)
{
}

void Shape::OnErase(void)
{
  if (!visible)
    return;

  OnEraseContents();
}

void Shape::OnEraseContents(void)
{
  if (!visible)
    return;

  float maxX, maxY, minX, minY;
  float xp = GetX();
  float yp = GetY();
  GetBoundingBoxMin(&minX, &minY);
  GetBoundingBoxMax(&maxX, &maxY);
  float topLeftX = (float)(xp - (maxX / 2.0));
  float topLeftY = (float)(yp - (maxY / 2.0));

  if (dc)
  {
    int penWidth = 0;
    if (pen)
      penWidth = pen->GetWidth();

    dc->SetPen(white_background_pen);
    dc->SetBrush(white_background_brush);
    dc->DrawRectangle((float)(topLeftX - penWidth), (float)(topLeftY - penWidth), 
                      (float)(maxX + penWidth*2.0), (float)(maxY + penWidth*2.0));
  }
}

void Shape::OnHighlight(void) {}
void Shape::OnLeftClick(float x, float y, int keys) {}
void Shape::OnRightClick(float x, float y, int keys) {}

float DragOffsetX = 0.0;
float DragOffsetY = 0.0;

void Shape::OnDragLeft(Bool draw, float x, float y, int keys)
{
  float xx, yy;
  xx = x + DragOffsetX;
  yy = y + DragOffsetY;

  canvas->Snap(&xx, &yy);
  xpos = xx; ypos = yy;
  OnDrawOutline();
}

void Shape::OnBeginDragLeft(float x, float y, int keys)
{
  DragOffsetX = xpos - x;
  DragOffsetY = ypos - y;

  Erase();
  float xx, yy;
  xx = x + DragOffsetX;
  yy = y + DragOffsetY;
  canvas->Snap(&xx, &yy);
  xpos = xx; ypos = yy;
  dc->SetLogicalFunction(wxXOR);

  dc->SetPen(black_dashed_pen);
  dc->SetBrush(transparent_brush);

  OnDrawOutline();
}

void Shape::OnEndDragLeft(float x, float y, int keys)
{
  dc->SetLogicalFunction(wxCOPY);

  canvas->Snap(&xpos, &ypos);
  Move(xpos, ypos);
  if (canvas && !canvas->quick_edit_mode) canvas->Redraw();
}

void Shape::OnDragRight(Bool draw, float x, float y, int keys) {}
void Shape::OnBeginDragRight(float x, float y, int keys) {}
void Shape::OnEndDragRight(float x, float y, int keys) {}

void Shape::OnDrawOutline(void)
{
  float maxX, maxY, minX, minY;
  GetBoundingBoxMax(&maxX, &maxY);
  GetBoundingBoxMin(&minX, &minY);

  float top_left_x = (float)(xpos - minX/2.0);
  float top_left_y = (float)(ypos - minY/2.0);
  float top_right_x = (float)(top_left_x + maxX);
  float top_right_y = (float)top_left_y;
  float bottom_left_x = (float)top_left_x;
  float bottom_left_y = (float)(top_left_y + maxY);
  float bottom_right_x = (float)top_right_x;
  float bottom_right_y = (float)bottom_left_y;

  wxPoint points[5];
  points[0].x = top_left_x; points[0].y = top_left_y;
  points[1].x = top_right_x; points[1].y = top_right_y;
  points[2].x = bottom_right_x; points[2].y = bottom_right_y;
  points[3].x = bottom_left_x; points[3].y = bottom_left_y;
  points[4].x = top_left_x; points[4].y = top_left_y;

  dc->DrawLines(5, points);
}

void Shape::Attach(ShapeCanvas *can)
{
  canvas = can;
}

void Shape::Detach(void)
{
  canvas = NULL;
}

void Shape::Move(float x, float y, Bool display)
{
  float old_x = xpos;
  float old_y = ypos;
  xpos = x; ypos = y;
  OnMove(x, y, old_x, old_y, display);
  ResetControlPoints();
  if (display)
    Draw();
  if (display)
    OnDrawControlPoints();
}

void Shape::Draw(void)
{
  if (visible)
  {
    OnDraw();
    OnDrawContents();
  }
}

void Shape::Flash(void)
{
  if (dc)
  {
    dc->SetLogicalFunction(wxXOR);
    Draw();
    dc->SetLogicalFunction(wxCOPY);
    Draw();
  }
}

void Shape::Show(Bool show)
{
  visible = show;
}

void Shape::Erase(void)
{
  OnErase();
  OnEraseControlPoints();
}

void Shape::EraseContents(void)
{
  OnEraseContents();
}

void Shape::SetSize(float x, float y, Bool recursive) {}
void Shape::MakeControlPoints(void)
{
  float maxX, maxY, minX, minY;
  GetBoundingBoxMax(&maxX, &maxY);
  GetBoundingBoxMin(&minX, &minY);
  float widthMin = (float)(minX + CONTROL_POINT_SIZE + 2);
  float heightMin = (float)(minY + CONTROL_POINT_SIZE + 2);
  float top = (float)(- (heightMin / 2.0));
  float bottom = (float)(heightMin / 2.0 + (maxY - minY));
  float left = (float)(- (widthMin / 2.0));
  float right = (float)(widthMin / 2.0 + (maxX - minX));
  ControlPoint *control = new ControlPoint(canvas, this, CONTROL_POINT_SIZE, left, top, 
                                           CONTROL_POINT_DIAGONAL);
  canvas->AddObject(control);
  control_points.Append(control);
  control = new ControlPoint(canvas, this, CONTROL_POINT_SIZE, 0, top, 
                                           CONTROL_POINT_VERTICAL);
  canvas->AddObject(control);
  control_points.Append(control);
  control = new ControlPoint(canvas, this, CONTROL_POINT_SIZE, right, top, 
                                           CONTROL_POINT_DIAGONAL);
  canvas->AddObject(control);
  control_points.Append(control);
  control = new ControlPoint(canvas, this, CONTROL_POINT_SIZE, right, 0, 
                                           CONTROL_POINT_HORIZONTAL);
  canvas->AddObject(control);
  control_points.Append(control);
  control = new ControlPoint(canvas, this, CONTROL_POINT_SIZE, right, bottom, 
                                           CONTROL_POINT_DIAGONAL);
  canvas->AddObject(control);
  control_points.Append(control);
  control = new ControlPoint(canvas, this, CONTROL_POINT_SIZE, 0, bottom, 
                                           CONTROL_POINT_VERTICAL);
  canvas->AddObject(control);
  control_points.Append(control);
  control = new ControlPoint(canvas, this, CONTROL_POINT_SIZE, left, bottom, 
                                           CONTROL_POINT_DIAGONAL);
  canvas->AddObject(control);
  control_points.Append(control);
  control = new ControlPoint(canvas, this, CONTROL_POINT_SIZE, left, 0, 
                                           CONTROL_POINT_HORIZONTAL);
  canvas->AddObject(control);
  control_points.Append(control);
}

void Shape::ResetControlPoints(void)
{
  if (control_points.Number() < 1) return;
  float maxX, maxY, minX, minY;
  GetBoundingBoxMax(&maxX, &maxY);
  GetBoundingBoxMin(&minX, &minY);
  float widthMin = (float)(minX + CONTROL_POINT_SIZE + 2);
  float heightMin = (float)(minY + CONTROL_POINT_SIZE + 2);
  float top = (float)(- (heightMin / 2.0));
  float bottom = (float)(heightMin / 2.0 + (maxY - minY));
  float left = (float)(- (widthMin / 2.0));
  float right = (float)(widthMin / 2.0 + (maxX - minX));
  wxNode *node = control_points.First();
  ControlPoint *control = (ControlPoint *)node->Data();
  control->xoffset = left; control->yoffset = top;
  node = node->Next(); control = (ControlPoint *)node->Data();
  control->xoffset = 0; control->yoffset = top;
  node = node->Next(); control = (ControlPoint *)node->Data();
  control->xoffset = right; control->yoffset = top;
  node = node->Next(); control = (ControlPoint *)node->Data();
  control->xoffset = right; control->yoffset = 0;
  node = node->Next(); control = (ControlPoint *)node->Data();
  control->xoffset = right; control->yoffset = bottom;
  node = node->Next(); control = (ControlPoint *)node->Data();
  control->xoffset = 0; control->yoffset = bottom;
  node = node->Next(); control = (ControlPoint *)node->Data();
  control->xoffset = left; control->yoffset = bottom;
  node = node->Next(); control = (ControlPoint *)node->Data();
  control->xoffset = left; control->yoffset = 0;
}

void Shape::DeleteControlPoints(void)
{
  wxNode *node = control_points.First();
  while (node)
  {
    ControlPoint *control = (ControlPoint *)node->Data();
    control->OnErase();
    canvas->RemoveObject(control);
    delete control; delete node;
    node = control_points.First();
  }
}

void Shape::OnDrawControlPoints(void)
{
  wxNode *node = control_points.First();
  while (node)
  {
    ControlPoint *control = (ControlPoint *)node->Data();
    control->Draw();
    node = node->Next();
  }
}

void Shape::OnEraseControlPoints(void)
{
  wxNode *node = control_points.First();
  while (node)
  {
    ControlPoint *control = (ControlPoint *)node->Data();
    control->Erase();
    node = node->Next();
  }
}

void Shape::Select(Bool select)
{
  selected = select;
  if (select && (control_points.Number() == 0))
  {
    MakeControlPoints();
    OnDrawControlPoints();
  }
  if (!select && (control_points.Number() > 0))
  {
    DeleteControlPoints();
  }
}

Bool Shape::Selected(void)
{
  return selected;
}

void Shape::GetBoundingBoxMax(float *w, float *h)
{
  float ww, hh;
  GetBoundingBoxMin(&ww, &hh);
  *w = ww;
  *h = hh;
}

// Rectangle object

RectangleShape::RectangleShape(float w, float h)
{
  width = w; height = h;
  __type = SHAPE_RECTANGLE;
}

void RectangleShape::OnDraw(void)
{
  if (dc)
  {
    float x1 = (float)(xpos - width/2.0);
    float y1 = (float)(ypos - height/2.0);
    if (pen) dc->SetPen(pen);
    if (brush)dc->SetBrush(brush);
    dc->DrawRectangle(x1, y1, width, height);
  }
}

void RectangleShape::GetBoundingBoxMin(float *the_width, float *the_height)
{
  *the_width = width;
  *the_height = height;
}

void RectangleShape::SetSize(float x, float y, Bool recursive)
{
  width = x;
  height = y;
}

// Control points
ControlPoint::ControlPoint(ShapeCanvas *the_canvas, Shape *object, float size, float the_xoffset, float the_yoffset, int the_type):RectangleShape(size, size)
{
  __type = SHAPE_CONTROL_POINT;
  canvas = the_canvas;
  canvas_object = object;
  xoffset = the_xoffset;
  yoffset = the_yoffset;
  type = the_type;
  SetPen(black_foreground_pen);
  SetBrush(black_brush);
  old_cursor = NULL;
  visible = TRUE;
}

void ControlPoint::OnDrawContents(void) {}

void ControlPoint::OnDraw(void)
{
  xpos = canvas_object->GetX() + xoffset;
  ypos = canvas_object->GetY() + yoffset;
  RectangleShape::OnDraw();
}

// Implement resizing of canvas object
void ControlPoint::OnDragLeft(Bool draw, float x, float y, int keys)
{
  float bound_x;
  float bound_y;
  canvas_object->GetBoundingBoxMin(&bound_x, &bound_y);
  float new_width = (float)(2.0*fabs(x - canvas_object->xpos));
  float new_height = (float)(2.0*fabs(y - canvas_object->ypos));

  // Constrain sizing according to what control point you're dragging
  if (type == CONTROL_POINT_HORIZONTAL)
    new_height = bound_y;
  if (type == CONTROL_POINT_VERTICAL)
    new_width = bound_x;
  if (type == CONTROL_POINT_DIAGONAL && (keys & KEY_SHIFT))
  {
    new_height = bound_y*(new_width/bound_x);
  }
  canvas_object->OnBeginSize(new_width, new_height);
  canvas_object->SetSize(new_width, new_height);
  canvas_object->OnDrawOutline();
}

void ControlPoint::OnBeginDragLeft(float x, float y, int keys)
{
  canvas_object->Erase();
  dc->SetLogicalFunction(wxXOR);
  float bound_x;
  float bound_y;
  canvas_object->GetBoundingBoxMin(&bound_x, &bound_y);
  float new_width = (float)(2.0*fabs(x - canvas_object->xpos));
  float new_height = (float)(2.0*fabs(y - canvas_object->ypos));
  // Constrain sizing according to what control point you're dragging
  if (type == CONTROL_POINT_HORIZONTAL)
    new_height = bound_y;
  if (type == CONTROL_POINT_VERTICAL)
    new_width = bound_x;
  if (type == CONTROL_POINT_DIAGONAL && (keys & KEY_SHIFT))
    new_height = bound_y*(new_width/bound_x);

  canvas_object->SetSize(new_width, new_height, FALSE);
  dc->SetPen(black_dashed_pen);
  dc->SetBrush(transparent_brush);
  canvas_object->OnDrawOutline();
}

void ControlPoint::OnEndDragLeft(float x, float y, int keys)
{
  dc->SetLogicalFunction(wxCOPY);
  canvas_object->ResetControlPoints();
  canvas_object->Move(canvas_object->GetX(), canvas_object->GetY());
  if (!canvas->quick_edit_mode) canvas->Redraw();
  float width, height;
  canvas_object->GetBoundingBoxMax(&width, &height);
  canvas_object->OnEndSize(width, height);
}

void GraphicsInitialize(void)
{
  GraphicsBullseyeCursor = new wxCursor(wxCURSOR_BULLSEYE);
  normal_font = new wxFont(12, wxMODERN, wxNORMAL, wxNORMAL);
  italic_font = new wxFont(12, wxROMAN, wxITALIC, wxNORMAL);
  swiss_font_4 = new wxFont(4, wxSWISS, wxNORMAL, wxNORMAL);
  swiss_font_6 = new wxFont(6, wxSWISS, wxNORMAL, wxNORMAL);
  swiss_font_8 = new wxFont(8, wxSWISS, wxNORMAL, wxNORMAL);
  swiss_font_10 = new wxFont(10, wxSWISS, wxNORMAL, wxNORMAL);
  swiss_font_12 = new wxFont(12, wxSWISS, wxNORMAL, wxNORMAL);
  swiss_font_14 = new wxFont(14, wxSWISS, wxNORMAL, wxNORMAL);
  swiss_font_18 = new wxFont(18, wxSWISS, wxNORMAL, wxNORMAL);
  swiss_font_24 = new wxFont(24, wxSWISS, wxNORMAL, wxNORMAL);
  red_pen = new wxPen("RED", 3, wxSOLID);
  cyan_pen = new wxPen("CYAN", 3, wxSOLID);
  green_pen = new wxPen("GREEN", 1, wxSOLID);
  black_pen = new wxPen("BLACK", 1, wxSOLID);
  blue_brush = new wxBrush("BLUE", wxSOLID);
  green_brush = new wxBrush("GREEN", wxSOLID);
  white_brush = new wxBrush("WHITE", wxSOLID);
  black_brush = new wxBrush("BLACK", wxSOLID);
  cyan_brush = new wxBrush("CYAN", wxSOLID);
  red_brush = new wxBrush("RED", wxSOLID);
  white_background_pen = new wxPen("WHITE", 1, wxSOLID);
  transparent_pen = new wxPen("WHITE", 1, wxTRANSPARENT);
  transparent_brush = new wxBrush("BLACK", wxTRANSPARENT);
  white_background_brush = new wxBrush("WHITE", wxSOLID);
  black_foreground_pen = new wxPen("BLACK", 1, wxSOLID);
  black_dashed_pen = new wxPen("BLACK", 1, wxSHORT_DASH);
  // Set up type hierarchy
  wxAllTypes.AddType(SHAPE_BASIC,       wxTYPE_OBJECT,      "Basic shape");
  wxAllTypes.AddType(SHAPE_RECTANGLE,   SHAPE_BASIC,        "Rectangle shape");
  wxAllTypes.AddType(SHAPE_CONTROL_POINT, SHAPE_RECTANGLE,  "Control point shape");
}

void GraphicsCleanUp(void) {}

// Object canvas
ShapeCanvas::ShapeCanvas(wxFrame *frame, int x, int y, int w, int h, int style):
  wxCanvas(frame, x, y, w, h, style)
{
  quick_edit_mode = FALSE;
  snap_to_grid = TRUE;
  grid_spacing = 5.0;
  object_list = new wxList;
  DragState = NoDragging;
  DraggedObject = NULL;
  old_drag_x = 0;
  old_drag_y = 0;
  first_drag_x = 0;
  first_drag_y = 0;
  mouseTolerance = DEFAULT_MOUSE_TOLERANCE;
}

ShapeCanvas::~ShapeCanvas(void)
{
  if (object_list)
    delete object_list;
}

void ShapeCanvas::SetSnapToGrid(Bool snap)
{
  snap_to_grid = snap;
}

void ShapeCanvas::SetGridSpacing(float spacing)
{
  grid_spacing = spacing;
}

void ShapeCanvas::Snap(float *x, float *y)
{
  if (snap_to_grid)
  {
    *x = grid_spacing * ((int)(*x/grid_spacing + 0.5));
    *y = grid_spacing * ((int)(*y/grid_spacing + 0.5));
  }
}


void ShapeCanvas::OnPaint(void)
{
  GetDC()->Clear(); Redraw();
}

void ShapeCanvas::Redraw(void)
{
  GetDC()->BeginDrawing();
  if (object_list)
  {
    wxCursor *old_cursor = SetCursor(wxHOURGLASS_CURSOR);
    wxNode *current = object_list->First();
    while (current)
    {
      Shape *object = (Shape *)current->Data();
      object->Draw();
      current = current->Next();
    }
    SetCursor(old_cursor);
  }
  GetDC()->EndDrawing();
}

void ShapeCanvas::Clear(void)
{
   wxCanvas::Clear();
}

void ShapeCanvas::AddObject(Shape *object)
{
  object_list->Append(object);
  object->SetCanvas(this);
  object->SetDC(GetDC());
}

void ShapeCanvas::InsertObject(Shape *object)
{
  object_list->Insert(object);
  object->SetCanvas(this);
}

void ShapeCanvas::RemoveObject(Shape *object)
{
  object_list->DeleteObject(object);
}

// Should this delete the actual objects too? I think not.
void ShapeCanvas::RemoveAllObjects(void)
{
  object_list->Clear();
}

void ShapeCanvas::ShowAll(Bool show)
{
  wxNode *current = object_list->First();
  while (current)
  {
    Shape *object = (Shape *)current->Data();
    object->Show(show);
    current = current->Next();
  }
}

void ShapeCanvas::OnEvent(wxMouseEvent& event)
{
  float x, y;
  event.Position(&x, &y);
  int keys = 0;
  if (event.ShiftDown())
    keys = keys | KEY_SHIFT;
  if (event.ControlDown())
    keys = keys | KEY_CTRL;
  Bool dragging = event.Dragging();
  if (dragging)
  {
    wxDC *dc = GetDC();
    int dx = abs(dc->LogicalToDeviceX(x - first_drag_x));
    int dy = abs(dc->LogicalToDeviceY(y - first_drag_y));
    if ((dx <= mouseTolerance) && (dy <= mouseTolerance)) return;
  }
  if (dragging && DraggedObject && DragState == StartDraggingLeft)
  {
    DragState = ContinueDraggingLeft;

    // If the object isn't draggable, transfer message to canvas
    if (DraggedObject->Draggable())
      DraggedObject->OnBeginDragLeft((float)x, (float)y, keys);
    else
    {
      DraggedObject = NULL;
      OnBeginDragLeft((float)x, (float)y, keys);
    }
    old_drag_x = x; old_drag_y = y;
  }
  else if (dragging && DraggedObject && DragState == ContinueDraggingLeft)
  { 
    DraggedObject->OnDragLeft(FALSE, old_drag_x, old_drag_y, keys);
    DraggedObject->OnDragLeft(TRUE, (float)x, (float)y, keys);
    old_drag_x = x; old_drag_y = y;
  }
  else if (event.LeftUp() && DraggedObject && DragState == ContinueDraggingLeft)
  {
    DragState = NoDragging;
    DraggedObject->OnDragLeft(FALSE, old_drag_x, old_drag_y, keys);
    DraggedObject->OnEndDragLeft((float)x, (float)y, keys);
    DraggedObject = NULL;
  }
  else if (dragging && DraggedObject && DragState == StartDraggingRight)
  {
    DragState = ContinueDraggingRight;
    if (DraggedObject->Draggable())
      DraggedObject->OnBeginDragRight((float)x, (float)y, keys);
    else
    {
      DraggedObject = NULL;
      OnBeginDragRight((float)x, (float)y, keys);
    }
    old_drag_x = x; old_drag_y = y;
  }
  else if (dragging && DraggedObject && DragState == ContinueDraggingRight)
  { 
    DraggedObject->OnDragRight(FALSE, old_drag_x, old_drag_y, keys);
    DraggedObject->OnDragRight(TRUE, (float)x, (float)y, keys);
    old_drag_x = x; old_drag_y = y;
  }
  else if (event.RightUp() && DraggedObject && DragState == ContinueDraggingRight)
  {
    DragState = NoDragging;
    DraggedObject->OnDragRight(FALSE, old_drag_x, old_drag_y, keys);
    DraggedObject->OnEndDragRight((float)x, (float)y, keys);
    DraggedObject = NULL;
  }
  else if (dragging && !DraggedObject && DragState == StartDraggingLeft)
  {
    DragState = ContinueDraggingLeft;
    OnBeginDragLeft((float)x, (float)y, keys);
    old_drag_x = x; old_drag_y = y;
  }
  else if (dragging && !DraggedObject && DragState == ContinueDraggingLeft)
  { 
    OnDragLeft(FALSE, old_drag_x, old_drag_y, keys);
    OnDragLeft(TRUE, (float)x, (float)y, keys);
    old_drag_x = x; old_drag_y = y;
  }
  else if (event.LeftUp() && !DraggedObject && DragState == ContinueDraggingLeft)
  {
    DragState = NoDragging;
    OnDragLeft(FALSE, old_drag_x, old_drag_y, keys);
    OnEndDragLeft((float)x, (float)y, keys);
    DraggedObject = NULL;
  }
  else if (dragging && !DraggedObject && DragState == StartDraggingRight)
  {
    DragState = ContinueDraggingRight;
    OnBeginDragRight((float)x, (float)y, keys);
    old_drag_x = x; old_drag_y = y;
  }
  else if (dragging && !DraggedObject && DragState == ContinueDraggingRight)
  { 
    OnDragRight(FALSE, old_drag_x, old_drag_y, keys);
    OnDragRight(TRUE, (float)x, (float)y, keys);
    old_drag_x = x; old_drag_y = y;
  }
  else if (event.RightUp() && !DraggedObject && DragState == ContinueDraggingRight)
  {
    DragState = NoDragging;
    OnDragRight(FALSE, old_drag_x, old_drag_y, keys);
    OnEndDragRight((float)x, (float)y, keys);
    DraggedObject = NULL;
  }
  else if (event.IsButton())
  {
    Shape *nearest_object = FindObject(x, y);
    if (nearest_object)
    {
      if (event.LeftDown())
      {
        DraggedObject = nearest_object;
        DragState = StartDraggingLeft;
        first_drag_x = x;
        first_drag_y = y;
      }
      else if (event.LeftUp())
      {
        if (nearest_object == DraggedObject)
          nearest_object->OnLeftClick((float)x, (float)y, keys);
        DraggedObject = NULL;
        DragState = NoDragging;
      }
      else if (event.RightDown())
      {
        DraggedObject = nearest_object;
        DragState = StartDraggingRight;
        first_drag_x = x;
        first_drag_y = y;
      }
      else if (event.RightUp())
      {
        if (nearest_object == DraggedObject)
          nearest_object->OnRightClick((float)x, (float)y, keys);
        DraggedObject = NULL;
        DragState = NoDragging;
      }
    }
    else
    {
      if (event.LeftDown())
      {
        DraggedObject = NULL;
        DragState = StartDraggingLeft;
      }
      else if (event.LeftUp())
      {
        OnLeftClick((float)x, (float)y, keys);
        DraggedObject = NULL;
        DragState = NoDragging;
      }
      else if (event.RightDown())
      {
        DraggedObject = NULL;
        DragState = StartDraggingRight;
      }
      else if (event.RightUp())
      {
        OnRightClick((float)x, (float)y, keys);
        DraggedObject = NULL;
        DragState = NoDragging;
      }
    }
  }
}

Shape *ShapeCanvas::FindObject(float x, float y)
{
  float nearest = 100000.0;
  Shape *nearest_object = NULL;
  wxNode *current = object_list->Last();
  while (current)
  {
    Shape *object = (Shape *)current->Data();
    float dist;
    if (object->HitTest(x, y, &dist))
    {
      nearest = dist;
      nearest_object = object;
      current = NULL;
    }
    if (current)
      current = current->Previous();
  }
  return nearest_object;
}

void ShapeCanvas::DrawOutline(float x1, float y1, float x2, float y2)
{
  wxDC *dc = GetDC();
  dc->SetPen(black_dashed_pen);
  dc->SetBrush(transparent_brush);
  wxPoint points[5];
  points[0].x = x1; points[0].y = y1; points[1].x = x2; points[1].y = y1;
  points[2].x = x2; points[2].y = y2; points[3].x = x1; points[3].y = y2;
  points[4].x = x1; points[4].y = y1; dc->DrawLines(5, points);
}

/*
 * Higher-level events called by OnEvent
 *
 */

void ShapeCanvas::OnLeftClick(float x, float y, int keys) {}
void ShapeCanvas::OnRightClick(float x, float y, int keys) {}
void ShapeCanvas::OnDragLeft(Bool draw, float x, float y, int keys) {}
void ShapeCanvas::OnBeginDragLeft(float x, float y, int keys) {}
void ShapeCanvas::OnEndDragLeft(float x, float y, int keys) {}
void ShapeCanvas::OnDragRight(Bool draw, float x, float y, int keys) {}
void ShapeCanvas::OnBeginDragRight(float x, float y, int keys) {}
void ShapeCanvas::OnEndDragRight(float x, float y, int keys) {}
