/*
 * File:	bsubwin.cc
 * Purpose:	wxWindows GUI builder - subwindows
 * Author:	Julian Smart
 * Created:	1993
 * Updated:	
 * Copyright:	(c) 1993, AIAI, University of Edinburgh
 */

static const char sccsid[] = "%W% %G%";

#include "wx.h"
#include "wx_form.h"
#include <ctype.h>
#include <stdlib.h>

#include "wxbuild.h"
#include "bapp.h"
#include "namegen.h"
#include "bframe.h"
#include "bsubwin.h"
#include "bgdi.h"

#ifdef wx_x
#define DEFAULT_EDITOR "emacs"
#endif
#ifdef wx_msw
#define DEFAULT_EDITOR "notepad"
#endif

BuildSubwindowData::BuildSubwindowData(BuildFrameData *theParent):
  BuildWindowData(theParent)
{
  percentOfFrame = 100;
  hasBorder = FALSE;
  resizeMode = RESIZE_PROPORTIONAL;
  resizeModeString = copystring("Proportional");
  labelFont = NULL;
  buttonFont = NULL;
}

BuildSubwindowData::~BuildSubwindowData(void)
{
  if (resizeModeString)
    delete[] resizeModeString;
  if (labelFont)
    delete[] labelFont;
  if (buttonFont)
    delete[] buttonFont;
}

Bool BuildSubwindowData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildWindowData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValue("percent_of_frame", &percentOfFrame);
  expr->AssignAttributeValue("border", &hasBorder);
  expr->AssignAttributeValue("resize_mode", &resizeMode);
  expr->AssignAttributeValue("label_font", &labelFont);
  expr->AssignAttributeValue("button_font", &buttonFont);
  return TRUE;
}

Bool BuildSubwindowData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildWindowData::WritePrologAttributes(expr, database);
  expr->AddAttributeValue("percent_of_frame", (long)percentOfFrame);
  expr->AddAttributeValue("border", (long)hasBorder);
  expr->AddAttributeValue("resize_mode", (long)resizeMode);
  expr->AddAttributeValueString("label_font", labelFont);
  expr->AddAttributeValueString("button_font", buttonFont);
  return TRUE;
}

void BuildSubwindowData::FindScreenPosition(int *sx, int *sy)
{
  if (!userWindow)
  {
    *sx = x; *sy = y; return;
  }

#ifdef wx_msw
  if (buildParent)
  {
    userWindow->GetPosition(sx, sy);
    buildParent->userWindow->ClientToScreen(sx, sy);
  }
#endif
#ifdef wx_x
  cout << "ClientToScreen NOT YET IMPLEMENTED IN wxWINDOWS!!!\n";
  cout << "Screen position could be erroneous.\n";
  // Screen position is parent position plus own position.
  int px = 0;
  int py = 0;

  if (buildParent && buildParent->userWindow)
    buildParent->userWindow->GetPosition(&px, &py);

  userWindow->GetPosition(&x, &y); // x and y are member variables  
  *sx = (x + px);
  *sy = (y + py);
#endif
}

void BuildSubwindowData::FindClientPosition(int sx, int sy, int *cx, int *cy)
{
#ifdef wx_msw
  if (buildParent)
  {
    *cx = sx;
    *cy = sy;
    buildParent->userWindow->ScreenToClient(cx, cy);
  }
  x = *cx; y = *cy;
#endif
#ifdef wx_x
  cout << "ScreenToClient NOT YET IMPLEMENTED IN wxWINDOWS!!!\n";
  cout << "Client position could be erroneous.\n";
  // Client position is parent screen position minus own position
  int parentX, parentY;
  buildParent->FindScreenPosition(&parentX, &parentY);

  *cx = parentX - sx; *cy = parentY - sy;
  x = *cx; y = *cy;
#endif
}

void BuildSubwindowData::PositionWindowObject(void)
{
  if (!userWindow || !windowObject)
    return;

  windowObject->Erase();

  int screenX, screenY;
  FindScreenPosition(&screenX, &screenY);

  int uW, uH;
  userWindow->GetSize(&uW, &uH);

  if (!dontResize)
  {
    dontResize = TRUE;
    windowObject->SetSize(uW, uH, TRUE);
    dontResize = FALSE;
  }

  windowObject->xpos = (float)(screenX + (uW/2.0));
  windowObject->ypos = (float)(screenY + (uH/2.0));
  windowObject->Move(windowObject->GetX(), windowObject->GetY());
  wxNode *node = children.First();

  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();
    if (child->windowObject)
      child->PositionWindowObject();
    node = node->Next();
  }
}

// Generation
void BuildSubwindowData::WriteClassImplementation(ostream& stream)
{
}

void BuildSubwindowData::WriteClassDeclaration(ostream& stream)
{
}

void BuildSubwindowData::GenerateWindowStyleString(char *buf)
{
  buf[0] = 0;
  if (windowStyle & wxVSCROLL)
  {
    if (strlen(buf) > 0)
      strcat(buf, " | ");
    strcat(buf, "wxVSCROLL");
  }
  if (windowStyle & wxHSCROLL)
  {
    if (strlen(buf) > 0)
      strcat(buf, " | ");
    strcat(buf, "wxHSCROLL");
  }
  if (windowStyle & wxCAPTION)
  {
    if (strlen(buf) > 0)
      strcat(buf, " | ");
    strcat(buf, "wxCAPTION");
  }
  if (windowStyle & wxABSOLUTE_POSITIONING)
  {
    if (strlen(buf) > 0)
      strcat(buf, " | ");
    strcat(buf, "wxABSOLUTE_POSITIONING");
  }
  if (windowStyle & wxBORDER)
  {
    if (strlen(buf) > 0)
      strcat(buf, " | ");
    strcat(buf, "wxBORDER");
  }
  if (windowStyle & wxRETAINED)
  {
    if (strlen(buf) > 0)
      strcat(buf, " | ");
    strcat(buf, "wxRETAINED");
  }
  if (windowStyle & wxBACKINGSTORE)
  {
    if (strlen(buf) > 0)
      strcat(buf, " | ");
    strcat(buf, "wxBACKINGSTORE");
  }
  if (strlen(buf) == 0)
    strcat(buf, "0");
}

// Add class-specific items to form
void BuildSubwindowData::AddFormItems(wxForm *form)
{
  form->Add(wxMakeFormString("Name", &name, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL,
               300));
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormString("Description", &description, wxFORM_MULTITEXT, NULL, NULL, wxVERTICAL,
               300, 80));
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormString("Class name", &className, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL,
               200));
  if (wxSubType(windowType, wxTYPE_DIALOG_BOX))
  {
    form->Add(wxMakeFormNewLine());
    form->Add(wxMakeFormString("Title", &title, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL,
               300));
    form->Add(wxMakeFormNewLine());
  }
  if (!wxSubType(windowType, wxTYPE_DIALOG_BOX))
  {
    form->Add(wxMakeFormBool("Border", &hasBorder));
    form->Add(wxMakeFormNewLine());
    form->Add(wxMakeFormString("Resize strategy", &resizeModeString, wxFORM_RADIOBOX,
                new wxList(wxMakeConstraintStrings("Fixed", "Grow", "Proportional", NULL), NULL),
              NULL, wxVERTICAL));
    form->Add(wxMakeFormNewLine());
  }
  form->Add(wxMakeFormShort("Width", &width, wxFORM_DEFAULT, NULL, NULL, NULL, 100));
  form->Add(wxMakeFormShort("Height", &height, wxFORM_DEFAULT, NULL, NULL, NULL, 100));
  if (!wxSubType(windowType, wxTYPE_DIALOG_BOX))
    form->Add(wxMakeFormShort("% frame", &percentOfFrame, wxFORM_DEFAULT, NULL, NULL, NULL, 100));
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
void BuildSubwindowData::AddDialogItems(wxDialogBox *dialog)
{
}

/*
 * Panels
 *
 */
 
BuildPanelData::BuildPanelData(BuildFrameData *theParent):
  BuildSubwindowData(theParent)
{
  x = 30;
  y = 30;
  width = 300;
  height = 300;
  name = copystring(GetNewObjectName("panel"));
  memberName = copystring(name);
  className = copystring(GetNewObjectName("PanelClass"));
  title = copystring("untitled");
  windowStyle = wxABSOLUTE_POSITIONING | wxBORDER;
  windowType = wxTYPE_PANEL;

  fitContents = FALSE;
  relativeLayout = FALSE;
  horizLabelPosition = TRUE;
}

BuildPanelData::~BuildPanelData(void)
{
}

Bool BuildPanelData::EditAttributes(void)
{
  char nameBuf[200];
  sprintf(nameBuf, "Panel Properties for %s", name);
  wxDialogBox *dialog = new wxDialogBox(NULL, nameBuf, TRUE, 10, 10);
  dialog->SetLabelFont(SmallLabelFont);
  dialog->SetButtonFont(SmallButtonFont);

  BuildForm *form = new BuildForm("Creating a panel subwindow");

  AddFormItems(form);

  form->AssociatePanel(dialog);
  form->dialog = dialog;
  AddDialogItems(dialog);

  dialog->Fit();
  dialog->Centre(wxBOTH);

  form->RevertValues();

  dialog->Show(TRUE);
  if (strcmp(resizeModeString, "Proportional") == 0)
    resizeMode = RESIZE_PROPORTIONAL;
  else if (strcmp(resizeModeString, "Fixed") == 0)
    resizeMode = RESIZE_FIXED;
  else if (strcmp(resizeModeString, "Grow") == 0)
    resizeMode = RESIZE_GROW;

  BuildFrameData *frame = (BuildFrameData *)buildParent;
  frame->userWindow->OnSize(-1, -1);
  return TRUE;
}

// Add class-specific items to form
void BuildPanelData::AddFormItems(wxForm *form)
{
  BuildSubwindowData::AddFormItems(form);
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormBool("Fit contents", &fitContents));
//  form->Add(wxMakeFormBool("Relative layout", &relativeLayout));
  form->Add(wxMakeFormBool("Horizontal labels", &horizLabelPosition));
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormString("Label font", &labelFont, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL,
               200));
  form->Add(wxMakeFormString("Button font", &buttonFont, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL,
               200));
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
void BuildPanelData::AddDialogItems(wxDialogBox *dialog)
{
}

Bool BuildPanelData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildSubwindowData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValue("fit_contents", &fitContents);
  expr->AssignAttributeValue("relative_layout", &relativeLayout);
  expr->AssignAttributeValue("label_position", &horizLabelPosition);
  return TRUE;
}

Bool BuildPanelData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildSubwindowData::WritePrologAttributes(expr, database);
  expr->AddAttributeValue("fit_contents", (long)fitContents);
  expr->AddAttributeValue("relative_layout", (long)relativeLayout);
  expr->AddAttributeValue("label_position", (long)horizLabelPosition);
  return TRUE;
}

Bool BuildPanelData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  windowStyle = wxABSOLUTE_POSITIONING;
  if (hasBorder)
    windowStyle |= wxBORDER;

  UserPanel *panel = new UserPanel((wxFrame *)buildParent->userWindow, x, y,
                 width, height, windowStyle);
  userWindow = panel;
  panel->buildWindow = this;

  wxFont *bFont = NULL;
  wxFont *lFont = NULL;
  if (buttonFont && (bFont = FindFont(buttonFont)))
  {
    panel->SetButtonFont(bFont);
  }
  if (labelFont && (lFont = FindFont(labelFont)))
  {
    panel->SetLabelFont(lFont);
  }
  
  MakeRealWindowChildren();

  if (fitContents)
    panel->Fit();
  if (horizLabelPosition)
    panel->SetLabelPosition(wxHORIZONTAL);
  else
    panel->SetLabelPosition(wxVERTICAL);

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildPanelData::DestroyRealWindow(void)
{
  userWindow->GetPosition(&x, &y);
  userWindow->GetSize(&width, &height);
  delete userWindow;
  userWindow = NULL;
  return TRUE;
}

void BuildPanelData::WriteClassDeclaration(ostream& stream)
{
  if (description && (strlen(description) > 0))
  {
    stream << "/*\n" << description << "\n*/\n";
  }

  stream << "class " << className << ": public wxPanel\n{\n";
  stream << " private:\n protected:\n public:\n";

  // Want to have as member variables all child subwindows
  if (children.Number() > 0)
  {
    stream << "  // Panel items for reference within the program.\n";
    nameSpace.BeginBlock();
    wxNode *node = children.First();
    while (node)
    {
      BuildWindowData *child = (BuildWindowData *)node->Data();
//      if (child->memberName) delete[] child->memberName;
//      child->memberName = copystring(child->name);

      stream << "  " << child->className << " *" << child->memberName << ";\n";
      node = node->Next();
    }
    nameSpace.EndBlock();
    stream << "\n";
  }

  stream << "  // Constructor and destructor\n";
  stream << "  " << className ;
  stream << "(wxFrame *parent, int x, int y, int width, int height, int style, char *name);\n";
  stream << "  ~" << className << "(void);\n\n";
  stream << " void OnSize(int w, int h);\n";
  stream << "};\n\n";

  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();

    child->WriteClassDeclaration(stream);
    node = node->Next();
  }
}

// Generation
void BuildPanelData::WriteClassImplementation(ostream& stream)
{
  if (description && (strlen(description) > 0))
  {
    stream << "/*\n" << description << "\n*/\n";
  }

  stream << className << "::" << className ;
  stream << "(wxFrame *parent, int x, int y, int width, int height, int style, char *name):\n";
  stream << "  wxPanel(parent, x, y, width, height, style, name)\n{\n";

  nameSpace.BeginBlock();

  // Set the default label orientation
  if (!horizLabelPosition)
  {
    stream << "  SetLabelPosition(wxVERTICAL);\n";
  }

  // Set button and label fonts if they exist
  wxFont *bFont = NULL;
  wxFont *lFont = NULL;
  if (buttonFont && (bFont = FindFont(buttonFont)))
  {
    stream << "  SetButtonFont(theApp." << buttonFont << ");\n";
  }
  if (labelFont && (lFont = FindFont(labelFont)))
  {
    stream << "  SetLabelFont(theApp." << labelFont << ");\n";
  }
  
  // Create all child subwindows
  if (children.Number() > 0)
  {
    stream << "  // Create panel items\n";
    wxNode *node = children.First();
    while (node)
    {
      BuildWindowData *child = (BuildWindowData *)node->Data();

      child->GenerateWindowInitialisationPre(stream, child->memberName, "  ");
      stream << "  " << child->memberName << " = ";
      child->GenerateConstructorCall(stream, "this");
      stream << ";\n";
      child->GenerateWindowInitialisationPost(stream, child->memberName, "  ");

      node = node->Next();
    }
  }

  if (fitContents)
  {
    stream << "  // Fit panel to contents\n";
    stream << "  Fit()\n\n";
  }

  nameSpace.EndBlock();
  stream << "}\n\n";

  stream << className << "::~" << className << "(void)\n{\n}\n\n";

  stream << "void " << className << "::OnSize(int w, int h)\n{\n";
  stream << "  wxPanel::OnSize(w, h);\n";
  stream << "}\n\n";

  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();

    child->WriteClassImplementation(stream);
    node = node->Next();
  }
}

void BuildPanelData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << className << "(" << parentName;
  stream << ", " << x << ", " << y << ", ";
  stream << width << ", " << height << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}

/*
 * Text window
 *
 */
 
BuildTextWindowData::BuildTextWindowData(BuildFrameData *theParent):
  BuildSubwindowData(theParent)
{
  x = 30;
  y = 30;
  width = 300;
  height = 300;
  name = copystring(GetNewObjectName("textwindow"));
  memberName = copystring(name);
  className = copystring(GetNewObjectName("TextWindowClass"));
  title = copystring("untitled");
  windowStyle = wxBORDER;
  windowType = wxTYPE_TEXT_WINDOW;

  defaultFile = NULL;
}

BuildTextWindowData::~BuildTextWindowData(void)
{
}

Bool BuildTextWindowData::EditAttributes(void)
{
  char nameBuf[200];
  sprintf(nameBuf, "Text Window Properties for %s", name);
  wxDialogBox *dialog = new wxDialogBox(NULL, nameBuf, TRUE, 10, 10);
  dialog->SetLabelFont(SmallLabelFont);
  dialog->SetButtonFont(SmallButtonFont);

  BuildForm *form = new BuildForm("Creating a text subwindow");
  AddFormItems(form);

  form->AssociatePanel(dialog);
  form->dialog = dialog;
  AddDialogItems(dialog);

  dialog->Fit();
  dialog->Centre(wxBOTH);

  form->RevertValues();

  dialog->Show(TRUE);
  if (strcmp(resizeModeString, "Proportional") == 0)
    resizeMode = RESIZE_PROPORTIONAL;
  else if (strcmp(resizeModeString, "Fixed") == 0)
    resizeMode = RESIZE_FIXED;
  else if (strcmp(resizeModeString, "Grow") == 0)
    resizeMode = RESIZE_GROW;

  BuildFrameData *frame = (BuildFrameData *)buildParent;
  frame->userWindow->OnSize(-1, -1);
  return TRUE;
}

// Add class-specific items to form
void BuildTextWindowData::AddFormItems(wxForm *form)
{
  BuildSubwindowData::AddFormItems(form);
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormString("Text file", &defaultFile));
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
void BuildTextWindowData::AddDialogItems(wxDialogBox *dialog)
{
}

Bool BuildTextWindowData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildSubwindowData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValue("file", &defaultFile);
  return TRUE;
}

Bool BuildTextWindowData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildSubwindowData::WritePrologAttributes(expr, database);
  expr->AddAttributeValueString("file", defaultFile);
  return TRUE;
}

Bool BuildTextWindowData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  windowStyle = 0;
  if (hasBorder)
    windowStyle |= wxBORDER;

  UserTextWindow *text = new UserTextWindow((wxFrame *)buildParent->userWindow, x, y,
                 width, height, windowStyle);
  userWindow = text;
  text->buildWindow = this;
  MakeRealWindowChildren();

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildTextWindowData::DestroyRealWindow(void)
{
  userWindow->GetPosition(&x, &y);
  userWindow->GetSize(&width, &height);
  delete userWindow;
  return TRUE;
}

void BuildTextWindowData::WriteClassDeclaration(ostream& stream)
{
  if (description && (strlen(description) > 0))
  {
    stream << "/*\n" << description << "\n*/\n";
  }

  stream << "class " << className << ": public wxTextWindow\n{\n";
  stream << " private:\n protected:\n public:\n";


  stream << "  // Constructor and destructor\n";
  stream << "  " << className ;
  stream << "(wxFrame *parent, int x, int y, int width, int height, int style, char *name);\n";
  stream << "  ~" << className << "(void);\n\n";
  stream << " void OnSize(int w, int h);\n";
  stream << "};\n\n";

  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();

    child->WriteClassDeclaration(stream);
    node = node->Next();
  }
}

// Generation
void BuildTextWindowData::WriteClassImplementation(ostream& stream)
{
  if (description && (strlen(description) > 0))
  {
    stream << "/*\n" << description << "\n*/\n";
  }

  stream << className << "::" << className ;
  stream << "(wxFrame *parent, int x, int y, int width, int height, int style, char *name):\n";
  stream << "  wxTextWindow(parent, x, y, width, height, style, name)\n{\n";

  if (defaultFile && strlen(defaultFile) > 0)
  {
    stream << "  // Load a default file\n";
    stream << "  LoadFile(" << "\"" << defaultFile << "\"" << ");\n\n";
  }

  stream << "}\n\n";

  stream << className << "::~" << className << "(void)\n{\n}\n\n";

  stream << "void " << className << "::OnSize(int w, int h)\n{\n";
  stream << "  wxTextWindow::OnSize(w, h);\n";
  stream << "}\n\n";

  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();

    child->WriteClassImplementation(stream);
    node = node->Next();
  }
}

void BuildTextWindowData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << className << "(" << parentName;
  stream << ", " << x << ", " << y << ", ";
  stream << width << ", " << height << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}

/*
 * Canvas
 *
 */
 
BuildCanvasData::BuildCanvasData(BuildFrameData *theParent):
  BuildSubwindowData(theParent)
{
  x = 30;
  y = 30;
  width = 300;
  height = 300;
  name = copystring(GetNewObjectName("canvas"));
  memberName = copystring(name);
  className = copystring(GetNewObjectName("CanvasClass"));
  title = copystring("untitled");
  windowStyle = wxBORDER;
  windowType = wxTYPE_CANVAS;

  simulationBitmap = NULL;

  // Scrollbar
  unitSizeX = 20;
  unitSizeY = 20;
  noUnitsX = 50;
  noUnitsY = 50;
  unitsPerPageX = 10;
  unitsPerPageY = 10;
}

BuildCanvasData::~BuildCanvasData(void)
{
}

Bool BuildCanvasData::EditAttributes(void)
{
  char nameBuf[200];
  sprintf(nameBuf, "Canvas Properties for %s", name);
  wxDialogBox *dialog = new wxDialogBox(NULL, nameBuf, TRUE, 10, 10);
  dialog->SetLabelFont(SmallLabelFont);
  dialog->SetButtonFont(SmallButtonFont);

  BuildForm *form = new BuildForm("Creating a canvas subwindow");
  AddFormItems(form);

  form->AssociatePanel(dialog);
  form->dialog = dialog;
  AddDialogItems(dialog);

  dialog->Fit();
  dialog->Centre(wxBOTH);

  form->RevertValues();

  dialog->Show(TRUE);
  if (strcmp(resizeModeString, "Proportional") == 0)
    resizeMode = RESIZE_PROPORTIONAL;
  else if (strcmp(resizeModeString, "Fixed") == 0)
    resizeMode = RESIZE_FIXED;
  else if (strcmp(resizeModeString, "Grow") == 0)
    resizeMode = RESIZE_GROW;

  BuildFrameData *frame = (BuildFrameData *)buildParent;
  frame->userWindow->OnSize(-1, -1);

  return TRUE;
}

// Add class-specific items to form
void BuildCanvasData::AddFormItems(wxForm *form)
{
  BuildSubwindowData::AddFormItems(form);
  form->Add(wxMakeFormNewLine());
//  form->Add(wxMakeFormString("Bitmap", &simulationBitmap, wxFORM_DEFAULT, NULL, NULL, NULL, 150));
  form->Add(wxMakeFormBool("Retained canvas", &isRetained));
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormShort("Pixels/unit X", &unitSizeX, wxFORM_DEFAULT, NULL, NULL, NULL, 150));
  form->Add(wxMakeFormShort("Pixels/unit Y", &unitSizeY, wxFORM_DEFAULT, NULL, NULL, NULL, 150));
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormShort("No units X", &noUnitsX, wxFORM_DEFAULT, NULL, NULL, NULL, 150));
  form->Add(wxMakeFormShort("No units Y", &noUnitsY, wxFORM_DEFAULT, NULL, NULL, NULL, 150));
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormShort("Units/page X", &unitsPerPageX, wxFORM_DEFAULT, NULL, NULL, NULL, 150));
  form->Add(wxMakeFormShort("Units/page Y", &unitsPerPageY, wxFORM_DEFAULT, NULL, NULL, NULL, 150));
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
void BuildCanvasData::AddDialogItems(wxDialogBox *dialog)
{
}


Bool BuildCanvasData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildSubwindowData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValue("unit_size_x", &unitSizeX);
  expr->AssignAttributeValue("unit_size_y", &unitSizeY);
  expr->AssignAttributeValue("no_units_x", &noUnitsX);
  expr->AssignAttributeValue("no_units_y", &noUnitsY);
  expr->AssignAttributeValue("unit_size_x", &unitSizeX);
  expr->AssignAttributeValue("units_page_x", &unitsPerPageX);
  expr->AssignAttributeValue("units_page_y", &unitsPerPageY);
  expr->AssignAttributeValue("bitmap", &simulationBitmap);

  return TRUE;
}

Bool BuildCanvasData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildSubwindowData::WritePrologAttributes(expr, database);
  expr->AddAttributeValue("unit_size_x", (long)unitSizeX);
  expr->AddAttributeValue("unit_size_y", (long)unitSizeY);
  expr->AddAttributeValue("no_units_x", (long)noUnitsX);
  expr->AddAttributeValue("no_units_y", (long)noUnitsY);
  expr->AddAttributeValue("unit_size_x", (long)unitSizeX);
  expr->AddAttributeValue("units_page_x", (long)unitsPerPageX);
  expr->AddAttributeValue("units_page_y", (long)unitsPerPageY);
  expr->AddAttributeValueString("bitmap", simulationBitmap);
  return TRUE;
}

Bool BuildCanvasData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  windowStyle = 0;
  if (hasBorder)
    windowStyle |= wxBORDER;
  if (isRetained)
    windowStyle |= wxRETAINED;

  UserCanvas *canvas = new UserCanvas((wxFrame *)buildParent->userWindow, x, y,
                 width, height, windowStyle);
  userWindow = canvas;
  canvas->buildWindow = this;
  MakeRealWindowChildren();

  if ((unitSizeX > 0) && (unitSizeY > 0))
    canvas->SetScrollbars(unitSizeX, unitSizeY, noUnitsX, noUnitsY,
                          unitsPerPageX, unitsPerPageY);

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildCanvasData::DestroyRealWindow(void)
{
  userWindow->GetPosition(&x, &y);
  userWindow->GetSize(&width, &height);
  delete userWindow;
  return TRUE;
}

void BuildCanvasData::WriteClassDeclaration(ostream& stream)
{
  if (description && (strlen(description) > 0))
  {
    stream << "/*\n" << description << "\n*/\n";
  }

  stream << "class " << className << ": public wxCanvas\n{\n";
  stream << " private:\n protected:\n public:\n";

  stream << "  // Constructor and destructor\n";
  stream << "  " << className ;
  stream << "(wxFrame *parent, int x, int y, int width, int height, int style, char *name);\n";
  stream << "  ~" << className << "(void);\n\n";
  stream << " void OnSize(int w, int h);\n";
  stream << " void OnPaint(void);\n";
  stream << " void OnEvent(wxMouseEvent& event);\n";
  stream << " void OnChar(wxKeyEvent& event);\n";
  stream << "};\n\n";

  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();

    child->WriteClassDeclaration(stream);
    node = node->Next();
  }
}

// Generation
void BuildCanvasData::WriteClassImplementation(ostream& stream)
{
  if (description && (strlen(description) > 0))
  {
    stream << "/*\n" << description << "\n*/\n";
  }

  stream << className << "::" << className ;
  stream << "(wxFrame *parent, int x, int y, int width, int height, int style, char *name):\n";
  stream << "  wxCanvas(parent, x, y, width, height, style, name)\n{\n";

  if (unitSizeX > 0 && unitSizeY > 0 && noUnitsX > 0 && noUnitsY > 0)
  {
    stream << "  /* Set scrollbars to:\n";
    stream << "   *   " << unitSizeX << " pixels per horizontal units (1 unit = 1 scroll line)\n";
    stream << "   *   " << unitSizeY << " pixels per vertical units\n";
    stream << "   *   " << noUnitsX << " horizontal units\n";
    stream << "   *   " << noUnitsY << " vertical units\n";
    stream << "   *   " << unitsPerPageX << " units per horizontal page\n";
    stream << "   *   " << unitsPerPageY << " units per vertical page.\n";
    stream << "   * Virtual canvas is therefore " << unitSizeX*noUnitsX << " pixels by ";
    stream << unitSizeY*noUnitsY << " pixels.\n";
    stream << "   */\n";
    stream << "  SetScrollbars(" << unitSizeX << ", " << unitSizeY << ", ";
    stream << noUnitsX << ", " << noUnitsY << ", ";
    stream << unitsPerPageX << ", " << unitsPerPageY << ");\n";
  }

  stream << "}\n\n";

  stream << className << "::~" << className << "(void)\n{\n}\n\n";

  stream << "// Called when canvas is resized.\n";
  stream << "void " << className << "::OnSize(int w, int h)\n{\n";
  stream << "  wxCanvas::OnSize(w, h);\n";
  stream << "}\n\n";

  stream << "// Called when canvas needs to be repainted.\n";
  stream << "void " << className << "::OnPaint(void)\n{\n";
  stream << "  // Speeds up drawing under Windows.\n";
  stream << "  GetDC()->BeginDrawing();\n\n";
  stream << "  // Insert your drawing code here.\n";
  stream << "\n";
  stream << "  GetDC()->EndDrawing();\n";
  stream << "}\n\n";

  stream << "// Called when the canvas receives a mouse event.\n";
  stream << "void " << className << "::OnEvent(wxMouseEvent& event)\n{\n";
  stream << "  wxCanvas::OnEvent(event);\n";
  stream << "}\n\n";

  stream << "// Called when the canvas receives a key event.\n";
  stream << "void " << className << "::OnChar(wxKeyEvent& event)\n{\n";
  stream << "  wxCanvas::OnChar(event);\n";
  stream << "}\n\n";

  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();

    child->WriteClassImplementation(stream);
    node = node->Next();
  }
}

void BuildCanvasData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << className << "(" << parentName;
  stream << ", " << x << ", " << y << ", ";
  stream << width << ", " << height << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}

/*
 * USER WINDOWS
 *
 */

/*
 * User panel
 *
 */
 
void UserPanel::OnSize(int w, int h)
{
  wxPanel::OnSize(w, h);
}

/*
 * User Text subwindow
 *
 */
 
void UserTextWindow::OnSize(int w, int h)
{
  wxTextWindow::OnSize(w, h);
}

/*
 * User canvas
 *
 */
 
void UserCanvas::OnSize(int w, int h)
{
  wxCanvas::OnSize(w, h);
}

void UserCanvas::OnChar(wxKeyEvent& event)
{
  wxCanvas::OnChar(event);
}

void UserCanvas::OnEvent(wxMouseEvent& event)
{
  wxCanvas::OnEvent(event);
}

void UserCanvas::OnPaint(void)
{
  wxCanvas::OnPaint();
}


/*
 * Dialog boxes
 *
 */
 
BuildDialogBoxData::BuildDialogBoxData(BuildFrameData *theParent):
  BuildPanelData(theParent)
{
  x = 30;
  y = 30;
  width = 300;
  height = 300;
  name = copystring(GetNewObjectName("dialogbox"));
  memberName = copystring(name);
  className = copystring(GetNewObjectName("DialogBoxClass"));
  title = copystring("untitled");
  windowStyle = wxABSOLUTE_POSITIONING;
  windowType = wxTYPE_DIALOG_BOX;

  modal = FALSE;
}

BuildDialogBoxData::~BuildDialogBoxData(void)
{
}

Bool BuildDialogBoxData::EditAttributes(void)
{
  char nameBuf[200];
  sprintf(nameBuf, "Dialog Box Properties for %s", name);
  wxDialogBox *dialog = new wxDialogBox(NULL, nameBuf, TRUE, 10, 10);
  dialog->SetLabelFont(SmallLabelFont);
  dialog->SetButtonFont(SmallButtonFont);

  BuildForm *form = new BuildForm("Creating a dialog box");
  AddFormItems(form);

  form->AssociatePanel(dialog);
  form->dialog = dialog;
  AddDialogItems(dialog);

  dialog->Fit();
  dialog->Centre(wxBOTH);

  form->RevertValues();

  dialog->Show(TRUE);
  return TRUE;
}

// Add class-specific items to form
void BuildDialogBoxData::AddFormItems(wxForm *form)
{
  BuildPanelData::AddFormItems(form);
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormBool("Modal", &modal));
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
void BuildDialogBoxData::AddDialogItems(wxDialogBox *dialog)
{
}

Bool BuildDialogBoxData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildPanelData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValue("modal", &modal);
  return TRUE;
}

Bool BuildDialogBoxData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildPanelData::WritePrologAttributes(expr, database);
  expr->AddAttributeValue("modal", (long)modal);
  return TRUE;
}

Bool BuildDialogBoxData::MakeRealWindow(void)
{
  windowStyle = wxABSOLUTE_POSITIONING;

  UserDialogBox *dialog = new UserDialogBox(NULL, title, FALSE, x, y,
                 width, height, windowStyle);
  userWindow = dialog;
  dialog->buildWindow = this;

  if (horizLabelPosition)
    dialog->SetLabelPosition(wxHORIZONTAL);
  else
    dialog->SetLabelPosition(wxVERTICAL);

  wxFont *bFont = NULL;
  wxFont *lFont = NULL;
  if (buttonFont && (bFont = FindFont(buttonFont)))
  {
    dialog->SetButtonFont(bFont);
  }
  if (labelFont && (lFont = FindFont(labelFont)))
  {
    dialog->SetLabelFont(lFont);
  }

  MakeRealWindowChildren();

  if (fitContents && (children.Number() > 0))
    dialog->Fit();

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildDialogBoxData::DestroyRealWindow(void)
{
  userWindow->GetPosition(&x, &y);
  userWindow->GetSize(&width, &height);
  delete userWindow;
  userWindow = NULL;
  return TRUE;
}

void BuildDialogBoxData::WriteClassDeclaration(ostream& stream)
{
  if (description && (strlen(description) > 0))
  {
    stream << "/*\n" << description << "\n*/\n";
  }

  stream << "class " << className << ": public wxDialogBox\n{\n";
  stream << " private:\n protected:\n public:\n";

  // Want to have as member variables all child items
  if (children.Number() > 0)
  {
    stream << "  // Panel items for reference within the program.\n";
    nameSpace.BeginBlock();
    wxNode *node = children.First();
    while (node)
    {
      BuildWindowData *child = (BuildWindowData *)node->Data();
//      if (child->memberName) delete[] child->memberName;
//      child->memberName = copystring(child->name);

      stream << "  " << child->className << " *" << child->memberName << ";\n";
      node = node->Next();
    }
    nameSpace.EndBlock();
    stream << "\n";
  }

  stream << "  // Constructor and destructor\n";
  stream << "  " << className ;
  stream << "(wxFrame *parent, char *title, Bool modal, int x, int y, int width, int height, int style, char *name);\n";
  stream << "  ~" << className << "(void);\n\n";
  stream << " void OnSize(int w, int h);\n";
  stream << "};\n\n";

  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();

    child->WriteClassDeclaration(stream);
    node = node->Next();
  }
}

// Generation
void BuildDialogBoxData::WriteClassImplementation(ostream& stream)
{
  if (description && (strlen(description) > 0))
  {
    stream << "/*\n" << description << "\n*/\n";
  }

  stream << className << "::" << className ;
  stream << "(wxFrame *parent, char *title, Bool modal, int x, int y, int width, int height, int style, char *name):\n";
  stream << "  wxDialogBox(parent, title, modal, x, y, width, height, style, name)\n{\n";

  nameSpace.BeginBlock();

  // Set the default label orientation
  if (!horizLabelPosition)
  {
    stream << "  SetLabelPosition(wxVERTICAL);\n";
  }

  // Set button and label fonts if they exist
  wxFont *bFont = NULL;
  wxFont *lFont = NULL;
  if (buttonFont && (bFont = FindFont(buttonFont)))
  {
    stream << "  SetButtonFont(theApp." << buttonFont << ");\n";
  }
  if (labelFont && (lFont = FindFont(labelFont)))
  {
    stream << "  SetLabelFont(theApp." << labelFont << ");\n";
  }
  
  // Create all child subwindows
  if (children.Number() > 0)
  {
    stream << "  // Create panel items\n";
    wxNode *node = children.First();
    while (node)
    {
      BuildWindowData *child = (BuildWindowData *)node->Data();

      child->GenerateWindowInitialisationPre(stream, child->memberName, "  ");

      stream << "  " << child->memberName << " = ";
      child->GenerateConstructorCall(stream, "this");
      stream << ";\n";

      child->GenerateWindowInitialisationPost(stream, child->memberName, "  ");

      node = node->Next();
    }
    stream << "\n";
  }

  if (fitContents)
  {
    stream << "  // Fit panel to contents\n";
    stream << "  Fit()\n\n";
  }

  nameSpace.EndBlock();
  stream << "\n}\n\n";

  stream << className << "::~" << className << "(void)\n{\n}\n\n";

  stream << "void " << className << "::OnSize(int w, int h)\n{\n";
  stream << "  wxDialogBox::OnSize(w, h);\n";
  stream << "}\n\n";

  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();

    child->WriteClassImplementation(stream);
    node = node->Next();
  }
}

void BuildDialogBoxData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << className << "(" << parentName << ", " << "\"" << title << "\"";
  stream << ", " << (modal ? "TRUE" : "FALSE") << ", " << x << ", " << y << ", ";
  stream << width << ", " << height << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}


void BuildDialogBoxData::FindScreenPosition(int *sx, int *sy)
{
  if (!userWindow)
  {
    *sx = x; *sy = y; return;
  }

  userWindow->GetPosition(&x, &y); // x and y are member variables  
  *sx = x;
  *sy = y;
}

void BuildDialogBoxData::FindClientPosition(int sx, int sy, int *cx, int *cy)
{
  *cx = x; *cy = y;
  x = *cx; y = *cy;
}

void BuildDialogBoxData::PositionWindowObject(void)
{
  if (!userWindow || !windowObject)
    return;

  windowObject->Erase();

  int screenX, screenY, uW, uH;
  FindScreenPosition(&screenX, &screenY);
  userWindow->GetSize(&uW, &uH);
  windowObject->SetSize(uW, uH, TRUE);

  windowObject->xpos = (float)(screenX + (uW/2.0));
  windowObject->ypos = (float)(screenY + (uH/2.0));
  windowObject->Move(windowObject->GetX(), windowObject->GetY());
  wxNode *node = children.First();

  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();
    if (child->windowObject)
      child->PositionWindowObject();

    node = node->Next();
  }
  MainFrame->canvas->Redraw();
}

void UserDialogBox::OnSize(int w, int h)
{
  wxDialogBox::OnSize(w, h);
}
