/*
 * File:     tex2any.h
 * Purpose:  Header file for LaTeX --> wxHelp conversion
 *
 *                       wxWindows 1.50
 * Copyright (c) 1993 Artificial Intelligence Applications Institute,
 *                   The University of Edinburgh
 *
 *                     Author: Julian Smart
 *                        Date: 7-9-93
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice, author statement and this permission
 * notice appear in all copies of this software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, EXPRESS,
 * IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE ARTIFICIAL INTELLIGENCE APPLICATIONS INSTITUTE OR THE
 * UNIVERSITY OF EDINBURGH BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF
 * DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include "wx.h"
#include "wx_utils.h"
#include "wx_list.h"
#include "wx_hash.h"
#include "wxhlpblk.h"

/*
 * Conversion modes
 *
 */
 
#define TEX_RTF  1
#define TEX_XLP  2
#define TEX_HTML 3

/*
 * We have a list of macro definitions which we must define
 * in advance to enable the parsing to recognize macros.
 */

#define FORBID_OK         0
#define FORBID_WARN       1
#define FORBID_ABSOLUTELY 2

class TexMacroDef: public wxObject
{
 public:
  int no_args;
  char *name;
  Bool ignore;
  int forbidden;
  int macroId;

  TexMacroDef(int the_id, char *the_name, int n, Bool ig, Bool forbidLevel = FORBID_OK);
  ~TexMacroDef(void);
};

#define CHUNK_TYPE_MACRO    1
#define CHUNK_TYPE_ARG      2
#define CHUNK_TYPE_STRING   3

/*
 We have nested lists to represent the Tex document.
 Each element of a list of chunks can be one of:
  - a plain string
  - a macro with/without arguments. Arguments are lists of TexChunks.

Example (\toplevel is implicit but made explicit here):

AddMacroDef(ltMYMAT, "mymat", 2);

\toplevel{The cat sat on the \mymat{very coarse and {\it cheap}}{mat}}.

Parsed as:

TexChunk: type = macro, name = toplevel, no_args = 1
  Children:

    TexChunk: type = argument

      Children:
        TexChunk: type = string, value = "The cat sat on the "
        TexChunk: type = macro, name = mymat, no_args = 2

          Children:
            TexChunk: type = argument

              Children:
                TexChunk: type = string, value = "very coarse and "
                TexChunk: type = macro, name = it, no_args = 1

                  Children:
                    TexChunk: type = argument

                      Children:
                        TexChunk: type = string, value = "cheap"

            TexChunk: type = argument

              Children:
                TexChunk: type = string, value = mat
 */

class TexChunk
{
 public:
  int type;
//  char *name;
  TexMacroDef *def;
  char *value;
  int macroId;
  int no_args;
  int argn;
  Bool optional;      // Is an optional argument

  wxList children;
  TexChunk(int the_type, TexMacroDef *the_def = NULL);
  TexChunk(TexChunk& toCopy);
  virtual ~TexChunk(void);
};

extern TexChunk     *TopLevel;
extern wxHashTable  MacroDefs;
extern wxStringList IgnorableInputFiles; // Ignorable \input files, e.g. psbox.tex

Bool read_a_line(char *buf);
Bool TexLoadFile(char *filename);
int ParseArg(TexChunk *thisArg, wxList& children, char *buffer, int pos,
           char *environment = NULL, Bool parseArgToBrace = TRUE, TexChunk *customMacroArgs = NULL);
int ParseMacroBody(char *macro_name, TexChunk *parent, int no_args,
           char *buffer, int pos, char *environment = NULL, Bool parseArgToBrace = TRUE, TexChunk *customMacroArgs = NULL);
void TraverseDocument(void);
void TraverseFromChunk(TexChunk *chunk, wxNode *thisNode = NULL, Bool childrenOnly = FALSE);
#define TraverseChildrenFromChunk(arg) TraverseFromChunk(arg, NULL, TRUE)
void SetCurrentOutput(FILE *fd);
void SetCurrentOutputs(FILE *fd1, FILE *fd2);
extern FILE *CurrentOutput1;
extern FILE *CurrentOutput2;
void AddMacroDef(int the_id, char *name, int n, Bool ignore = FALSE, Bool forbidden = FALSE);
void TexInitialize(int bufSize);
void TexCleanUp(void);
void TexOutput(char *s, Bool ordinaryText = FALSE);
char *GetArgData(TexChunk *chunk);
char *GetArgData(void);             // Get the string for the current argument
int GetNoArgs(void);                // Get the number of arguments for the current macro
TexChunk *GetArgChunk(void);        // Get the chunk for the current argument
TexChunk *GetTopLevelChunk(void);   // Get the chunk for the top level
TexChunk *GetNextChunk(void);       // Look ahead to the next chunk
Bool IsArgOptional(void);           // Is this argument an optional argument?
void DefineDefaultMacros(void);     // Optional set of default macros
int GetCurrentColumn(void);         // number of characters on current line
extern wxPathList TexPathList;      // Path list, can be used for file searching.

// Major document styles
#define LATEX_REPORT    1
#define LATEX_ARTICLE   2
#define LATEX_LETTER    3
#define LATEX_BOOK      4
#define LATEX_SLIDES    5

extern TexChunk *DocumentTitle;
extern TexChunk *DocumentAuthor;
extern TexChunk *DocumentDate;
extern int DocumentStyle;
extern int MinorDocumentStyle;
extern char *BibliographyStyleString;
extern char *DocumentStyleString;
extern char *MinorDocumentStyleString;

extern int normalFont;
extern int smallFont;
extern int tinyFont;
extern int largeFont1;
extern int LargeFont2;
extern int LARGEFont3;
extern int hugeFont1;
extern int HugeFont2;
extern int HUGEFont3;

/*
 * USER-ADJUSTABLE SETTINGS
 *
 */

// Section font sizes
extern int chapterFont;
extern int sectionFont;
extern int subsectionFont;
extern int titleFont;
extern int authorFont;
extern Bool winHelp;  // Output in Windows Help format if TRUE, linear otherwise
extern Bool isInteractive;
extern Bool runTwice;
extern Bool convertMode;
extern Bool stopRunning;
extern int  mirrorMargins;
extern Bool headerRule;
extern Bool footerRule;
extern int labelIndentTab;  // From left indent to item label (points)
extern int itemIndentTab;   // From left indent to item (points)
extern Bool useUpButton;
extern Bool htmlBrowseButtons;
extern Bool useHeadingStyles; // Insert \s1, s2 etc.
extern Bool insertTOC; // Insert table of contents
extern Bool insertIndex; // Insert index

/*
 * HTML button identifiers: what kind of browse buttons
 * are placed in HTML files, if any.
 *
 */

#define HTML_BUTTONS_NONE       0
#define HTML_BUTTONS_BITMAP     1
#define HTML_BUTTONS_TEXT       2

/*
 * Section numbering
 *
 */

extern int chapterNo;
extern int sectionNo;
extern int subsectionNo;
extern int subsubsectionNo;
extern int figureNo;

extern int ParSkip;
extern int ParIndent;

extern Bool isSync;

// Set by client and by Tex2Any
extern TexChunk *currentSection;

// Header/footers/pagestyle
extern TexChunk *      LeftHeaderOdd;
extern TexChunk *      LeftFooterOdd;
extern TexChunk *      CentreHeaderOdd;
extern TexChunk *      CentreFooterOdd;
extern TexChunk *      RightHeaderOdd;
extern TexChunk *      RightFooterOdd;
extern TexChunk *      LeftHeaderEven;
extern TexChunk *      LeftFooterEven;
extern TexChunk *      CentreHeaderEven;
extern TexChunk *      CentreFooterEven;
extern TexChunk *      RightHeaderEven;
extern TexChunk *      RightFooterEven;
extern char *          PageStyle;

// Repeat the currentSection, either real (Chapter) or simulated (References)
extern void OutputCurrentSection(void);

// Called by Tex2Any to simulate a section
extern void FakeCurrentSection(char *fakeSection);

/*
 * Local to Tex2Any library
 *
 */
 
extern char *currentArgData;
extern Bool haveArgData; // If TRUE, we're simulating the data.
void StartSimulateArgument(char *data);
void EndSimulateArgument(void);

/*
 * Client-defined
 *
 */

// Called on start/end of macro examination
void OnMacro(int macroId, int no_args, Bool start);

// Called on start/end of argument examination.
// Return TRUE at the start of an argument to traverse
// (output) the argument.
Bool OnArgument(int macroId, int arg_no, Bool start);

// Default: library-defined
void DefaultOnMacro(int macroId, int no_args, Bool start);

// Default: library-defined
Bool DefaultOnArgument(int macroId, int arg_no, Bool start);

// Called on error
void OnError(char *msg);

// Called for information
void OnInform(char *msg);

/*
 * Useful utilities
 *
 */

// Look for \label macro, use this ref name if found or
// make up a topic name otherwise.
char *FindTopicName(TexChunk *chunk);
void ResetTopicCounter(void);

// Parse unit eg. 14, 12pt, 34cm and return value in points.
int ParseUnitArgument(char *unitArg);

// Set small, large, normal etc. point sizes for reference size
void SetFontSizes(int pointSize);

/*
 * Strip off any extension (dot something) from end of file,
 * IF one exists. Inserts zero into buffer.
 *
 */
 
void StripExtension(char *buffer);

/*
 * Reference structure
 *
 */

class TexRef: public wxObject
{
 public:
  char *refLabel;      // Reference label
  char *refFile;       // Reference filename (can be NULL)
  char *sectionNumber; // Section or figure number (as a string)
  TexRef(char *label, char *file, char *section)
  {
    refLabel = copystring(label);
    refFile = file ? copystring(file) : NULL;
    sectionNumber = section ? copystring(section) : copystring("??");
  }
};

extern wxHashTable TexReferences;

/*
 * Add a reference
 *
 */
 
void AddTexRef(char *name, char *file = NULL, char *sectionName = NULL,
         int chapter = 0, int section = 0, int subsection = 0, int subsubsection = 0);

/*
 * Read and write reference file (.ref), to resolve refs for second pass.
 *
 */
void WriteTexReferences(char *filename);
void ReadTexReferences(char *filename);

/*
 * Bibliography stuff
 *
 */

class BibEntry: public wxObject
{
 public:
  char *key;

  /*
   * book, inbook, article, phdthesis, inproceedings, techreport
   */
  char *type;

  /*
   * Possible fields
   *
   */
  char *editor;
  char *title;
  char *booktitle;
  char *author;
  char *journal;
  char *volume;
  char *number;
  char *year;
  char *month;
  char *pages;
  char *chapter;
  char *publisher;
  char *address;
  char *institution;
  char *organization;
  char *comment;

  inline BibEntry(void)
  {
    key = NULL;
    type = NULL;
    editor = NULL;
    title = NULL;
    booktitle = NULL;
    author = NULL;
    journal = NULL;
    volume = NULL;
    number = NULL;
    chapter = NULL;
    year = NULL;
    month = NULL;
    pages = NULL;
    publisher = NULL;
    address = NULL;
    institution = NULL;
    organization = NULL;
    comment = NULL;
  }
};

extern wxList BibList;
extern wxStringList CitationList;

Bool ReadBib(char *filename);
void OutputBib(void);
void ResolveBibReferences(void);
void AddCitation(char *citeKey);
TexRef *FindReference(char *key);

/*
 * Ability to customize, or at least suppress unknown macro errors
 *
 */

extern wxList CustomMacroList;

#define CUSTOM_MACRO_IGNORE 0
#define CUSTOM_MACRO_OUTPUT 1
#define CUSTOM_MACRO_MARK   2

class CustomMacro: public wxObject
{
 public:
  char *macroName;
  char *macroBody;
  int noArgs;
  inline CustomMacro(char *name, int args, char *body)
  {
    noArgs = args;
    macroName = copystring(name);
    if (body)
      macroBody = copystring(body);
    else
      macroBody = NULL;
  }
};

Bool ReadCustomMacros(char *filename);
void ShowCustomMacros(void);
CustomMacro *FindCustomMacro(char *name);
char *ParseMultifieldString(char *s, int *pos);

#define ltABSTRACT          1
#define ltADDCONTENTSLINE   2
#define ltADDTOCOUNTER      3
#define ltALPH1             4
#define ltALPH2             5
#define ltAPPENDIX          6
#define ltARABIC            7
#define ltARRAY             8
#define ltAUTHOR            9

#define ltBACKSLASH         30
#define ltBASELINESKIP      31
#define ltBF                32
#define ltBIBITEM           33
#define ltBIBLIOGRAPHYSTYLE 34
#define ltBIBLIOGRAPHY      35
#define ltBOXIT             36

#define ltCAPTIONSTAR       50
#define ltCAPTION           51
#define ltCDOTS             52
#define ltCENTERLINE        53
#define ltCENTERING         54
#define ltCENTER            55
#define ltCEXTRACT          56
#define ltCHAPTERHEADING    57
#define ltCHAPTERSTAR       58
#define ltCHAPTER           59
#define ltCINSERT           60
#define ltCITE              61
#define ltCLASS             62
#define ltCLEARDOUBLEPAGE   63
#define ltCLEARPAGE         64
#define ltCLINE             65
#define ltCLIPSFUNC         66
#define ltCOLUMNSEP         67
#define ltCOMMENT           68
#define ltCOPYRIGHT         69
#define ltCPARAM            70

#define ltCHEAD             71
#define ltCFOOT             72

#define ltDATE              90
#define ltDESCRIPTION       91
#define ltDESTRUCT          92
#define ltDOCUMENTSTYLE     93
#define ltDOCUMENT          94
#define ltDOUBLESPACE       95

#define ltEM                120
#define ltENUMERATE         121
#define ltEQUATION          122
#define ltEVENSIDEMARGIN    123

#define ltFBOX              150
#define ltFIGURE            151
#define ltFLUSHLEFT         152
#define ltFLUSHRIGHT        153
#define ltFOOTHEIGHT        154
#define ltFOOTNOTE          155
#define ltFOOTSKIP          156
#define ltFRAMEBOX          157
#define ltFUNCTIONSECTION   158
#define ltFUNC              159
#define ltFIGURESTAR        160
#define ltFOOTNOTESIZE      161
#define ltFOOTNOTEPOPUP     162
#define ltFANCYPLAIN        163

#define ltGLOSSARY          180
#define ltGLOSS             181

#define ltHEADHEIGHT        200
#define ltHELPGLOSSARY      201
#define ltHELPIGNORE        202
#define ltHELPONLY          203
#define ltHELPINPUT         204
#define ltHELPFONTFAMILY    205
#define ltHELPFONTSIZE      206
#define ltHELPREFN          207
#define ltHELPREF           208
#define ltHFILL             209
#define ltHLINE             210
#define ltHRULE             211
#define ltHSPACESTAR        212
#define ltHSPACE            213
#define ltHSKIPSTAR         214
#define ltHSKIP             215
#define lthuge              216
#define ltHuge              217
#define ltHUGE              218
#define ltHTMLIGNORE        219
#define ltHTMLONLY          220

#define ltINCLUDEONLY       240
#define ltINCLUDE           241
#define ltINDEX             242
#define ltINPUT             243
#define ltITEMIZE           244
#define ltITEM              245
#define ltIMAGE             246
#define ltIT                247
#define ltITEMSEP           248
#define ltINDENTED          249

#define ltKILL              260

#define ltLABEL             280
#define ltlarge             281
#define ltLarge             282
#define ltLARGE             283
#define ltLATEX             284
#define ltLBOX              285
#define ltLDOTS             286
#define ltLINEBREAK         287
#define ltLISTOFFIGURES     288
#define ltLISTOFTABLES      289
#define ltLHEAD             290
#define ltLFOOT             291
#define ltLATEXIGNORE       292
#define ltLATEXONLY         293

#define ltMAKEGLOSSARY      300
#define ltMAKEINDEX         301
#define ltMAKETITLE         302
#define ltMARKRIGHT         303
#define ltMARKBOTH          304
#define ltMARGINPARWIDTH    305
#define ltMARGINPAR         306
#define ltMARGINPARODD      307
#define ltMARGINPAREVEN     308
#define ltMBOX              309
#define ltMEMBERSECTION     310
#define ltMEMBER            311
#define ltMULTICOLUMN       312
#define ltMARGINPARSEP      313

#define ltNEWCOUNTER        330
#define ltNEWLINE           331
#define ltNEWPAGE           332
#define ltNOCITE            333
#define ltNOINDENT          334
#define ltNOLINEBREAK       335
#define ltNOPAGEBREAK       336
#define ltNORMALSIZE        337
#define ltNORMALBOX         338
#define ltNORMALBOXD        339
#define ltNUMBEREDBIBITEM   340

#define ltONECOLUMN         360
#define ltODDSIDEMARGIN     361

#define ltPAGEBREAK         380
#define ltPAGEREF           381
#define ltPAGESTYLE         382
#define ltPAGENUMBERING     383
#define ltPARAGRAPHSTAR     384
#define ltPARAGRAPH         385
#define ltPARAM             386
#define ltPARINDENT         387
#define ltPARSKIP           388
#define ltPARTSTAR          389
#define ltPART              390
#define ltPAR               391
#define ltPFUNC             392
#define ltPICTURE           393
#define ltPOPREF            394
#define ltPOUNDS            395
#define ltPRINTINDEX        396
#define ltPSBOXTO           397
#define ltPSBOX             398

#define ltQUOTE             420
#define ltQUOTATION         421

#define ltRAGGEDBOTTOM      440
#define ltRAGGEDLEFT        441
#define ltRAGGEDRIGHT       442
#define ltREF               443
#define ltRM                444
#define ltROMAN             445
#define ltROMAN2            446
#define ltRTFSP             447
#define ltRULE              448
#define ltRULEDROW          449
#define ltDRULED            450
#define ltRHEAD             451
#define ltRFOOT             452
#define ltROW               453
#define ltRTFIGNORE         454
#define ltRTFONLY           455

#define ltSC                470
#define ltSECTIONHEADING    471
#define ltSECTIONSTAR       472
#define ltSECTION           473
#define ltSETCOUNTER        474
#define ltSF                475
#define ltSHORTCITE         476
#define ltSINGLESPACE       477
#define ltSLOPPYPAR         478
#define ltSLOPPY            479
#define ltSL                480
#define ltSMALL             481
#define ltSUBITEM           482
#define ltSUBPARAGRAPHSTAR  483
#define ltSUBPARAGRAPH      484
#define ltSPECIAL           485
#define ltSUBSECTIONSTAR    486
#define ltSUBSECTION        487
#define ltSUBSUBSECTIONSTAR 488
#define ltSUBSUBSECTION     489
#define ltSCRIPTSIZE        490
#define ltSETHEADER         491
#define ltSETFOOTER         492
#define ltSIZEDBOX          493
#define ltSIZEDBOXD         494

#define ltTABBING           510
#define ltTABLEOFCONTENTS   511
#define ltTABLE             512
#define ltTABULAR           513
#define ltTAB               514
#define ltTEX               515
#define ltTEXTWIDTH         516
#define ltTEXTHEIGHT        517
#define ltTHEBIBLIOGRAPHY   518
#define ltTITLEPAGE         519
#define ltTITLE             520
#define ltTINY              521
#define ltTODAY             522
#define ltTOPMARGIN         523
#define ltTOPSKIP           524
#define ltTT                525
#define ltTYPEIN            526
#define ltTYPEOUT           527
#define ltTWOCOLUMN         528
#define ltTHEPAGE           529
#define ltTHECHAPTER        530
#define ltTHESECTION        531
#define ltTHISPAGESTYLE     532

#define ltTWOCOLWIDTHA      533
#define ltTWOCOLWIDTHB      534
#define ltTWOCOLSPACING     535
#define ltTWOCOLITEM        536
#define ltTWOCOLITEMRULED   537
#define ltTWOCOLLIST        538

#define ltUNDERLINE         550
  
#define ltVDOTS             570
#define ltVERBATIMINPUT     571
#define ltVERBATIM          572
#define ltVERB              573
#define ltVERSE             574
#define ltVFILL             575
#define ltVLINE             576
#define ltVOID              577
#define ltVRULE             578
#define ltVSPACESTAR        579
#define ltVSKIPSTAR         580
#define ltVSPACE            581
#define ltVSKIP             582
#define ltVERBSTAR          583

#define ltWXCLIPS           600
#define ltWINHELPIGNORE     601
#define ltWINHELPONLY       602

#define ltXLPIGNORE         603
#define ltXLPONLY           604

#define ltSPACE             620
#define ltBACKSLASHCHAR     621
#define ltPIPE              622
#define ltFORWARDSLASH      623
#define ltUNDERSCORE        624
#define ltAMPERSAND         625
#define ltPERCENT           626
#define ltDOLLAR            627
#define ltHASH              628
#define ltLPARENTH          629
#define ltRPARENTH          630
#define ltLBRACE            631
#define ltRBRACE            632
#define ltEQUALS            633
#define ltRANGLEBRA         634
#define ltLANGLEBRA         635
#define ltPLUS              636
#define ltDASH              637
#define ltSINGLEQUOTE       638
#define ltBACKQUOTE         639
#define ltTILDE             640

// Characters, not macros but with special Latex significance
#define ltSPECIALDOLLAR     660
#define ltSPECIALDOUBLEDOLLAR 661
#define ltSPECIALTILDE      662
#define ltSPECIALHASH       663
#define ltSPECIALAMPERSAND  664
#define ltSUPERTABULAR      665

// Accents
#define ltACCENT_GRAVE      700
#define ltACCENT_ACUTE      701
#define ltACCENT_CARET      702
#define ltACCENT_UMLAUT     703
#define ltACCENT_TILDE      704
#define ltACCENT_DOT        705
#define ltACCENT_CADILLA    706

// Pseudo-macros
#define ltTOPLEVEL          801
#define ltCUSTOM_MACRO      802
#define ltSOLO_BLOCK        803


