/*
 * rtfutils.cc
 *
 * Utility functions for helping convert Latex files
 * into RTF files suitable for compiling into WinHelp
 * files.
 *
 * Julian Smart September 1993
 *
 */

#include "wx.h"
#include "tex2any.h"
#include "tex2rtf.h"
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include "bmputils.h"

wxList itemizeStack;
static int indentLevel = 0;
static int forbidParindent = 0; // if > 0, no parindent (e.g. in center environment)

static char *contentsLineSection = NULL;
static char *contentsLineValue = NULL;
static TexChunk *descriptionItemArg = NULL;
static wxStringList environmentStack; // Stack of paragraph styles we need to remember
static int footnoteCount = 0;
static int citeCount = 1;
extern char *FileRoot;
extern Bool winHelp;
extern Bool startedSections;
extern FILE *Contents;
extern FILE *Chapters;
extern FILE *Popups;
extern char *RTFCharset;
// This is defined in the Tex2Any library and isn't in use after parsing
extern char *BigBuffer;
// Are we in verbatim mode? If so, format differently.
static Bool inVerbatim = FALSE;

Bool compatibilityMode = FALSE; // If TRUE, maximum Latex compatibility
                                // (Quality of RTF generation deteriorate)

// Linear RTF requires us to set the style per section.
static char *currentNumberStyle = NULL;
static int currentItemSep = 8;
static int CurrentTextWidth = 8640; // Say, six inches
static int CurrentLeftMarginOdd = 400;
static int CurrentLeftMarginEven = 1440;
static int CurrentRightMarginOdd = 1440;
static int CurrentRightMarginEven = 400;
static int CurrentMarginParWidth = 2000;
static int CurrentMarginParSep = 400;  // Gap between marginpar and text
static int CurrentMarginParX = CurrentLeftMarginOdd + CurrentTextWidth + CurrentMarginParSep;
static int GutterWidth = 2300;

// Two-column table dimensions, in twips
static int TwoColWidthA = 1500;
static int TwoColWidthB = 3000;
static int TwoColSpacing = 1;

const int PageWidth = 12242; // 8.25 inches wide for A4

/*
 * Table dimensions
 *
 */

struct ColumnData
{
  char justification; // l, r, c
  int width;          // -1 or a width in twips
  int spacing;        // Space between columns in twips
  Bool leftBorder;
  Bool rightBorder;
};

ColumnData TableData[20];
Bool inTable = FALSE;
Bool startRows = FALSE;
Bool tableVerticalLineLeft = FALSE;
Bool tableVerticalLineRight = FALSE;
int noColumns = 0;   // Current number of columns in table
static int ruleTop = 0;
static int ruleBottom = 0;
int currentRowNumber = 0;

/*
 * Flag to say we've just issued a \par\pard command, so don't
 * repeat this unnecessarily.
 *
 */

Bool issuedNewParagraph = FALSE;

/*
 * Current topics
 *
 */
static char *CurrentChapterName = NULL;
static char *CurrentSectionName = NULL;
static char *CurrentSubsectionName = NULL;
static char *CurrentTopic = NULL;

void SetCurrentTopic(char *s)
{
  if (CurrentTopic) delete[] CurrentTopic;
  CurrentTopic = copystring(s);
}

void SetCurrentChapterName(char *s)
{
  if (CurrentChapterName) delete[] CurrentChapterName;
  CurrentChapterName = copystring(s);
  SetCurrentTopic(s);
}
void SetCurrentSectionName(char *s)
{
  if (CurrentSectionName) delete[] CurrentSectionName;
  CurrentSectionName = copystring(s);
  SetCurrentTopic(s);
}
void SetCurrentSubsectionName(char *s)
{
  if (CurrentSubsectionName) delete[] CurrentSubsectionName;
  CurrentSubsectionName = copystring(s);
  SetCurrentTopic(s);
}

// Hash table for lists of keywords for topics (WinHelp).
static wxHashTable TopicTable(wxKEY_STRING);
void AddKeyWordForTopic(char *topic, char *entry)
{
  wxStringList *list = (wxStringList *)TopicTable.Get(topic);
  if (!list)
  {
    list = new wxStringList;
    TopicTable.Put(topic, (wxObject *)list);
  }
    
  if (!list->Member(entry))
    list->Add(entry);
}

void SplitIndexEntry(char *entry, char *buf1, char *buf2)
{
  int len = strlen(entry); int i = 0;
  while ((i < len) && entry[i] != '!')
  { buf1[i] = entry[i]; i ++; }
  buf1[i] = 0; buf2[0] = 0; int j = 0;

  if (entry[i] == '!')
  {
    i ++;
    while (i < len) { buf2[j] = entry[i]; i ++; j++; }
    buf2[j] = 0;
  }
}

/*
 * Output topic index entries in WinHelp RTF
 *
 */
void GenerateKeywordsForTopic(char *topic)
{
  wxStringList *list = (wxStringList *)TopicTable.Get(topic);
  if (list)
  {
    wxNode *node = list->First();
    while (node)
    {
      char *s = (char *)node->Data();

      // Must separate out main entry form subentry (only 1 subentry allowed)
      char buf1[100]; char buf2[100];
      SplitIndexEntry(s, buf1, buf2);

      TexOutput("K{\\footnote ");
      TexOutput(buf1);
      if (strlen(buf2) > 0)
      {
        // Output subentry
        TexOutput(", ");
        TexOutput(buf2);
      }
      TexOutput("}\n");
      node = node->Next();
    }
  }
}

/*
 * Output index entry in linear RTF
 *
 */
 
void GenerateIndexEntry(char *entry)
{
  char buf1[100]; char buf2[100];
  SplitIndexEntry(entry, buf1, buf2);

  TexOutput("{\\xe {");
  TexOutput(buf1);
  if (strlen(buf2) > 0)
  {
    TexOutput("\\:");
    TexOutput(buf2);
  }
  TexOutput("}}");
}

void ClearKeyWordTable(void)
{
  TopicTable.BeginFind();
  wxNode *node = TopicTable.Next();
  while (node)
  {
    wxStringList *list = (wxStringList *)node->Data();
    delete list;
    node = TopicTable.Next();
  }
  TopicTable.Clear();
}

 /*
  * Write a suitable RTF header.
  *
  */
  
void WriteRTFHeader(FILE *fd)
{
  fprintf(fd, "{\\rtf1\\%s \\deff0\n", RTFCharset);
  fprintf(fd, "{\\fonttbl{\\f0\\froman Times New Roman;}{\\f1\\fdecor Symbol;}{\\f2\\fswiss Arial;}\n");
  fprintf(fd, "{\\f3\\fmodern Courier;}\n}\n{\\colortbl;\n");
  fprintf(fd, "\\red0\\green0\\blue0;\\red0\\green0\\blue255;\\red0\\green255\\blue255;\n");
  fprintf(fd, "\\red0\\green255\\blue0;\n");
  fprintf(fd, "\\red255\\green0\\blue255;\\red255\\green0\\blue0;\\red255\\green255\\blue0;\n");
  fprintf(fd, "\\red255\\green255\\blue255;}");
  fprintf(fd, "\\ftnbj\\ftnrestart"); // Latex default is footnotes at bottom of page, not section.
  fprintf(fd, "\n");
}


/*
 * Given a TexChunk with a string value, scans through the string
 * converting Latex-isms into RTF-isms, such as 2 newlines -> \par,
 * and inserting spaces at the start of lines since in Latex, a newline
 * implies a space, but not in RTF.
 *
 */
 
void ProcessText2RTF(TexChunk *chunk)
{
  Bool changed = FALSE;
  int ptr = 0;
  int i = 0;
  char ch = 1;
  int len = strlen(chunk->value);
  while (ch != 0)
  {
    ch = chunk->value[i];

    if (ch == 10)
    {
      if (inVerbatim)
      {
        BigBuffer[ptr] = 0; strcat(BigBuffer, "\\par\n"); ptr += 5;
        i ++;
        changed = TRUE;
      }
      else
      {
/*
        // 2 newlines means 2 \pars
        if ((len > i+1 && chunk->value[i+1] == 10) ||
             ((len > i+1 && chunk->value[i+1] == 13) &&
              (len > i+2 && chunk->value[i+2] == 10)))
        {
          BigBuffer[ptr] = 0;
//          strcat(BigBuffer, "\\sa240\\par\\pard\n"); ptr += 16;
          strcat(BigBuffer, "\\par\\par\n"); ptr += 9;
          i += 2;
          changed = TRUE;
        }
*/        
        // If the first character of the next line is ASCII,
        // put a space in. Implicit in Latex, not in RTF.
        if ((i > 0) && (len > i+1 && isascii(chunk->value[i+1]) &&
                  !isspace(chunk->value[i+1])) ||
            ((len > i+1 && chunk->value[i+1] == 13) &&
             (len > i+2 && isascii(chunk->value[i+2]) &&
              !isspace(chunk->value[i+2]))))
        {
          // DOS files have a 13 after the 10
          BigBuffer[ptr] = 10;
          ptr ++;
          i ++;
          if (chunk->value[i] == 13)
          {
            BigBuffer[ptr] = 13;
            ptr ++;
            i ++;
          }

          BigBuffer[ptr] = ' ';
          ptr ++;

          // Note that the actual ASCII character seen is dealt with in the next
          // iteration
          changed = TRUE;
        }
        else
        {
          BigBuffer[ptr] = ch;
          i ++;
        }
      }
    }
    else if (!inVerbatim && ch == '`' && (len >= i+1 && chunk->value[i+1] == '`'))
    {
      BigBuffer[ptr] = '"'; ptr ++;
      i += 2;
      changed = TRUE;
    }
    else if (!inVerbatim && ch == '`') // Change ` to '
    {
      BigBuffer[ptr] = 39; ptr ++;
      i += 1;
      changed = TRUE;
    }
    else if (inVerbatim && ch == '\\') // Change backslash to two backslashes
    {
      BigBuffer[ptr] = '\\'; ptr ++;
      BigBuffer[ptr] = '\\'; ptr ++;
      i += 1;
      changed = TRUE;
    }
    else if (inVerbatim && (ch == '{' || ch == '}')) // Escape the curly bracket
    {
      BigBuffer[ptr] = '\\'; ptr ++;
      BigBuffer[ptr] = ch; ptr ++;
      i += 1;
      changed = TRUE;
    }
    else
    {
      BigBuffer[ptr] = ch;
      i ++;
      ptr ++;
    }
  }
  BigBuffer[ptr] = 0;

  if (changed)
  {
    delete[] chunk->value;
    chunk->value = copystring(BigBuffer);
  }
}

/*
 * Scan through all chunks starting from the given one,
 * calling ProcessText2RTF to convert Latex-isms to RTF-isms.
 * This should be called after Tex2Any has parsed the file,
 * and before TraverseDocument is called.
 *
 */
 
void Text2RTF(TexChunk *chunk)
{
  if (!isSync) wxYield();
  if (stopRunning) return;

  switch (chunk->type)
  {
    case CHUNK_TYPE_MACRO:
    {
      TexMacroDef *def = chunk->def;
      if (def && def->ignore)
        return;

      if (def && (def->macroId == ltVERBATIM || def->macroId == ltVERB))
        inVerbatim = TRUE;

      wxNode *node = chunk->children.First();
      while (node)
      {
        TexChunk *child_chunk = (TexChunk *)node->Data();
        Text2RTF(child_chunk);
        node = node->Next();
      }

      if (def && (def->macroId == ltVERBATIM || def->macroId == ltVERB))
        inVerbatim = FALSE;

      break;
    }
    case CHUNK_TYPE_ARG:
    {
      wxNode *node = chunk->children.First();
      while (node)
      {
        TexChunk *child_chunk = (TexChunk *)node->Data();
        Text2RTF(child_chunk);
        node = node->Next();
      }

      break;
    }
    case CHUNK_TYPE_STRING:
    {
      if (chunk->value)
        ProcessText2RTF(chunk);
      break;
    }
  }
}

/*
 * Not used yet
 *
 */
 
char browseBuf[10];
static long browseId = 0;
char *GetBrowseString(void)
{
  char buf[10];
  browseId ++;
  sprintf(buf, "%ld", browseId);
  int noZeroes = 5-strlen(buf);
  strcpy(browseBuf, "browse");
  for (int i = 0; i < noZeroes; i++)
    strcat(browseBuf, "0");
  strcat(browseBuf, buf);
  return browseBuf;
}

/*
 * Keeping track of environments to restore the styles after \pard.
 * Push strings like "\qc" onto stack.
 *
 */

void PushEnvironmentStyle(char *style)
{
  environmentStack.Add(style);
}

void PopEnvironmentStyle(void)
{
  wxNode *node = environmentStack.Last();
  if (node)
  {
    char *val = (char *)node->Data();
    delete[] val;
    delete node;
  }
}

// Write out the styles, most recent first.
void WriteEnvironmentStyles(void)
{
  wxNode *node = environmentStack.Last();
  while (node)
  {
    char *val = (char *)node->Data();
    TexOutput(val);
    node = node->Next();
  }
  if (!inTable && (ParIndent > 0) && (forbidParindent == 0))
  {
    char buf[15];
    sprintf(buf, "\\fi%d", ParIndent*20); // Convert points to TWIPS
    TexOutput(buf);
  }
  if (environmentStack.Number() > 0 || (ParIndent > 0))
    TexOutput("\n");
}

/*
 * Parse table argument
 *
 */

Bool ParseTableArgument(char *value)
{
  noColumns = 0;
  int i = 0;
  int len = strlen(value);
  Bool isBorder = FALSE;
  while (i < len)
  {
    int ch = value[i];
    if (ch == '|')
    {
      i ++;
      isBorder = TRUE;
    }
    else if (ch == 'l')
    {
      TableData[noColumns].leftBorder = isBorder;
      TableData[noColumns].rightBorder = FALSE;
      TableData[noColumns].justification = 'l';
      TableData[noColumns].width = 2000; // Estimate
//      TableData[noColumns].spacing = ??
      noColumns ++;
      i ++;
      isBorder = FALSE;
    }
    else if (ch == 'c')
    {
      TableData[noColumns].leftBorder = isBorder;
      TableData[noColumns].rightBorder = FALSE;
      TableData[noColumns].justification = 'c';
      TableData[noColumns].width = 2000; // Estimate
//      TableData[noColumns].spacing = ??
      noColumns ++;
      i ++;
      isBorder = FALSE;
    }
    else if (ch == 'r')
    {
      TableData[noColumns].leftBorder = isBorder;
      TableData[noColumns].rightBorder = FALSE;
      TableData[noColumns].justification = 'r';
      TableData[noColumns].width = 2000; // Estimate
//      TableData[noColumns].spacing = ??
      noColumns ++;
      i ++;
      isBorder = FALSE;
    }
    else if (ch == 'p')
    {
      i ++;
      int j = 0;
      char numberBuf[50];
      ch = value[i];
      if (ch == '{')
      {
        i++;
        ch = value[i];
      }
        
      while ((i < len) && (isdigit(ch) || ch == '.'))
      {
        numberBuf[j] = ch;
        j ++;
        i ++;
        ch = value[i];
      }
      // Assume we have 2 characters for units
      numberBuf[j] = value[i];
      j ++; i++;
      numberBuf[j] = value[i];
      j ++; i++;
      numberBuf[j] = 0;
      if (value[i] == '}') i++;
      
      TableData[noColumns].leftBorder = isBorder;
      TableData[noColumns].rightBorder = FALSE;
      TableData[noColumns].justification = 'r';
      TableData[noColumns].width = 20*ParseUnitArgument(numberBuf);
//      TableData[noColumns].spacing = ??
      noColumns ++;
      isBorder = FALSE;
    }
    else i++;
  }
  if (isBorder)
    TableData[noColumns-1].rightBorder = TRUE;
  return TRUE;
}

/*
 * Output a header
 *
 */

void OutputRTFHeaderCommands(void)
{
  char buf[300];
  if (PageStyle && strcmp(PageStyle, "plain") == 0)
  {
    TexOutput("{\\headerl }{\\headerr }");
  }
  else if (PageStyle && strcmp(PageStyle, "empty") == 0)
  {
    TexOutput("{\\headerl }{\\headerr }");
  }
  else if (PageStyle && strcmp(PageStyle, "headings") == 0)
  {
    // Left header
    TexOutput("{\\headerl\\fi0 ");

    if (headerRule)
      TexOutput("\\brdrb\\brdrs\\brdrw15\\brsp20 ");

    TexOutput("{\\i \\qr ");
    if (DocumentStyle == LATEX_ARTICLE)
    {
      sprintf(buf, "SECTION %d", sectionNo);
      TexOutput(buf);
    }
    else
    {
      sprintf(buf, "CHAPTER %d: ", chapterNo);
      TexOutput(buf);
    }
    TexOutput("{\\field{\\*\\fldinst PAGE \\\\* MERGEFORMAT }{\\fldrslt 1}}");
    TexOutput("}\\par\\pard}");

    // Right header
    TexOutput("{\\headerr\\fi0 ");

    if (headerRule)
      TexOutput("\\brdrb\\brdrs\\brdrw15\\brsp20 ");

    TexOutput("{\\i \\qc ");
    if (DocumentStyle == LATEX_ARTICLE)
    {
      sprintf(buf, "SECTION %d", sectionNo);
      TexOutput(buf);
    }
    else
    {
      sprintf(buf, "CHAPTER %d", chapterNo);
      TexOutput(buf);
    }
    TexOutput("{\\field{\\*\\fldinst PAGE \\\\* MERGEFORMAT }{\\fldrslt 1}}");
    TexOutput("}\\par\\pard}");
  }
  else
  {
    if (LeftHeaderEven || CentreHeaderEven || RightHeaderEven)
    {
      TexOutput("{\\headerl\\fi0 ");

      if (headerRule)
        TexOutput("\\brdrb\\brdrs\\brdrw15\\brsp20 ");

      if (LeftHeaderEven)
      {
        if (!CentreHeaderEven && !RightHeaderEven)
          TexOutput("\\ql ");
        TraverseChildrenFromChunk(LeftHeaderEven);
      }
      if (CentreHeaderEven)
      {
        if (!LeftHeaderEven && !RightHeaderEven)
          TexOutput("\\qc ");
        else
          TexOutput("\\tab\\tab\\tab ");
        TraverseChildrenFromChunk(CentreHeaderEven);
      }
      if (RightHeaderEven)
      {
        if (!LeftHeaderEven && !CentreHeaderEven)
          TexOutput("\\qr ");
        else
          TexOutput("\\tab\\tab\\tab\\tab\\tab\\tab ");
        TraverseChildrenFromChunk(RightHeaderEven);
      }
      TexOutput("\\par\\pard}");
    }

    if (LeftHeaderOdd || CentreHeaderOdd || RightHeaderOdd)
    {
      TexOutput("{\\headerr\\fi0 ");

      if (headerRule)
        TexOutput("\\brdrb\\brdrs\\brdrw15\\brsp20 ");

      if (LeftHeaderOdd)
      {
        if (!CentreHeaderOdd && !RightHeaderOdd)
          TexOutput("\\ql ");
        TraverseChildrenFromChunk(LeftHeaderOdd);
      }
      if (CentreHeaderOdd)
      {
        if (!LeftHeaderOdd && !RightHeaderOdd)
          TexOutput("\\qc ");
        else
          TexOutput("\\tab\\tab\\tab ");
        TraverseChildrenFromChunk(CentreHeaderOdd);
      }
      if (RightHeaderOdd)
      {
        if (!LeftHeaderOdd && !CentreHeaderOdd)
          TexOutput("\\qr ");
        else
          TexOutput("\\tab\\tab\\tab\\tab\\tab\\tab ");
        TraverseChildrenFromChunk(RightHeaderOdd);
      }
      TexOutput("\\par\\pard}");
    }
    // As an approximation, don't put a header on the first page of a section.
    // This may not always be desired, but it's a reasonable guess.
    TexOutput("{\\headerf }");
  }
}

void OutputRTFFooterCommands(void)
{
  if (PageStyle && strcmp(PageStyle, "plain") == 0)
  {
    TexOutput("{\\footerl\\fi0 ");
    if (footerRule)
      TexOutput("\\brdrt\\brdrs\\brdrw15\\brsp20 ");
    TexOutput("{\\qc ");
    TexOutput("{\\field{\\*\\fldinst PAGE \\\\* MERGEFORMAT }{\\fldrslt 1}}");
    TexOutput("}\\par\\pard}");

    TexOutput("{\\footerr\\fi0 ");
    if (footerRule)
      TexOutput("\\brdrt\\brdrs\\brdrw15\\brsp20 ");
    TexOutput("{\\qc ");
    TexOutput("{\\field{\\*\\fldinst PAGE \\\\* MERGEFORMAT }{\\fldrslt 1}}");
    TexOutput("}\\par\\pard}");
  }
  else if (PageStyle && strcmp(PageStyle, "empty") == 0)
  {
    TexOutput("{\\footerl }{\\footerr }");
  }
  else if (PageStyle && strcmp(PageStyle, "headings") == 0)
  {
    TexOutput("{\\footerl }{\\footerr }");
  }
  else
  {
    if (LeftFooterEven || CentreFooterEven || RightFooterEven)
    {
      TexOutput("{\\footerl\\fi0 ");
      if (footerRule)
        TexOutput("\\brdrt\\brdrs\\brdrw15\\brsp20 ");
      if (LeftFooterEven)
      {
        if (!CentreFooterEven && !RightFooterEven)
          TexOutput("\\ql ");
        TraverseChildrenFromChunk(LeftFooterEven);
      }
      if (CentreFooterEven)
      {
        if (!LeftFooterEven && !RightFooterEven)
          TexOutput("\\qc ");
        else
          TexOutput("\\tab\\tab\\tab ");
        TraverseChildrenFromChunk(CentreFooterEven);
      }
      if (RightFooterEven)
      {
        if (!LeftFooterEven && !CentreFooterEven)
          TexOutput("\\qr ");
        else
          TexOutput("\\tab\\tab\\tab\\tab\\tab\\tab ");
        TraverseChildrenFromChunk(RightFooterEven);
      }
      TexOutput("\\par\\pard}");
    }

    if (LeftFooterOdd || CentreFooterOdd || RightFooterOdd)
    {
      TexOutput("{\\footerr\\fi0 ");
      if (footerRule)
        TexOutput("\\brdrt\\brdrs\\brdrw15\\brsp20 ");
      if (LeftFooterOdd)
      {
        if (!CentreFooterOdd && !RightFooterOdd)
          TexOutput("\\ql ");
        TraverseChildrenFromChunk(LeftFooterOdd);
      }
      if (CentreFooterOdd)
      {
        if (!LeftFooterOdd && !RightFooterOdd)
          TexOutput("\\qc ");
        else
          TexOutput("\\tab\\tab\\tab ");
        TraverseChildrenFromChunk(CentreFooterOdd);
      }
      if (RightFooterOdd)
      {
        if (!LeftFooterOdd && !CentreFooterOdd)
          TexOutput("\\qr ");
        else
          TexOutput("\\tab\\tab\\tab\\tab\\tab\\tab ");
        TraverseChildrenFromChunk(RightFooterOdd);
      }
      TexOutput("\\par\\pard}");
    }

    // As an approximation, put a footer on the first page of a section.
    // This may not always be desired, but it's a reasonable guess.
    if (LeftFooterOdd || CentreFooterOdd || RightFooterOdd)
    {
      TexOutput("{\\footerf\\fi0 ");
      if (LeftFooterOdd)
      {
        if (!CentreFooterOdd && !RightFooterOdd)
          TexOutput("\\ql ");
        TraverseChildrenFromChunk(LeftFooterOdd);
      }
      if (CentreFooterOdd)
      {
        if (!LeftFooterOdd && !RightFooterOdd)
          TexOutput("\\qc ");
        else
          TexOutput("\\tab\\tab\\tab ");
        TraverseChildrenFromChunk(CentreFooterOdd);
      }
      if (RightFooterOdd)
      {
        if (!LeftFooterOdd && !CentreFooterOdd)
          TexOutput("\\qr ");
        else
          TexOutput("\\tab\\tab\\tab\\tab\\tab\\tab ");
        TraverseChildrenFromChunk(RightFooterOdd);
      }
      TexOutput("\\par\\pard}");
    }
  }
}

// Called on start/end of macro examination
void RTFOnMacro(int macroId, int no_args, Bool start)
{
  if (start && (macroId != ltPAR))
  {
    issuedNewParagraph = FALSE;
  }

  char buf[300];
  switch (macroId)
  {
  case ltCHAPTER:
  case ltCHAPTERSTAR:
  case ltCHAPTERHEADING:
  {
    if (start)
    {
      sectionNo = 0;
      figureNo = 0;
      subsectionNo = 0;
      subsubsectionNo = 0;
      footnoteCount = 0;

      if (macroId != ltCHAPTERSTAR)
        chapterNo ++;

      if (winHelp)
        SetCurrentOutputs(Contents, Chapters);

      if (!winHelp)
      {
        fprintf(Chapters, "\\sect\\titlepg\n");

        // If a non-custom page style, we generate the header now.
        if (PageStyle && (strcmp(PageStyle, "plain") == 0 ||
                          strcmp(PageStyle, "empty") == 0 ||
                          strcmp(PageStyle, "headings") == 0))
        {
          OutputRTFHeaderCommands();
          OutputRTFFooterCommands();
        }
        
        // Need to reset the current numbering style, or RTF forgets it.
        if (currentNumberStyle)
        {
          if (strcmp(currentNumberStyle, "arabic") == 0)
          {
            fprintf(Chapters, "\\pgndec");
          }
          else if (strcmp(currentNumberStyle, "roman") == 0)
          {
            fprintf(Chapters,"\\pgnlcrm");
          }
          else if (strcmp(currentNumberStyle, "Roman") == 0)
          {
            fprintf(Chapters,"\\pgnucrm");
          }
          else if (strcmp(currentNumberStyle, "alph") == 0)
          {
            fprintf(Chapters,"\\pgnlcltr");
          }
          else if (strcmp(currentNumberStyle, "Alph") == 0)
          {
            fprintf(Chapters,"\\pgnucltr");
          }
        }
        SetCurrentOutput(Contents);
      }
      else
        fprintf(Chapters, "\\page\n");

      if (winHelp)
        fprintf(Contents, "\n{\\uldb ");
      else if (macroId == ltCHAPTER)
      {
        // Section
        fprintf(Contents, "\\par\n\\pard{\\b %d\\tab ", chapterNo);
      }
      else if (macroId == ltCHAPTERHEADING)
      {
        fprintf(Contents, "\\par\n\\pard{\\b ");
      }
      else SetCurrentOutput(NULL);

      startedSections = TRUE;

      if (winHelp)
        fprintf(Chapters, "\n${\\footnote ");
    }
    else
    {
      char *topicName = FindTopicName(GetNextChunk());
      SetCurrentChapterName(topicName);

      AddTexRef(topicName, NULL, "chapter", chapterNo);
      
      if (winHelp)
        fprintf(Contents, "}{\\v %s}\\par\\pard\n", topicName);
      else if ((macroId == ltCHAPTER) || (macroId == ltCHAPTERHEADING))
        fprintf(Contents, "}\\par\\par\\pard\n");

      SetCurrentOutput(Chapters);

      if (winHelp)
      {
        fprintf(Chapters, "}\n#{\\footnote %s}\n", topicName);
        fprintf(Chapters, "+{\\footnote %s}\n", GetBrowseString());
        fprintf(Chapters, "K{\\footnote ");
        OutputCurrentSection();
        fprintf(Chapters, "}\n");
        GenerateKeywordsForTopic(topicName);
        if (useUpButton)
        {
          fprintf(Chapters, "!{\\footnote EnableButton(\"Up\");ChangeButtonBinding(\"Up\", \"JumpId(`%s.hlp', `%s')\")}\n",
               FileNameFromPath(FileRoot), "Contents");
        }
      }

      char *styleCommand = "";
      if (!winHelp && useHeadingStyles)
        styleCommand = "\\s1";
      fprintf(Chapters, "\\pard{%s\\b\\fs%d ", (winHelp ? "\\keepn" : styleCommand),
         chapterFont*2);
      if (!winHelp)
      {
        if (macroId == ltCHAPTER)
          fprintf(Chapters, "Chapter %d\\par\\par\n", chapterNo);
      }
      OutputCurrentSection();
      TexOutput("}\n");
//      if (winHelp) TexOutput("\\pard");
    }
    break;
  }
  case ltSECTION:
  case ltSECTIONSTAR:
  case ltSECTIONHEADING:
  case ltGLOSS:
  {
    FILE *jumpFrom;
    if (DocumentStyle == LATEX_ARTICLE)
      jumpFrom = Contents;
    else
      jumpFrom = Chapters;

    if (start)
    {
      subsectionNo = 0;
      subsubsectionNo = 0;
      if (DocumentStyle == LATEX_ARTICLE)
        footnoteCount = 0;

      if (macroId != ltSECTIONSTAR)
        sectionNo ++;

      if (!winHelp && (DocumentStyle == LATEX_ARTICLE))
      {
        TexOutput("\\sect\n");
        // If a non-custom page style, we generate the header now.
        if (PageStyle && (strcmp(PageStyle, "plain") == 0 ||
                          strcmp(PageStyle, "empty") == 0 ||
                          strcmp(PageStyle, "headings") == 0))
        {
          OutputRTFHeaderCommands();
          OutputRTFFooterCommands();
        }
      } 
        
      if (winHelp)
      {
        SetCurrentOutputs(jumpFrom, Sections);
        // Newline for a new section if this is an article
        if ((DocumentStyle == LATEX_ARTICLE) &&
            ((macroId == ltSECTION) || (macroId == ltSECTIONSTAR)))
          fprintf(Sections, "\\page\n");
      }
      else
        SetCurrentOutput(Contents);

      if (winHelp)
        fprintf(jumpFrom, "\n{\\uldb ");
      else if (macroId == ltSECTION)
      {
        if (DocumentStyle == LATEX_REPORT)
          fprintf(Contents, "\n\\pard{\\tab %d.%d\\tab ", chapterNo, sectionNo);
        else
          fprintf(Contents, "\\par\n\\pard{\\b %d\\tab ", sectionNo);
      }
      else if (macroId == ltSECTIONHEADING)
      {
        if (DocumentStyle == LATEX_REPORT)
          fprintf(Contents, "\n\\pard{\\tab ", chapterNo, sectionNo);
        else
          fprintf(Contents, "\\par\n\\pard{\\b ", sectionNo);
      }
      else SetCurrentOutput(NULL);

      if (startedSections)
      {
        if (winHelp)
          fprintf(Sections, "\\page\n");
      }

      startedSections = TRUE;

      if (winHelp)
        fprintf(Sections, "\n${\\footnote ");
    }
    else
    {
      char *topicName = FindTopicName(GetNextChunk());
      SetCurrentSectionName(topicName);
      AddTexRef(topicName, NULL, "section", chapterNo, sectionNo);

      if (winHelp)
        fprintf(jumpFrom, "}{\\v %s}\\par\\pard\n", topicName);
      else if ((macroId != ltSECTIONSTAR) && (macroId != ltGLOSS))
      {
        if (DocumentStyle == LATEX_REPORT)
          fprintf(Contents, "}\\par\\pard\n");
        else
          fprintf(Contents, "}\\par\\par\\pard\n");
      }
        
      SetCurrentOutput(winHelp ? Sections : Chapters);

      if (winHelp)
      {
        fprintf(Sections, "}\n#{\\footnote %s}\n", topicName);
        fprintf(Sections, "+{\\footnote %s}\n", GetBrowseString());
        fprintf(Sections, "K{\\footnote ");
        OutputCurrentSection();
        fprintf(Sections, "}\n");
        GenerateKeywordsForTopic(topicName);
        if (useUpButton)
        {
          if (DocumentStyle == LATEX_ARTICLE)
          {
            fprintf(Sections, "!{\\footnote EnableButton(\"Up\");ChangeButtonBinding(\"Up\", \"JumpId(`%s.hlp', `%s')\")}\n",
               FileNameFromPath(FileRoot), "Contents");
          }
          else if (CurrentChapterName)
          {
            fprintf(Sections, "!{\\footnote EnableButton(\"Up\");ChangeButtonBinding(\"Up\", \"JumpId(`%s.hlp', `%s')\")}\n",
               FileNameFromPath(FileRoot), CurrentChapterName);
          }
        }
      }

      char *styleCommand = "";
      if (!winHelp && useHeadingStyles)
      {
        if (DocumentStyle == LATEX_ARTICLE)
          styleCommand = "\\s1";
        else
          styleCommand = "\\s2";
      }
      char *keep = "";
      if (winHelp && (macroId != ltGLOSS))
        keep = "\\keepn";
        
      fprintf(winHelp ? Sections : Chapters, "\\pard{%s%s\\b\\fs%d ",
         keep, styleCommand, sectionFont*2);
      if (!winHelp)
      {
        if ((macroId != ltSECTIONSTAR) && (macroId != ltGLOSS))
        {
          if (DocumentStyle == LATEX_REPORT)
            fprintf(Chapters, "%d.%d. ", chapterNo, sectionNo);
          else
            fprintf(Chapters, "%d. ", sectionNo);
        }
      }
      OutputCurrentSection();
      TexOutput("}\n");
//      if (winHelp) TexOutput("\\pard");
    }
    break;
  }
  case ltSUBSECTION:
  case ltSUBSECTIONSTAR:
  case ltMEMBERSECTION:
  case ltFUNCTIONSECTION:
  {
    if (start)
    {
      subsubsectionNo = 0;

      if (macroId != ltSUBSECTIONSTAR)
        subsectionNo ++;

      if (winHelp)
        SetCurrentOutputs(Sections, Subsections);
      else
        SetCurrentOutput(Contents);

      if (winHelp)
      {
        SetCurrentOutputs(Sections, Subsections);
        fprintf(Sections, "\n{\\uldb ");
      }
      else if ((macroId != ltSUBSECTIONSTAR) && (macroId != ltMEMBERSECTION) &&
         (macroId != ltFUNCTIONSECTION))
      {
        SetCurrentOutput(Contents);
        if (DocumentStyle == LATEX_REPORT)
          fprintf(Contents, "\n\\pard\\tab\\tab %d.%d.%d\\tab ", chapterNo, sectionNo, subsectionNo);
        else
          fprintf(Contents, "\n\\pard\\tab %d.%d\\tab ", sectionNo, subsectionNo);
      } else SetCurrentOutput(NULL);

      if (startedSections)
      {
        if (winHelp)
          fprintf(Subsections, "\\page\n");
        else
          fprintf(Chapters, "\\par\n");
      }
      startedSections = TRUE;

      if (winHelp)
        fprintf(Subsections, "\n${\\footnote ");
    }
    else
    {
      char *topicName = FindTopicName(GetNextChunk());
      SetCurrentSubsectionName(topicName);

      AddTexRef(topicName, NULL, "section", chapterNo, sectionNo, subsectionNo);

      if (winHelp)
        fprintf(Sections, "}{\\v %s}\\par\\pard\n", topicName);
      else if ((macroId != ltSUBSECTIONSTAR) && (macroId != ltMEMBERSECTION) &&
         (macroId != ltFUNCTIONSECTION))
        fprintf(Contents, "\\par\\pard\n");

      SetCurrentOutput(winHelp ? Subsections : Chapters);
      if (winHelp)
      {
        fprintf(Subsections, "}\n#{\\footnote %s}\n", topicName);
        fprintf(Subsections, "+{\\footnote %s}\n", GetBrowseString());
        fprintf(Subsections, "K{\\footnote ");
        OutputCurrentSection();
        fprintf(Subsections, "}\n");
        GenerateKeywordsForTopic(topicName);
        if (useUpButton && CurrentSectionName)
        {
          fprintf(Subsections, "!{\\footnote EnableButton(\"Up\");ChangeButtonBinding(\"Up\", \"JumpId(`%s.hlp', `%s')\")}\n",
             FileNameFromPath(FileRoot), CurrentSectionName);
        }
      }

      char *styleCommand = "";
      if (!winHelp && useHeadingStyles)
      {
        if (DocumentStyle == LATEX_ARTICLE)
          styleCommand = "\\s2";
        else
          styleCommand = "\\s3";
      }
      char *keep = "";
      if (winHelp)
        keep = "\\keepn";

      fprintf(winHelp ? Subsections : Chapters, "\\pard{%s%s\\b\\fs%d ",
        keep, styleCommand, subsectionFont*2);

      if (!winHelp &&
         (macroId != ltSUBSECTIONSTAR) && (macroId != ltMEMBERSECTION) &&
         (macroId != ltFUNCTIONSECTION))
      {
        if (DocumentStyle == LATEX_REPORT)
          fprintf(Chapters, "%d.%d.%d. ", chapterNo, sectionNo, subsectionNo);
        else
          fprintf(Chapters, "%d.%d. ", sectionNo, subsectionNo);
      }
      OutputCurrentSection(); // Repeat section header
      TexOutput("}\n");
//      if (winHelp) TexOutput("\\pard");
    }
    break;
  }
  case ltSUBSUBSECTION:
  case ltSUBSUBSECTIONSTAR:
  {
    if (start)
    {
      if (macroId != ltSUBSUBSECTIONSTAR)
        subsubsectionNo ++;

      if (winHelp)
      {
        SetCurrentOutputs(Subsections, Subsubsections);
        fprintf(Subsections, "\n{\\uldb ");
      }
      else if (macroId != ltSUBSUBSECTIONSTAR)
      {
        if (DocumentStyle == LATEX_ARTICLE)
        {
          SetCurrentOutput(Contents);
          fprintf(Contents, "\n\\tab\\tab %d.%d.%d\\tab ",
                             sectionNo, subsectionNo, subsubsectionNo);
	}
        else
          SetCurrentOutput(NULL); // Don't write it into the contents, or anywhere else
      }
      else
        SetCurrentOutput(NULL);

      if (startedSections)
      {
        if (winHelp)
          fprintf(Subsubsections, "\\page\n");
        else
          fprintf(Chapters, "\\par\n");
      }

      startedSections = TRUE;

      if (winHelp)
        fprintf(Subsubsections, "\n${\\footnote ");
    }
    else
    {
      char *topicName = FindTopicName(GetNextChunk());
      SetCurrentTopic(topicName);
      
      AddTexRef(topicName, NULL, "section", chapterNo, sectionNo, subsectionNo, subsubsectionNo);

      if (winHelp)
        fprintf(Subsections, "}{\\v %s}\\par\\pard\n", topicName);
      else if ((DocumentStyle == LATEX_ARTICLE) && (macroId != ltSUBSUBSECTIONSTAR))
        fprintf(Contents, "\\par\\pard\n");
        
      SetCurrentOutput(winHelp ? Subsubsections : Chapters);
      if (winHelp)
      {
        fprintf(Subsubsections, "}\n#{\\footnote %s}\n", topicName);
        fprintf(Subsubsections, "+{\\footnote %s}\n", GetBrowseString());
        fprintf(Subsubsections, "K{\\footnote ");
        OutputCurrentSection();
        fprintf(Subsubsections, "}\n");
        GenerateKeywordsForTopic(topicName);
        if (useUpButton && CurrentSubsectionName)
        {
          fprintf(Subsubsections, "!{\\footnote EnableButton(\"Up\");ChangeButtonBinding(\"Up\", \"JumpId(`%s.hlp', `%s')\")}\n",
             FileNameFromPath(FileRoot), CurrentSubsectionName);
        }
      }

      char *styleCommand = "";
      if (!winHelp && useHeadingStyles)
      {
        if (DocumentStyle == LATEX_ARTICLE)
          styleCommand = "\\s3";
        else
          styleCommand = "\\s4";
      }
      char *keep = "";
      if (winHelp)
        keep = "\\keepn";

      fprintf(winHelp ? Subsubsections : Chapters, "\\pard{%s%s\\b\\fs%d ",
         keep, styleCommand, subsectionFont*2);
         
      if (!winHelp && (DocumentStyle == LATEX_ARTICLE))
        fprintf(Chapters, "%d.%d.%d. ", sectionNo, subsectionNo, subsubsectionNo);
      
      OutputCurrentSection(); // Repeat section header
      TexOutput("}\n");
//      if (winHelp) TexOutput("\\pard");
    }
    break;
  }
  case ltCAPTION:
  case ltCAPTIONSTAR:
  {
    if (start)
    {
      figureNo ++;

      TexOutput("\\pard\\par");
      char figBuf[40];
      if (DocumentStyle != LATEX_ARTICLE)
        sprintf(figBuf, "Figure %d.%d: ", chapterNo, figureNo);
      else
        sprintf(figBuf, "Figure %d: ", figureNo);

      TexOutput("\\qc{\\b ");
      TexOutput(figBuf);
    }
    else
    {
      TexOutput("}\\par\\pard\n");
      WriteEnvironmentStyles();

      char *topicName = FindTopicName(GetNextChunk());

      AddTexRef(topicName, NULL, NULL,
           ((DocumentStyle != LATEX_ARTICLE) ? chapterNo : figureNo),
            ((DocumentStyle != LATEX_ARTICLE) ? figureNo : 0));
    }
    break;
  }
  case ltFUNC:
  case ltPFUNC:
  {
    SetCurrentOutput(winHelp ? Subsections : Chapters);
    if (start)
    {
      TexOutput("{");
    }
    else
    {
      TexOutput("}\n");
      if (winHelp)
      {
        fprintf(Subsections, "K{\\footnote ");
        suppressNameDecoration = TRUE;
        TraverseChildrenFromChunk(currentMember);
        suppressNameDecoration = FALSE;
        fprintf(Subsections, "}\n");
      }
    }
    break;
  }
  case ltCLIPSFUNC:
  {
    SetCurrentOutput(winHelp ? Subsections : Chapters);
    if (start)
    {
      TexOutput("{");
    }
    else
    {
      TexOutput("}\n");
      if (winHelp)
      {
        fprintf(Subsections, "K{\\footnote ");
        suppressNameDecoration = TRUE;  // Necessary so don't print "(\\bf" etc.
        TraverseChildrenFromChunk(currentMember);
        suppressNameDecoration = FALSE;
        fprintf(Subsections, "}\n");
      }
    }
    break;
  }
  case ltMEMBER:
  {
    SetCurrentOutput(winHelp ? Subsections : Chapters);
    if (start)
    {
      TexOutput("{\\b ");
    }
    else
    {
      TexOutput("}\n");
      if (winHelp)
      {
        fprintf(Subsections, "K{\\footnote ");
        TraverseChildrenFromChunk(currentMember);
        fprintf(Subsections, "}\n");
      }
    }
    break;
  }
  case ltDOCUMENT:
  {
    if (start)
      SetCurrentOutput(Chapters);
    break;
  }
  case ltTABLEOFCONTENTS:
  {
    if (start)
    {
      if (!winHelp && insertTOC)
      {
        // Insert Word for Windows table of contents
        TexOutput("\\par\\pard");
        sprintf(buf, "{\\b\\fs%d Contents}\\par\\par\\pard\n\n", titleFont*2);
        TexOutput(buf);
        TexOutput("{\\field{\\*\\fldinst TOC \\\\o \"1-3\" }{\\fldrslt SELECT ALL THEN F9 TO REFORMAT TOC}}\n");
        TexOutput("\\sect\\sectd");
      }
      else
      {
        FILE *fd = fopen(ContentsName, "r");
        if (fd)
        {
          char ch = getc(fd);
          while (ch != EOF)
          {
            putc(ch, Chapters);
            ch = getc(fd);
          }
          fclose(fd);
        }
        else
        {
          TexOutput("{\\i RUN TEX2RTF AGAIN FOR CONTENTS PAGE}\\par\n");
          OnInform("Run Tex2RTF again to include contents page.");
        }
      }
    }
    break;
  }
  case ltVOID:
  {
    if (start)
      TexOutput("{\\b void}");
    break;
  }
  case ltHARDY:
  {
    if (start)
      TexOutput("{\\scaps HARDY}");
    break;
  }
  case ltWXCLIPS:
  {
    if (start)
      TexOutput("wxCLIPS");
    break;
  }
  case ltSPECIALAMPERSAND:
  {
/*
    if (inTable)
      TexOutput("\\tab\n");
    else
      TexOutput("&");
*/
    if (start)
    {
      if (inTable)
        TexOutput("\\cell ");
      else
        TexOutput("&");
    }
    break;
  }
  case ltBACKSLASHCHAR:
  {
    if (start)
    {
      if (inTable)
      {
//        TexOutput("\\cell\\row\\trowd\\trgaph108\\trleft-108\n");
        TexOutput("\\cell\\row\\trowd\\trgaph108\n");
        int currentWidth = 0;
        for (int i = 0; i < noColumns; i++)
        {
          currentWidth += TableData[i].width;
          if (TableData[i].rightBorder)
            TexOutput("\\clbrdrr\\brdrs\\brdrw15");

          if (TableData[i].leftBorder)
            TexOutput("\\clbrdrl\\brdrs\\brdrw15");
          
          sprintf(buf, "\\cellx%d", currentWidth);
          TexOutput(buf);
        }
        TexOutput("\\pard\\intbl\n");
      }
      else
        TexOutput("\\line\n");
    }
    break;
/*
    if (start)
    {
    if (inTable)
//      TexOutput("\\cell\n\\row\\pard\n\\intbl\n");
    {
      if (tableVerticalLineLeft)
        TexOutput("\\brdrl\\brdrs");
      if (tableVerticalLineRight)
        TexOutput("\\brdrr\\brdrs");

      TexOutput("\\par\\pard\n");

      // Calculate a rough size for each column
      int colSize = 6000/(noColumns-1);
      int colPos = colSize;
      for (int j = 0; j < noColumns; j++)
      {
        sprintf(buf, "\\tx%d", colPos);
        TexOutput(buf);
        colPos += colSize;
      }
      TexOutput("\n");
    }
    else
      TexOutput("\\line\n");
    }
*/
    break;
  }
  case ltRANGLEBRA:
  {
    if (start)
      TexOutput("\tab ");
    break;
  }
  case ltRTFSP:  // Explicit space, RTF only
  {
    if (start)
      TexOutput(" ");
   break;
  }
  case ltITEMIZE:
  case ltENUMERATE:
  case ltDESCRIPTION:
  {
    if (start)
    {
      if (indentLevel > 0)
      {
        TexOutput("\\par\\par\n");
      }
      indentLevel ++;
      TexOutput("\\fi0\n");
      int listType;
      if (macroId == ltENUMERATE)
        listType = LATEX_ENUMERATE;
      else if (macroId == ltITEMIZE)
        listType = LATEX_ITEMIZE;
      else
        listType = LATEX_DESCRIPTION;

      int oldIndent = 0;
      wxNode *node = itemizeStack.First();
      if (node)
        oldIndent = ((ItemizeStruc *)node->Data())->indentation;

      int indentSize1 = oldIndent + 20*labelIndentTab;
      int indentSize2 = oldIndent + 20*itemIndentTab;

      ItemizeStruc *struc = new ItemizeStruc(listType, indentSize2, indentSize1);
      itemizeStack.Insert(struc);
      
      sprintf(buf, "\\tx%d\\tx%d\\li%d", indentSize1, indentSize2, indentSize2);
      PushEnvironmentStyle(buf);
    }
    else
    {
      currentItemSep = 8; // Reset to the default
      indentLevel --;
      PopEnvironmentStyle();

      if (itemizeStack.First())
      {
        ItemizeStruc *struc = (ItemizeStruc *)itemizeStack.First()->Data();
        delete struc;
        delete itemizeStack.First();
      }

      if (itemizeStack.Number() == 0)
      {
        if (!issuedNewParagraph)
        {
          TexOutput("\\par\\pard\n");
          issuedNewParagraph = TRUE;
        }
        WriteEnvironmentStyles();
      }
    }
    break;
  }
  case ltTWOCOLLIST:
  {
    if (start)
    {
      indentLevel ++;
      int oldIndent = 0;
      wxNode *node = itemizeStack.First();
      if (node)
        oldIndent = ((ItemizeStruc *)node->Data())->indentation;

      int indentSize = oldIndent + TwoColWidthA;

      ItemizeStruc *struc = new ItemizeStruc(LATEX_TWOCOL, indentSize);
      itemizeStack.Insert(struc);
      
      sprintf(buf, "\\tx%d\\li%d\\ri%d", indentSize, indentSize, TwoColWidthA+TwoColWidthB+oldIndent);
      PushEnvironmentStyle(buf);
    }
    else
    {
      indentLevel --;
      PopEnvironmentStyle();
      if (itemizeStack.First())
      {
        ItemizeStruc *struc = (ItemizeStruc *)itemizeStack.First()->Data();
        delete struc;
        delete itemizeStack.First();
      }
    }
    break;
  }
  case ltITEM:
  {
    wxNode *node = itemizeStack.First();
    if (node)
    {
      ItemizeStruc *struc = (ItemizeStruc *)node->Data();
      if (!start)
      {
        struc->currentItem += 1;
        char indentBuf[60];

        int indentSize1 = struc->labelIndentation;
        int indentSize2 = struc->indentation;

        TexOutput("\n");
        if (struc->currentItem > 1)
        {
          if (currentItemSep > 0)
            TexOutput("\\par");

          TexOutput("\\par");
//          WriteEnvironmentStyles();
        }

        sprintf(buf, "\\tx%d\\tx%d\\li%d\\fi-%d\n", indentSize1, indentSize2,
                  indentSize2, 20*itemIndentTab);
        TexOutput(buf);

        switch (struc->listType)
        {
          case LATEX_ENUMERATE:
          {
            sprintf(indentBuf, "\\tab{\\b %d.}\\tab", struc->currentItem);
            TexOutput(indentBuf);
            break;
          }
          case LATEX_ITEMIZE:
          {
            if (bulletFile && winHelp)
              sprintf(indentBuf, "\\tab\\{bmc %s\\}\\tab", bulletFile);
            else if (winHelp)
              sprintf(indentBuf, "\\tab{\\b o}\\tab");
            else
              sprintf(indentBuf, "\\tab{\\f1\\'b7}\\tab");
            TexOutput(indentBuf);
            break;
          }
          default:
          case LATEX_DESCRIPTION:
          {
            if (descriptionItemArg)
            {
              TexOutput("\\tab{\\b ");
              TraverseChildrenFromChunk(descriptionItemArg);
              TexOutput("}  ");
              descriptionItemArg = NULL;
            }
            break;
          }
	}
      }
    }
    break;
  }
  case ltTWOCOLITEM:
  case ltTWOCOLITEMRULED:
  {
    wxNode *node = itemizeStack.First();
    if (node)
    {
      ItemizeStruc *struc = (ItemizeStruc *)node->Data();
      if (start)
      {
        struc->currentItem += 1;

        int indentSize = struc->indentation;
        int oldIndent = 0;
        wxNode *node2 = itemizeStack.Nth(1);
        if (node2)
          oldIndent = ((ItemizeStruc *)node2->Data())->indentation;

        TexOutput("\n");
        if (struc->currentItem > 1)
        {
          if (currentItemSep > 0)
            TexOutput("\\par");

//          WriteEnvironmentStyles();
        }

        sprintf(buf, "\\tx%d\\li%d\\fi-%d\\ri%d\n", TwoColWidthA,
             TwoColWidthA, TwoColWidthA, TwoColWidthA+TwoColWidthB+oldIndent);
        TexOutput(buf);
      }
    }
    break;
  }
  case ltVERBATIM:
  case ltVERB:
  {
    if (start)
    {
      sprintf(buf, "{\\f3\\fs20 ");
      TexOutput(buf);
    }
    else
    {
      TexOutput("}");
      if (macroId == ltVERBATIM)
      {
        TexOutput("\\pard\n");
//        issuedNewParagraph = TRUE;
        WriteEnvironmentStyles();
      }
    }
    break;
  }
  case ltCENTERLINE:
  case ltCENTER:
  {
    if (start)
    {
      TexOutput("\\fi0\\qc ");
      forbidParindent ++;
      PushEnvironmentStyle("\\qc");
    }
    else
    {
      TexOutput("\\par\\pard\n");
      issuedNewParagraph = TRUE;
      forbidParindent --;
      PopEnvironmentStyle();
      WriteEnvironmentStyles();
    }
    break;
  }
  case ltFLUSHLEFT:
  {
    if (start)
    {
      TexOutput("\\fi0\\ql ");
      forbidParindent ++;
      PushEnvironmentStyle("\\ql");
    }
    else
    {
      TexOutput("\\par\\pard\n");
      issuedNewParagraph = TRUE;
      forbidParindent --;
      PopEnvironmentStyle();
      WriteEnvironmentStyles();
    }
    break;
  }
  case ltFLUSHRIGHT:
  {
    if (start)
    {
      TexOutput("\\fi0\\qr ");
      forbidParindent ++;
      PushEnvironmentStyle("\\qr");
    }
    else
    {
      TexOutput("\\par\\pard\n");
      issuedNewParagraph = TRUE;
      forbidParindent --;
      PopEnvironmentStyle();
      WriteEnvironmentStyles();
    }
    break;
  }
  case ltSMALL:
  {
    if (start)
    {
      sprintf(buf, "{\\fs%d\n", smallFont*2);
      TexOutput(buf);
    }
    else TexOutput("}\n");
    break;
  }
  case ltTINY:
  {
    if (start)
    {
      sprintf(buf, "{\\fs%d\n", tinyFont*2);
      TexOutput(buf);
    }
    else TexOutput("}\n");
    break;
  }
  case ltNORMALSIZE:
  {
    if (start)
    {
      sprintf(buf, "{\\fs%d\n", normalFont*2);
      TexOutput(buf);
    }
    else TexOutput("}\n");
    break;
  }
  case ltlarge:
  {
    if (start)
    {
      sprintf(buf, "{\\fs%d\n", largeFont1*2);
      TexOutput(buf);
    }
    else TexOutput("}\n");
    break;
  }
  case ltLarge:
  {
    if (start)
    {
      sprintf(buf, "{\\fs%d\n", LargeFont2*2);
      TexOutput(buf);
    }
    else TexOutput("}\n");
    break;
  }
  case ltLARGE:
  {
    if (start)
    {
      sprintf(buf, "{\\fs%d\n", LARGEFont3*2);
      TexOutput(buf);
    }
    else TexOutput("}\n");
    break;
  }
  case lthuge:
  {
    if (start)
    {
      sprintf(buf, "{\\fs%d\n", hugeFont1*2);
      TexOutput(buf);
    }
    else TexOutput("}\n");
    break;
  }
  case ltHuge:
  {
    if (start)
    {
      sprintf(buf, "{\\fs%d\n", HugeFont2*2);
      TexOutput(buf);
    }
    else TexOutput("}\n");
    break;
  }
  case ltHUGE:
  {
    if (start)
    {
      sprintf(buf, "{\\fs%d\n", HUGEFont3*2);
      TexOutput(buf);
    }
    else TexOutput("}\n");
    break;
  }
  case ltBF:
  {
    if (start)
    {
      TexOutput("{\\b ");
    }
    else TexOutput("}");
    break;
  }
  case ltUNDERLINE:
  {
    if (start)
    {
      TexOutput("{\\ul ");
    }
    else TexOutput("}");
    break;
  }
  case ltIT:
  case ltEM:
  {
    if (start)
    {
      TexOutput("{\\i ");
    }
    else TexOutput("}");
    break;
  }
  case ltRM:
  {
/*    
    if (start)
    {
      TexOutput("{\\plain ");
    }
    else TexOutput("}");
 */
    break;
  }
  case ltSC:
  {
    if (start)
    {
      TexOutput("{\\scaps ");
    }
    else TexOutput("}");
    break;
  }
  case ltTT:
  {
    if (start)
    {
      TexOutput("{\\f3 ");
    }
    else TexOutput("}");
    break;
  }
  case ltLBRACE:
  {
    if (start)
      TexOutput("\\{");
    break;
  }
  case ltRBRACE:
  {
    if (start)
      TexOutput("\\}");
    break;
  }
  case ltBACKSLASH:
  {
    if (start)
      TexOutput("\\\\");
    break;
  }
  case ltPAR:
  {
    if (start)
    {
      if (!issuedNewParagraph)
        TexOutput("\\par\\pard");

      issuedNewParagraph = FALSE;

      // Extra par if parskip is more than zero (usually looks best.)
      if (!inTable && (ParSkip > 0))
        TexOutput("\\par");

      WriteEnvironmentStyles();
      TexOutput("\n");
    }
    break;
  }
  case ltNEWPAGE:
  {
    // In Windows Help, no newpages until we've started some chapters or sections
    if (!(winHelp && !startedSections))
      if (start)
        TexOutput("\\page\n");
    break;
  }
  case ltMAKETITLE:
  {
    if (start && DocumentTitle)
    {
      TexOutput("\\par\\pard");
      if (!winHelp)
        TexOutput("\\par");
      sprintf(buf, "\\qc{\\fs%d\\b ", titleFont*2);
      TexOutput(buf);
      TraverseChildrenFromChunk(DocumentTitle);
      TexOutput("}\\par\\pard\n");

      if (DocumentAuthor)
      {
        if (!winHelp)
          TexOutput("\\par");
        sprintf(buf, "\\par\\qc{\\fs%d ", authorFont*2);
        TexOutput(buf);
        TraverseChildrenFromChunk(DocumentAuthor);
        TexOutput("}");
        TexOutput("\\par\\pard\n");
      }
      if (DocumentDate)
      {
        TexOutput("\\par");
        sprintf(buf, "\\qc{\\fs%d ", authorFont*2);
        TexOutput(buf);
        TraverseChildrenFromChunk(DocumentDate);
        TexOutput("}\\par\\pard\n");
      }
      // If linear RTF, we want this titlepage to be in a separate
      // section with its own (blank) header and footer
      if (!winHelp && (DocumentStyle != LATEX_ARTICLE))
      {
        TexOutput("{\\header }{\\footer }\\sect\n");
      }
    }
    break;
  }
  case ltADDCONTENTSLINE:
  {
    if (!start)
    {
    if (contentsLineSection && contentsLineValue)
    {
      if (strcmp(contentsLineSection, "chapter") == 0)
      {
        fprintf(Contents, "\\par\n{\\b %s}\\par\n", contentsLineValue);
      }
      else if (strcmp(contentsLineSection, "section") == 0)
      {
        if (DocumentStyle != LATEX_ARTICLE)
          fprintf(Contents, "\n\\tab%s\\par\n", contentsLineValue);
        else
          fprintf(Contents, "\\par\n{\\b %s}\\par\n", contentsLineValue);
      }
    }
    }
    break;
  }
  case ltHRULE:
  {
    if (start)
    {
      TexOutput("\\brdrb\\brdrs\\par\\pard\n");
      issuedNewParagraph = TRUE;
      WriteEnvironmentStyles();
    }
    break;
  }
  case ltRULE:
  {
    if (start)
    {
      TexOutput("\\brdrb\\brdrs\\par\\pard\n");
      issuedNewParagraph = TRUE;
      WriteEnvironmentStyles();
    }
    break;
  }
  case ltHLINE:
  {
    if (start)
      ruleTop ++;
    break;
  }
  case ltNUMBEREDBIBITEM:
  {
    if (start)
      TexOutput("\\li260\\fi-260 "); // Indent from 2nd line
    else
      TexOutput("\\par\\pard\\par\n\n");
    break;
  }
  case ltTHEPAGE:
  {
    if (start)
    {
      TexOutput("{\\field{\\*\\fldinst PAGE \\\\* MERGEFORMAT }{\\fldrslt 1}}");
    }
    break;
  }
  case ltTHECHAPTER:
  {
    if (start)
    {
//      TexOutput("{\\field{\\*\\fldinst SECTION \\\\* MERGEFORMAT }{\\fldrslt 1}}");
      sprintf(buf, "%d", chapterNo);
      TexOutput(buf);
    }
    break;
  }
  case ltTHESECTION:
  {
    if (start)
    {
//      TexOutput("{\\field{\\*\\fldinst SECTION \\\\* MERGEFORMAT }{\\fldrslt 1}}");
      sprintf(buf, "%d", sectionNo);
      TexOutput(buf);
    }
    break;
  }
  case ltTWOCOLUMN:
  {
    if (!start && !winHelp)
    {
      TexOutput("\\cols2\n");
    }
    break;
  }
  case ltONECOLUMN:
  {
    if (!start && !winHelp)
    {
      TexOutput("\\cols1\n");
    }
    break;
  }
  case ltPRINTINDEX:
  {
    if (start && !winHelp)
    {
      TexOutput("\\par\\field{\\*\\fldinst INDEX \\h \"\\emdash A\\emdash \"\\\\c \"2\"}\n");
    }
  }
  default:
  {
    DefaultOnMacro(macroId, no_args, start);
    break;
  }
  }
}

// Called on start/end of argument examination
Bool RTFOnArgument(int macroId, int arg_no, Bool start)
{
  char buf[300];
  switch (macroId)
  {
  case ltCHAPTER:
  case ltCHAPTERSTAR:
  case ltCHAPTERHEADING:
  case ltSECTION:
  case ltSECTIONSTAR:
  case ltSECTIONHEADING:
  case ltSUBSECTION:
  case ltSUBSECTIONSTAR:
  case ltSUBSUBSECTION:
  case ltSUBSUBSECTIONSTAR:
  case ltGLOSS:
  case ltMEMBERSECTION:
  case ltFUNCTIONSECTION:
  {
    if (!start && (arg_no == 1))
      currentSection = GetArgChunk();
    break;
  }
  case ltFUNC:
  {
    if (start && (arg_no == 1))
      TexOutput("\\pard\\li260\\fi-260{\\b ");

    if (!start && (arg_no == 1))
      TexOutput("} ");

    if (start && (arg_no == 2))
    {
      if (!suppressNameDecoration) TexOutput("{\\b ");
      currentMember = GetArgChunk();
    }
    if (!start && (arg_no == 2))
    {
      if (!suppressNameDecoration) TexOutput("}");
    }
    
    if (start && (arg_no == 3))
      TexOutput("(");
    if (!start && (arg_no == 3))
    {
      TexOutput(")\\li0\\fi0");
      WriteEnvironmentStyles();
    }
    break;
  }
  case ltCLIPSFUNC:
  {
    if (start && (arg_no == 1))
      TexOutput("\\pard\\li260\\fi-260{\\b ");
    if (!start && (arg_no == 1))
      TexOutput("} ");

    if (start && (arg_no == 2))
    {
      if (!suppressNameDecoration) TexOutput("({\\b ");
      currentMember = GetArgChunk();
    }
    if (!start && (arg_no == 2))
    {
      if (!suppressNameDecoration) TexOutput("}");
    }

    if (!start && (arg_no == 3))
    {
      TexOutput(")\\li0\\fi0");
      WriteEnvironmentStyles();
    }
    break;
  }
  case ltPFUNC:
  {
    if (start && (arg_no == 1))
      TexOutput("\\pard\\li260\\fi-260");

    if (!start && (arg_no == 1))
      TexOutput(" ");

    if (start && (arg_no == 2))
      TexOutput("(*");
    if (!start && (arg_no == 2))
      TexOutput(")");

    if (start && (arg_no == 2))
      currentMember = GetArgChunk();

    if (start && (arg_no == 3))
      TexOutput("(");
    if (!start && (arg_no == 3))
    {
      TexOutput(")\\li0\\fi0");
      WriteEnvironmentStyles();
    }
    break;
  }
  case ltPARAM:
  {
    if (start && (arg_no == 1))
      TexOutput("{\\b ");
    if (!start && (arg_no == 1))
      TexOutput("}");
    if (start && (arg_no == 2))
    {
      TexOutput("{\\i ");
    }
    if (!start && (arg_no == 2))
    {
      TexOutput("}");
    }
    break;
  }
  case ltCPARAM:
  {
    if (start && (arg_no == 1))
      TexOutput("{\\b ");
    if (!start && (arg_no == 1))
      TexOutput("} ");  // This is the difference from param - one space!
    if (start && (arg_no == 2))
    {
      TexOutput("{\\i ");
    }
    if (!start && (arg_no == 2))
    {
      TexOutput("}");
    }
    break;
  }
  case ltMEMBER:
  {
    if (!start && (arg_no == 1))
      TexOutput(" ");

    if (start && (arg_no == 2))
      currentMember = GetArgChunk();
    break;
  }
  case ltREF:
  {
    if (start)
    {
      char *sec = NULL;
      
      char *refName = GetArgData();
      if (refName)
      {
        TexRef *texRef = FindReference(refName);
        if (texRef)
        {
          sec = texRef->sectionNumber;
        }
      }
      if (sec)
      {
        TexOutput(sec);
      }
      return FALSE;
    }
    break;
  }
  case ltHELPREF:
  case ltHELPREFN:
  {
    if (winHelp)
    {
        if ((GetNoArgs() - arg_no) == 1)
        {
          if (start)
            TexOutput("{\\uldb ");
          else
            TexOutput("}");
        }
        if ((GetNoArgs() - arg_no) == 0) // Arg = 2, or 3 if first is optional
        {
          if (start)
            TexOutput("{\\v ");
          else TexOutput("}");
        }
    }
    else // If a linear document, must resolve the references ourselves
    {
      if ((GetNoArgs() - arg_no) == 1)
      {
        // In a linear document we display the anchor text in bold plus
        // the section/figure number.
        if (start)
          TexOutput("{\\b ");
        else
          TexOutput("}");
        return TRUE;
      }
      else if ((GetNoArgs() - arg_no) == 0) // Arg = 2, or 3 if first is optional
      {
        if (macroId != ltHELPREFN)
        {
          if (start)
          {
            TexOutput(" (");
            char *refName = GetArgData();
            if (refName)
            {
              TexRef *texRef = FindReference(refName);
              if (texRef)
              {
                TexOutput(texRef->sectionNumber);
              }
              else
              {
                TexOutput("??");
                sprintf(buf, "Warning: unresolved reference %s.", refName); 
                OnInform(buf);
              }
            }
            else TexOutput("??");
          }
          else TexOutput(")");
        }
        return FALSE;
      }
    }
    break;
  }
  case ltPOPREF:
  {
    if (winHelp)
    {
      if ((GetNoArgs() - arg_no) == 1)
      {
        if (start)
          TexOutput("{\\ul ");
        else
          TexOutput("}");
      }
      if ((GetNoArgs() - arg_no) == 0) // Arg = 2, or 3 if first is optional
      {
        if (start)
          TexOutput("{\\v ");
        else TexOutput("}");
      }
    }
    else // A linear document...
    {
      if ((GetNoArgs() - arg_no) == 1)
      {
        // In a linear document we just display the anchor text in italic
        if (start)
          TexOutput("{\\i ");
        else
          TexOutput("}");
        return TRUE;
      }
      else return FALSE;
    }
    break;
  }
  case ltADDCONTENTSLINE:
  {
    if (start && !winHelp)
    {
      if (arg_no == 2)
        contentsLineSection = copystring(GetArgData());
      else if (arg_no == 3)
        contentsLineValue = copystring(GetArgData());
      return FALSE;
    }
    else return FALSE;
    break;
  }
  case ltIMAGE:
  case ltPSBOXTO:
  {
    static int imageWidth = NULL;
    static int imageHeight = NULL;
    
    if (start && (arg_no == 1))
    {
      char *imageDimensions = copystring(GetArgData());
      char buf1[50];
      strcpy(buf1, imageDimensions);
      char *tok1 = strtok(buf1, ";:");
      char *tok2 = strtok(NULL, ";:");
      // Convert points to TWIPS (1 twip = 1/20th of point)
      imageWidth = (int)(20*(tok1 ? ParseUnitArgument(tok1) : 0));
      imageHeight = (int)(20*(tok2 ? ParseUnitArgument(tok2) : 0));
    }  

    if (start && (arg_no == 2 ))
    {
      char *filename = copystring(GetArgData());
      strcpy(buf, filename);
      StripExtension(buf);
      strcat(buf, ".bmp");
      char *f = TexPathList.FindValidPath(buf);
      if (winHelp)
      {
        if (!f) // Try for a metafile instead
        {
          strcpy(buf, filename);
          StripExtension(buf);
          strcat(buf, ".wmf");
          f = TexPathList.FindValidPath(buf);
        }
        if (f)
        {
          TexOutput("\\{bmc ");
          TexOutput(FileNameFromPath(f));
          TexOutput("\\}");
        }
        else
        {
          TexOutput("[No BMP or MF for image file ");
          TexOutput(filename);
          TexOutput("]");
          sprintf(buf, "Warning: could not find a BMP or MF equivalent for %s.", filename);
          OnInform(buf);
        }
      }
      else // If linear RTF, no metafile support -- try for BMP only.
      {
        if (f)
        {
          FILE *fd = fopen(f, "rb");
          if (OutputBitmapHeader(fd, winHelp))
            OutputBitmapData(fd);
        }
        else // Try for a metafile instead
        {
#ifdef wx_msw
          strcpy(buf, filename);
          StripExtension(buf);
          strcat(buf, ".wmf");
          f = TexPathList.FindValidPath(buf);
          if (f)
          {
//            HFILE handle = _lopen(f, READ);
            FILE *fd = fopen(f, "rb");
            if (OutputMetafileHeader(fd, winHelp, imageWidth, imageHeight))
              OutputMetafileData(fd);
          }
          else
          {
#endif            
            TexOutput("[No BMP or WMF for image file ");
            TexOutput(filename);
            TexOutput("]");
            sprintf(buf, "Warning: could not find a BMP or WMF equivalent for %s.", filename);
            OnInform(buf);
#ifdef wx_msw            
          }
#endif
        }
      }
      return FALSE;
    }
    else
      return FALSE;
    break;
  }
  case ltTABULAR:
  case ltSUPERTABULAR:
  {
    if (arg_no == 1)
    {
      if (start)
      {
        currentRowNumber = 0;
        inTable = TRUE;
        startRows = TRUE;
        tableVerticalLineLeft = FALSE;
        tableVerticalLineRight = FALSE;
        int currentWidth = 0;

        char *alignString = copystring(GetArgData());
        ParseTableArgument(alignString);

//        TexOutput("\\trowd\\trgaph108\\trleft-108");
        TexOutput("\\trowd\\trgaph108");

        // Write the first row formatting for compatibility
        // with standard Latex
        if (compatibilityMode)
        {
          for (int i = 0; i < noColumns; i++)
          {
            currentWidth += TableData[i].width;
            sprintf(buf, "\\cellx%d", currentWidth);
            TexOutput(buf);
          }
          TexOutput("\\pard\\intbl\n");
        }

        return FALSE;
      }
    }
    else if (arg_no == 2 && !start)
    {
      TexOutput("\\pard\n");
      WriteEnvironmentStyles();
      inTable = FALSE;
    }
    break;
  }

/*
  case ltTABULAR:
  case ltSUPERTABULAR:
  {
    if (arg_no == 1)
    {
      if (start)
      {
        inTable = TRUE;
        startRows = TRUE;
        tableVerticalLineLeft = FALSE;
        tableVerticalLineRight = FALSE;

        char *alignString = copystring(GetArgData());

        // Count the number of columns
        noColumns = 0;
        int len = strlen(alignString);
        if (len > 0)
        {
          if (alignString[0] == '|')
            tableVerticalLineLeft = TRUE;
          if (alignString[len-1] == '|')
            tableVerticalLineRight = TRUE;
        }

        for (int i = 0; i < len; i++)
          if (isalpha(alignString[i]))
            noColumns ++;

        // Experimental
        TexOutput("\\brdrt\\brdrs");
        if (tableVerticalLineLeft)
          TexOutput("\\brdrl\\brdrs");
        if (tableVerticalLineRight)
        TexOutput("\\brdrr\\brdrs");

        // Calculate a rough size for each column
        int colSize = 6000/(noColumns-1);
        int colPos = colSize;
        for (int j = 0; j < noColumns; j++)
        {
          sprintf(buf, "\\tx%d ", colPos);
          TexOutput(buf);
          colPos += colSize;
        }

        return FALSE;
      }
    }
    else if (arg_no == 2 && !start)
    {
      TexOutput("\\par\\pard\n");
      WriteEnvironmentStyles();
      inTable = FALSE;
    }
    break;
  }
*/
  case ltQUOTE:
  case ltVERSE:
  {
    if (start)
    {
      TexOutput("\\li360\n");
      forbidParindent ++;
      PushEnvironmentStyle("\\li360");
    }
    else
    {
      TexOutput("\\par\\pard\n");
      issuedNewParagraph = TRUE;
      forbidParindent --;
      PopEnvironmentStyle();
      WriteEnvironmentStyles();
    }
    break;
  }
  case ltQUOTATION:
  {
    if (start)
    {
      TexOutput("\\li360\n");
      PushEnvironmentStyle("\\li360");
    }
    else
    {
      TexOutput("\\par\\pard\n");
      issuedNewParagraph = TRUE;
      PopEnvironmentStyle();
      WriteEnvironmentStyles();
    }
    break;
  }
  case ltBOXIT:
  case ltFRAMEBOX:
  case ltFBOX:
  case ltNORMALBOX:
  case ltNORMALBOXD:
  {
    if (start)
    {
      sprintf(buf, "\\box\\trgaph108%s\n", ((macroId == ltNORMALBOXD) ? "\\brdrdb" : "\\brdrs"));
      TexOutput(buf);
      PushEnvironmentStyle(buf);
    }
    else
    {
      TexOutput("\\par\\pard\n");
      issuedNewParagraph = TRUE;
      PopEnvironmentStyle();
      WriteEnvironmentStyles();
    }
    break;
  }
  case ltHELPFONTSIZE:
  {
    if (start)
    {
      char *data = GetArgData();
      if (strcmp(data, "10") == 0)
        SetFontSizes(10);
      else if (strcmp(data, "11") == 0)
        SetFontSizes(11);
      else if (strcmp(data, "12") == 0)
        SetFontSizes(12);
      sprintf(buf, "\\fs%d\n", normalFont*2);
      TexOutput(buf);
      TexOutput(buf);
      return FALSE;
    }
    break;
  }
  case ltHELPFONTFAMILY:
  {
    if (start)
    {
      char *data = GetArgData();
      if (strcmp(data, "Swiss") == 0)
        TexOutput("\\f2\n");
      else if (strcmp(data, "Symbol") == 0)
        TexOutput("\\f1\n");
      else if (strcmp(data, "Times") == 0)
        TexOutput("\\f0\n");

      return FALSE;
    }
    break;
  }
  case ltPARINDENT:
  {
    if (start && arg_no == 1)
    {
      char *data = GetArgData();
      ParIndent = ParseUnitArgument(data);
      if (ParIndent == 0 || forbidParindent == 0)
      {
        sprintf(buf, "\\fi%d\n", ParIndent*20);
        TexOutput(buf);
      }
      return FALSE;
    }
    break;
  }
  case ltITEM:
  {
    if (start && IsArgOptional())
    {
      descriptionItemArg = GetArgChunk();
      return FALSE;
    }
    break;
  }
  case ltTWOCOLITEM:
  case ltTWOCOLITEMRULED:
  {
    switch (arg_no)
    {
      case 1:
      {
        if (!start)
          TexOutput("\\tab ");
        break;
      }
      case 2:
      {
        if (!start)
        {
          if (macroId == ltTWOCOLITEMRULED)
            TexOutput("\\brdrb\\brdrs\\brdrw15\\brsp20 ");
          TexOutput("\\par\\pard\n");
          issuedNewParagraph = TRUE;
          WriteEnvironmentStyles();
        }
        break;
      }
    }
    return TRUE;
    break;
  }
  /*
   * Accents
   *
   */
  case ltACCENT_GRAVE:
  {
    if (start)
    {
      char *val = GetArgData();
      if (val)
      {
        switch (val[0])
        {
          case 'a':
           TexOutput("\\'e0");
           break;
          case 'e':
           TexOutput("\\'e8");
           break;
          case 'i':
           TexOutput("\\'ec");
           break;
          case 'o':
           TexOutput("\\'f2");
           break;
          case 'u':
           TexOutput("\\'f9");
           break;
          case 'A':
           TexOutput("\\'c0");
           break;
          case 'E':
           TexOutput("\\'c8");
           break;
          case 'I':
           TexOutput("\\'cc");
           break;
          case 'O':
           TexOutput("\\'d2");
           break;
          case 'U':
           TexOutput("\\'d9");
           break;
          default:
           break;
        }
      }
    }
    return FALSE;
    break;
  }
  case ltACCENT_ACUTE:
  {
    if (start)
    {
      char *val = GetArgData();
      if (val)
      {
        switch (val[0])
        {
          case 'a':
           TexOutput("\\'e1");
           break;
          case 'e':
           TexOutput("\\'e9");
           break;
          case 'i':
           TexOutput("\\'ed");
           break;
          case 'o':
           TexOutput("\\'f3");
           break;
          case 'u':
           TexOutput("\\'fa");
           break;
          case 'y':
           TexOutput("\\'fd");
           break;
          case 'A':
           TexOutput("\\'c1");
           break;
          case 'E':
           TexOutput("\\'c9");
           break;
          case 'I':
           TexOutput("\\'cd");
           break;
          case 'O':
           TexOutput("\\'d3");
           break;
          case 'U':
           TexOutput("\\'da");
           break;
          case 'Y':
           TexOutput("\\'dd");
           break;
          default:
           break;
        }
      }
    }
    return FALSE;
    break;
  }
  case ltACCENT_CARET:
  {
    if (start)
    {
      char *val = GetArgData();
      if (val)
      {
        switch (val[0])
        {
          case 'a':
           TexOutput("\\'e2");
           break;
          case 'e':
           TexOutput("\\'ea");
           break;
          case 'i':
           TexOutput("\\'ee");
           break;
          case 'o':
           TexOutput("\\'f4");
           break;
          case 'u':
           TexOutput("\\'fb");
           break;
          case 'A':
           TexOutput("\\'c2");
           break;
          case 'E':
           TexOutput("\\'ca");
           break;
          case 'I':
           TexOutput("\\'ce");
           break;
          case 'O':
           TexOutput("\\'d4");
           break;
          case 'U':
           TexOutput("\\'db");
           break;
          default:
           break;
        }
      }
    }
    return FALSE;
    break;
  }
  case ltACCENT_TILDE:
  {
    if (start)
    {
      char *val = GetArgData();
      if (val)
      {
        switch (val[0])
        {
          case 'a':
           TexOutput("\\'e3");
           break;
          case 'n':
           TexOutput("\\'f1");
           break;
          case 'o':
           TexOutput("\\'f5");
           break;
          case 'A':
           TexOutput("\\'c3");
           break;
          case 'N':
           TexOutput("\\'d1");
           break;
          case 'O':
           TexOutput("\\'d5");
           break;
          default:
           break;
        }
      }
    }
    return FALSE;
    break;
  }
  case ltACCENT_UMLAUT:
  {
    if (start)
    {
      char *val = GetArgData();
      if (val)
      {
        switch (val[0])
        {
          case 'a':
           TexOutput("\\'e4");
           break;
          case 'e':
           TexOutput("\\'eb");
           break;
          case 'i':
           TexOutput("\\'ef");
           break;
          case 'o':
           TexOutput("\\'f6");
           break;
          case 'u':
           TexOutput("\\'fc");
           break;
          case 'y':
           TexOutput("\\'ff");
           break;
          case 'A':
           TexOutput("\\'c4");
           break;
          case 'E':
           TexOutput("\\'cb");
           break;
          case 'I':
           TexOutput("\\'cf");
           break;
          case 'O':
           TexOutput("\\'d6");
           break;
          case 'U':
           TexOutput("\\'dc");
           break;
          case 'Y':
           TexOutput("\\'df");
           break;
          default:
           break;
        }
      }
    }
    return FALSE;
    break;
  }
  case ltACCENT_DOT:
  {
    if (start)
    {
      char *val = GetArgData();
      if (val)
      {
        switch (val[0])
        {
          case 'a':
           TexOutput("\\'e5");
           break;
          case 'A':
           TexOutput("\\'c5");
           break;
          default:
           break;
        }
      }
    }
    return FALSE;
    break;
  }
  case ltACCENT_CADILLA:
  {
    if (start)
    {
      char *val = GetArgData();
      if (val)
      {
        switch (val[0])
        {
          case 'c':
           TexOutput("\\'e7");
           break;
          case 'C':
           TexOutput("\\'c7");
           break;
          default:
           break;
        }
      }
    }
    return FALSE;
    break;
  }
  case ltFOOTNOTE:
  {
    static char *helpTopic = NULL;
    static FILE *savedOutput = NULL;
    if (winHelp)
    {
      if (arg_no == 1)
      {
        if (start)
        {
          OnInform("Consider using \\footnotepopup instead of \\footnote.");
          footnoteCount ++;
          char footBuf[20];
          sprintf(footBuf, "(%d)", footnoteCount);

          TexOutput(" {\\ul ");
          TexOutput(footBuf);
          TexOutput("}");
          helpTopic = FindTopicName(NULL);
          TexOutput("{\\v ");
          TexOutput(helpTopic);
          TexOutput("}");

          fprintf(Popups, "\\page\n");
//          fprintf(Popups, "\n${\\footnote }"); // No title
          fprintf(Popups, "\n#{\\footnote %s}\n", helpTopic);
          fprintf(Popups, "+{\\footnote %s}\n", GetBrowseString());
          savedOutput = CurrentOutput1;
          SetCurrentOutput(Popups);
	}
	else
	{
          SetCurrentOutput(savedOutput);
	}
	return TRUE;
      }
      return TRUE;
    }
    else
    {
      if (start)
      {
        TexOutput(" {\\super \\chftn{\\footnote \\fs20 {\\super \\chftn}", TRUE);
      }
      else
      {
        TexOutput("}}", TRUE);
      }
      return TRUE;
    }
    break;
  }
  case ltFOOTNOTEPOPUP:
  {
    static char *helpTopic = NULL;
    static FILE *savedOutput = NULL;
    if (winHelp)
    {
      if (arg_no == 1)
      {
        if (start)
        {
          TexOutput("{\\ul ");
	}
	else TexOutput("}");
	return TRUE;
      }
      else if (arg_no == 2)
      {
        if (start)
        {
          helpTopic = FindTopicName(NULL);
          TexOutput("{\\v ");
          TexOutput(helpTopic);
          TexOutput("}");

          fprintf(Popups, "\\page\n");
//          fprintf(Popups, "\n${\\footnote }"); // No title
          fprintf(Popups, "\n#{\\footnote %s}\n", helpTopic);
          fprintf(Popups, "+{\\footnote %s}\n", GetBrowseString());
          savedOutput = CurrentOutput1;
          SetCurrentOutput(Popups);
        }
        else
        {
          SetCurrentOutput(savedOutput);
        }
        return TRUE;
      }
    }
    else
    {
      if (arg_no == 1)
        return TRUE;
      if (start)
      {
        TexOutput(" {\\super \\chftn{\\footnote \\fs20 {\\super \\chftn}", TRUE);
      }
      else
      {
        TexOutput("}}", TRUE);
      }
      return TRUE;
    }
    break;
  }
  case ltFANCYPLAIN:
  {
    if (start && (arg_no == 1))
      return FALSE;
    else
      return TRUE;
    break;
  }
  case ltSETHEADER:
  {
    if (winHelp) return FALSE;
    if (start)
    {
      switch (arg_no)
      {
        case 1:
          LeftHeaderEven = GetArgChunk();
          if (strlen(GetArgData(LeftHeaderEven)) == 0)
            LeftHeaderEven = NULL;
          break;
        case 2:
          CentreHeaderEven = GetArgChunk();
          if (strlen(GetArgData(CentreHeaderEven)) == 0)
            CentreHeaderEven = NULL;
          break;
        case 3:
          RightHeaderEven = GetArgChunk();
          if (strlen(GetArgData(RightHeaderEven)) == 0)
            RightHeaderEven = NULL;
          break;
        case 4:
          LeftHeaderOdd = GetArgChunk();
          if (strlen(GetArgData(LeftHeaderOdd)) == 0)
            LeftHeaderOdd = NULL;
          break;
        case 5:
          CentreHeaderOdd = GetArgChunk();
          if (strlen(GetArgData(CentreHeaderOdd)) == 0)
            CentreHeaderOdd = NULL;
          break;
        case 6:
          RightHeaderOdd = GetArgChunk();
          if (strlen(GetArgData(RightHeaderOdd)) == 0)
            RightHeaderOdd = NULL;
          OutputRTFHeaderCommands();
          break;
        default:
          break;
      }
    }
    return FALSE;
    break;
  }
  case ltSETFOOTER:
  {
    if (winHelp) return FALSE;
    if (start)
    {
      switch (arg_no)
      {
        case 1:
          LeftFooterEven = GetArgChunk();
          if (strlen(GetArgData(LeftFooterEven)) == 0)
            LeftFooterEven = NULL;
          break;
        case 2:
          CentreFooterEven = GetArgChunk();
          if (strlen(GetArgData(CentreFooterEven)) == 0)
            CentreFooterEven = NULL;
          break;
        case 3:
          RightFooterEven = GetArgChunk();
          if (strlen(GetArgData(RightFooterEven)) == 0)
            RightFooterEven = NULL;
          break;
        case 4:
          LeftFooterOdd = GetArgChunk();
          if (strlen(GetArgData(LeftFooterOdd)) == 0)
            LeftFooterOdd = NULL;
          break;
        case 5:
          CentreFooterOdd = GetArgChunk();
          if (strlen(GetArgData(CentreFooterOdd)) == 0)
            CentreFooterOdd = NULL;
          break;
        case 6:
          RightFooterOdd = GetArgChunk();
          if (strlen(GetArgData(RightFooterOdd)) == 0)
            RightFooterOdd = NULL;
          OutputRTFFooterCommands();
          break;
        default:
          break;
      }
    }
    return FALSE;
    break;
  }
  case ltMARKRIGHT:
  {
    if (winHelp) return FALSE;
    // Fake a SetHeader command
    if (start)
    {
      LeftHeaderOdd = NULL;
      CentreHeaderOdd = NULL;
      RightHeaderOdd = NULL;
      LeftHeaderEven = NULL;
      CentreHeaderEven = NULL;
      RightHeaderEven = NULL;
      OnInform("Consider using setheader/setfooter rather than markright.");
    }
    RTFOnArgument(ltSETHEADER, 4, start);
    if (!start)
      OutputRTFHeaderCommands();
    return FALSE;
    break;
  }
  case ltMARKBOTH:
  {
    if (winHelp) return FALSE;
    // Fake a SetHeader command
    switch (arg_no)
    {
      case 1:
      {
        if (start)
        {
          LeftHeaderOdd = NULL;
          CentreHeaderOdd = NULL;
          RightHeaderOdd = NULL;
          LeftHeaderEven = NULL;
          CentreHeaderEven = NULL;
          RightHeaderEven = NULL;
          OnInform("Consider using setheader/setfooter rather than markboth.");
        }
        return RTFOnArgument(ltSETHEADER, 1, start);
        break;
      }
      case 2:
      {
        RTFOnArgument(ltSETHEADER, 4, start);
        if (!start)
          OutputRTFHeaderCommands();
        return FALSE;
        break;
      }
    }
    break;
  }
  case ltPAGENUMBERING:
  {
    if (winHelp) return FALSE;
    if (start)
    {
      TexOutput("\\pgnrestart");
      char *data = GetArgData();
      if (currentNumberStyle) delete[] currentNumberStyle;
      currentNumberStyle = copystring(data);
      if (strcmp(data, "arabic") == 0)
      {
        TexOutput("\\pgndec");
      }
      else if (strcmp(data, "roman") == 0)
      {
        TexOutput("\\pgnlcrm");
      }
      else if (strcmp(data, "Roman") == 0)
      {
        TexOutput("\\pgnucrm");
      }
      else if (strcmp(data, "alph") == 0)
      {
        TexOutput("\\pgnlcltr");
      }
      else if (strcmp(data, "Alph") == 0)
      {
        TexOutput("\\pgnucltr");
      }
      TexOutput("\n");
    }
    return FALSE;
    break;
  }
  case ltTWOCOLUMN:
  {
    if (winHelp) return FALSE;
    if (start)
      return TRUE;
    break;
  }
  case ltITEMSEP:
  {
    if (start)
    {
      char *val = GetArgData();
      currentItemSep = ParseUnitArgument(val);
      return FALSE;
    }
    break;
  }
  case ltEVENSIDEMARGIN:
  {
    return FALSE;
    break;
  }
  case ltODDSIDEMARGIN:
  {
    if (start)
    {
      char *val = GetArgData();
      int twips = (int)(20*ParseUnitArgument(val));
      // Add an inch since in LaTeX it's specified minus an inch
      twips += 1440;
      CurrentLeftMarginOdd = twips;
      sprintf(buf, "\\margl%d\n", twips);
      TexOutput(buf);

      CurrentMarginParX = CurrentLeftMarginOdd + CurrentTextWidth + CurrentMarginParSep;
    }
    return FALSE;
  }
  case ltMARGINPARWIDTH:
  {
    if (start)
    {
      char *val = GetArgData();
      int twips = (int)(20*ParseUnitArgument(val));
      CurrentMarginParWidth = twips;
    }
    return FALSE;
  }
  case ltMARGINPARSEP:
  {
    if (start)
    {
      char *val = GetArgData();
      int twips = (int)(20*ParseUnitArgument(val));
      CurrentMarginParSep = twips;
      CurrentMarginParX = CurrentLeftMarginOdd + CurrentTextWidth + CurrentMarginParSep;
    }
    return FALSE;
  }
  case ltTEXTWIDTH:
  {
    if (start)
    {
      char *val = GetArgData();
      int twips = (int)(20*ParseUnitArgument(val));
      CurrentTextWidth = twips;

      // Need to set an implicit right margin
      CurrentRightMarginOdd = PageWidth - CurrentTextWidth - CurrentLeftMarginOdd;
      CurrentRightMarginEven = PageWidth - CurrentTextWidth - CurrentLeftMarginEven;
      CurrentMarginParX = CurrentLeftMarginOdd + CurrentTextWidth + CurrentMarginParSep;
      sprintf(buf, "\\margr%d\n", CurrentRightMarginOdd);
      TexOutput(buf);
    }
    return FALSE;
  }
  case ltMARGINPAR:
  case ltMARGINPARODD:
  {
    if (start)
    {
      if (winHelp)
      {
        TexOutput("\\box\n");
        PushEnvironmentStyle("\\box");
      }
      else
      {
        sprintf(buf, "\\phpg\\posx%d\\absw%d\n", CurrentMarginParX, CurrentMarginParWidth);
        TexOutput(buf);
      }
      return TRUE;
    }
    else
    {
      if (winHelp)
      {
        TexOutput("\\par\\pard\n");
        PopEnvironmentStyle();
        WriteEnvironmentStyles();
      }
      else
        TexOutput("\\par\\pard\n");
      issuedNewParagraph = TRUE;
    }
    return FALSE;
  }
  case ltMARGINPAREVEN:
  {
    if (start)
    {
      if (winHelp)
      {
        TexOutput("\\box\n");
        PushEnvironmentStyle("\\box");
      }
      else
      {
        if (mirrorMargins)
        {
          // Have to calculate what the margins are changed to in WfW margin
          // mirror mode, on an even (left-hand) page.
          int x = PageWidth - CurrentRightMarginOdd - CurrentMarginParWidth - CurrentMarginParSep
                    - CurrentTextWidth + GutterWidth;
          sprintf(buf, "\\phpg\\posx%d\\absw%d\n", x, CurrentMarginParWidth);
          TexOutput(buf);
        }
        else
        {
          sprintf(buf, "\\phpg\\posx%d\\absw%d\n", CurrentMarginParX, CurrentMarginParWidth);
          TexOutput(buf);
        }
      }
      return TRUE;
    }
    else
    {
      if (winHelp)
      {
        TexOutput("\\par\\pard\n");
        PopEnvironmentStyle();
        WriteEnvironmentStyles();
      }
      else
        issuedNewParagraph = TRUE;
      TexOutput("\\par\\pard\n");
    }
    return FALSE;
  }
  case ltTWOCOLWIDTHA:
  {
    if (start)
    {
      char *val = GetArgData();
      int twips = (int)(20*ParseUnitArgument(val));
      TwoColWidthA = twips;
    }
    return FALSE;
    break;
  }
  case ltTWOCOLWIDTHB:
  {
    if (start)
    {
      char *val = GetArgData();
      int twips = (int)(20*ParseUnitArgument(val));
      TwoColWidthB = twips;
    }
    return FALSE;
    break;
  }
  case ltROW:
  case ltRULEDROW:
  {
    if (start)
    {
      int currentWidth = 0;

      if (!compatibilityMode || (currentRowNumber > 0))
      {
      TexOutput("\\pard\\intbl");

      if (macroId == ltRULEDROW)
        ruleBottom = 1;
      for (int i = 0; i < noColumns; i++)
      {
        currentWidth += TableData[i].width;
        if (ruleTop == 1)
        {
          TexOutput("\\clbrdrt\\brdrs\\brdrw15");
        }
        else if (ruleTop > 1)
        {
          TexOutput("\\clbrdrt\\brdrdb\\brdrw15");
        }
        if (ruleBottom == 1)
        {
          TexOutput("\\clbrdrb\\brdrs\\brdrw15");
        }
        else if (ruleBottom > 1)
        {
          TexOutput("\\clbrdrb\\brdrdb\\brdrw15");
        }

        if (TableData[i].rightBorder)
          TexOutput("\\clbrdrr\\brdrs\\brdrw15");

        if (TableData[i].leftBorder)
          TexOutput("\\clbrdrl\\brdrs\\brdrw15");
          
        sprintf(buf, "\\cellx%d", currentWidth);
        TexOutput(buf);
      }
      TexOutput("\\pard\\intbl\n");
      }
      ruleTop = 0;
      ruleBottom = 0;
      currentRowNumber ++;
      return TRUE;
    }
    else
    {
//      TexOutput("\\cell\\row\\trowd\\trgaph108\\trleft-108\n");
      TexOutput("\\cell\\row\\trowd\\trgaph108\n");
    }
    break;
  }
  case ltINDENTED:
  {
    if (start && (arg_no == 1))
    {
//      indentLevel ++;
//      TexOutput("\\fi0\n");
      int oldIndent = 0;
      wxNode *node = itemizeStack.First();
      if (node)
        oldIndent = ((ItemizeStruc *)node->Data())->indentation;

      int indentValue = 20*ParseUnitArgument(GetArgData());
      int indentSize = indentValue + oldIndent;

      ItemizeStruc *struc = new ItemizeStruc(LATEX_INDENT, indentSize);
      itemizeStack.Insert(struc);
      
      sprintf(buf, "\\tx%d\\li%d ", indentSize, indentSize);
      PushEnvironmentStyle(buf);
      TexOutput(buf);
      return FALSE;
    }
    if (!start && (arg_no == 2))
    {
      PopEnvironmentStyle();
      if (itemizeStack.First())
      {
        ItemizeStruc *struc = (ItemizeStruc *)itemizeStack.First()->Data();
        delete struc;
        delete itemizeStack.First();
      }
      if (itemizeStack.Number() == 0)
      {
        TexOutput("\\par\\pard\n");
        issuedNewParagraph = TRUE;
        WriteEnvironmentStyles();
      }
    }
    return TRUE;
    break;
  }
/*
  case ltSIZEDBOX:
  case ltSIZEDBOXD:
  {
    if (start && (arg_no == 1))
    {
      int oldIndent = 0;
      wxNode *node = itemizeStack.First();
      if (node)
        oldIndent = ((ItemizeStruc *)node->Data())->indentation;

      int boxWidth = 20*ParseUnitArgument(GetArgData());

      int indentValue = (int)((CurrentTextWidth - oldIndent - boxWidth)/2.0);
      int indentSize = indentValue + oldIndent;
      int indentSizeRight = indentSize + boxWidth;

      ItemizeStruc *struc = new ItemizeStruc(LATEX_INDENT, indentSize);
      itemizeStack.Insert(struc);
      
      sprintf(buf, "\\tx%d\\li%d\\lr%d\\box%s ", indentSize, indentSize, indentSizeRight,
        ((macroId == ltCENTEREDBOX) ? "\\brdrs" : "\\brdrdb"));
      PushEnvironmentStyle(buf);
      TexOutput(buf);
      return FALSE;
    }
    if (!start && (arg_no == 2))
    {
      PopEnvironmentStyle();
      if (itemizeStack.First())
      {
        ItemizeStruc *struc = (ItemizeStruc *)itemizeStack.First()->Data();
        delete struc;
        delete itemizeStack.First();
      }
      if (itemizeStack.Number() == 0)
      {
        TexOutput("\\par\\pard\n");
        issuedNewParagraph = TRUE;
        WriteEnvironmentStyles();
      }
    }
    return TRUE;
    break;
  }
*/
  case ltDOCUMENTSTYLE:
  {
    DefaultOnArgument(macroId, arg_no, start);
    if (!start && !IsArgOptional())
    {
      if (MinorDocumentStyleString)
      {
        if (StringMatch("twoside", MinorDocumentStyleString))
          // Mirror margins, switch on odd/even headers & footers, and break sections at odd pages
          TexOutput("\\margmirror\\facingp\\sbkodd");
        if (StringMatch("twocolumn", MinorDocumentStyleString))
          TexOutput("\\cols2");
      }
      TexOutput("\n");
    }
    return FALSE;
  }
  case ltBIBITEM:
  {
    if (arg_no == 1 && start)
    {
      char *citeKey = GetArgData();
      TexRef *ref = (TexRef *)TexReferences.Get(citeKey);
      if (ref)
      {
        if (ref->sectionNumber) delete[] ref->sectionNumber;
        sprintf(buf, "[%d]", citeCount);
        ref->sectionNumber = copystring(buf);
      }

      TexOutput("\\li260\\fi-260 "); // Indent from 2nd line
      sprintf(buf, "{\\b [%d]} ", citeCount);
      TexOutput(buf);
      citeCount ++;
      return FALSE;
    }
    if (arg_no == 2 && !start)
      TexOutput("\\par\\pard\\par\n\n");
    return TRUE;
    break;
  }
  case ltTHEBIBLIOGRAPHY:
  {
    if (start && (arg_no == 1))
    {
      citeCount = 1;
      if (winHelp)
        SetCurrentOutputs(Contents, Chapters);

      if (!winHelp)
      {
        fprintf(Chapters, "\\sect\\titlepg\n");

        // If a non-custom page style, we generate the header now.
        if (PageStyle && (strcmp(PageStyle, "plain") == 0 ||
                          strcmp(PageStyle, "empty") == 0 ||
                          strcmp(PageStyle, "headings") == 0))
        {
          OutputRTFHeaderCommands();
          OutputRTFFooterCommands();
        }
        
        // Need to reset the current numbering style, or RTF forgets it.
        if (currentNumberStyle)
        {
          if (strcmp(currentNumberStyle, "arabic") == 0)
          {
            fprintf(Chapters, "\\pgndec");
          }
          else if (strcmp(currentNumberStyle, "roman") == 0)
          {
            fprintf(Chapters,"\\pgnlcrm");
          }
          else if (strcmp(currentNumberStyle, "Roman") == 0)
          {
            fprintf(Chapters,"\\pgnucrm");
          }
          else if (strcmp(currentNumberStyle, "alph") == 0)
          {
            fprintf(Chapters,"\\pgnlcltr");
          }
          else if (strcmp(currentNumberStyle, "Alph") == 0)
          {
            fprintf(Chapters,"\\pgnucltr");
          }
        }
        SetCurrentOutput(Contents);
      }
      else
        fprintf(Chapters, "\\page\n");

      if (winHelp)
        fprintf(Contents, "\n{\\uldb References}");
      else
        fprintf(Contents, "\\par\n\\pard{\\b References}");

      startedSections = TRUE;

      if (winHelp)
        fprintf(Chapters, "\n${\\footnote References}");

      char *topicName = "bibliography";

      if (winHelp)
        fprintf(Contents, "{\\v %s}\\par\\pard\n", topicName);
      else
        fprintf(Contents, "\\par\\par\\pard\n");

      if (winHelp)
      {
        fprintf(Chapters, "\n#{\\footnote %s}\n", topicName);
        fprintf(Chapters, "+{\\footnote %s}\n", GetBrowseString());
        fprintf(Chapters, "K{\\footnote References}\n");
        GenerateKeywordsForTopic(topicName);
        if (useUpButton)
        {
          fprintf(Chapters, "!{\\footnote EnableButton(\"Up\");ChangeButtonBinding(\"Up\", \"JumpId(`%s.hlp', `%s')\")}\n",
               FileNameFromPath(FileRoot), "Contents");
        }
      }
      
      SetCurrentOutput(Chapters);
      fprintf(Chapters, "\\pard{%s\\b\\fs%d References}\n",
         (winHelp ? "\\keepn" : ""),
         chapterFont*2);

      return FALSE;
    }
    return TRUE;
    break;
  }
  case ltINDEX:
  {
    /*
     * In Windows help, all keywords should be at the start of the
     * topic, but Latex \index commands can be anywhere in the text.
     * So we're going to have to build up lists of keywords for a topic,
     * and insert them on the second pass.
     *
     * In linear RTF, we can embed the index entry now.
     *
     */
    if (start)
    {
      char *entry = GetArgData();
      if (winHelp)
      {
        if (CurrentTopic)
        {
          AddKeyWordForTopic(CurrentTopic, entry);
        }
      }
      else GenerateIndexEntry(entry);
      return FALSE;
    }
    break;
  }
  default:
  {
    return DefaultOnArgument(macroId, arg_no, start);
    break;
  }
  }
  return TRUE;
}

Bool RTFGo(void)
{
  // Reset variables
  indentLevel = 0;
  forbidParindent = 0;
  contentsLineSection = NULL;
  contentsLineValue = NULL;
  descriptionItemArg = NULL;
  inTable = FALSE;
  startRows = FALSE;
  tableVerticalLineLeft = FALSE;
  tableVerticalLineRight = FALSE;
  noColumns = 0;
  startedSections = FALSE;
  inVerbatim = FALSE;
  browseId = 0;
  
  if (InputFile && OutputFile)
  {
    // Do some RTF-specific transformations on all the strings,
    // recursively
    Text2RTF(GetTopLevelChunk());

    Contents = fopen(TmpContentsName, "w");
    Chapters = fopen("chapters.rtf", "w");
    if (winHelp)
    {
      Sections = fopen("sections.rtf", "w");
      Subsections = fopen("subsections.rtf", "w");
      Subsubsections = fopen("subsubsections.rtf", "w");
      Popups = fopen("popups.rtf", "w");
    }

    WriteRTFHeader(Chapters);
    if (winHelp)
    {
      fprintf(Chapters, "\n#{\\footnote Contents}\n");
      fprintf(Chapters, "${\\footnote Contents}\n");
      fprintf(Chapters, "+{\\footnote %s}\n", GetBrowseString());
      fprintf(Chapters, "K{\\footnote Contents}\n");
      fprintf(Chapters, "!{\\footnote DisableButton(\"Up\")}\n");
    }
    if (!winHelp)
      fprintf(Contents, "\\par\\pard");
    fprintf(Contents, "{\\b\\fs%d Contents}\\par\\par\\pard\n\n",
      titleFont*2);

    // By default, Swiss, 10 point.
    fprintf(Chapters, "\\f2\\fs20\n");

    SetCurrentOutput(Chapters);

    OnInform("Converting...");
    TraverseDocument();

    if (winHelp)
    {
//      fprintf(Contents, "\\page\n");
      fprintf(Chapters, "\\page\n");
      fprintf(Sections, "\\page\n");
      fprintf(Subsections, "\\page\n");
      fprintf(Subsubsections, "\\page\n\n");
      fprintf(Popups, "\\page\n}\n");
    }

//    TexOutput("\n\\info{\\doccomm Document created by Julian Smart's Tex2RTF.}\n");
    if (!winHelp)
      TexOutput("}\n");
    fclose(Contents);
    fclose(Chapters);
    if (winHelp)
    {
      fclose(Sections);
      fclose(Subsections);
      fclose(Subsubsections);
      fclose(Popups);
    }

    if (winHelp)
    {
      wxConcatFiles("chapters.rtf", "sections.rtf", "tmp1.rtf");
      wxConcatFiles("tmp1.rtf", "subsections.rtf", "tmp2.rtf");
      wxConcatFiles("tmp2.rtf", "subsubsections.rtf", "tmp3.rtf");
      wxConcatFiles("tmp3.rtf", "popups.rtf", OutputFile);

      wxRemoveFile("tmp1.rtf");
      wxRemoveFile("tmp2.rtf");
      wxRemoveFile("tmp3.rtf");
    }
    else
    {
      if (FileExists(OutputFile)) wxRemoveFile(OutputFile);
      wxCopyFile("chapters.rtf", OutputFile);
    }
    
    if (FileExists(ContentsName)) wxRemoveFile(ContentsName);
    wxRenameFile(TmpContentsName, ContentsName);

    wxRemoveFile("chapters.rtf");
      
    if (winHelp)
    {
      wxRemoveFile("sections.rtf");
      wxRemoveFile("subsections.rtf");
      wxRemoveFile("subsubsections.rtf");
      wxRemoveFile("popups.rtf");
    }
    return TRUE;
  }
  return FALSE;
}

