/*
 * File:	wx_utils.cc
 * Purpose:	Various utilities (X version)
 * Author:	Julian Smart
 * Created:	1993
 * Updated:	
 * Copyright:	(c) 1993, AIAI, University of Edinburgh
 */

static const char sccsid[] = "%W% %G%";

/*
  Feb, 94:
  Arthur.
  Added the #include of <sys/stat.h> so that calls to mkdir(2) etc will compile.
  Also added the second arg to mkdir(2), which is the perms args. Set up
  default value.
*/

#include <iostream.h>
#include <fstream.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <sys/stat.h>
#include "common.h"

#include "wx_setup.h"
#include "wx_utils.h"
#include "wx_main.h"

#ifdef wx_motif
#if !(defined(linux) || defined(__sgi) || defined(__hpux))
#include <tiuser.h>
#endif
#endif

#include <X11/Xlib.h>
#include <X11/Xresource.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include <sys/types.h>
#include <sys/time.h>
#include <sys/syscall.h>
#include <sys/file.h>
#include <signal.h>
#include <netdb.h>
#include <dirent.h>
#include <unistd.h>
#include <pwd.h>

/*** THIS WAS GETTING TOO HARD TO UNDERSTAND! See below.
#if !(defined(SUN_CC) || defined(__hpux))
extern "C" int select(int, fd_set *, fd_set *, fd_set *, struct timeval *);
extern "C" int sigsetmask( int );
extern "C" int sigblock( int );
#ifndef SVR4
#ifdef __sgi
extern "C" int gethostname(char *host, unsigned const namelen); // Some versions of GCC prefer this
#else
extern int gethostname(char *host, unsigned const len);
// extern "C" int gethostname(char *host, int namelen); // Some versions of GCC prefer this
#endif // __sgi
#else
#include <sys/systeminfo.h>
#endif // not SVR4
#else
#ifndef __hpux
#include <sysent.h>
#endif
#endif // not (SUN_CC or HP)
*/

#if !(defined(SUN_CC) || defined(__hpux))
extern "C" int select(int, fd_set *, fd_set *, fd_set *, struct timeval *);
extern "C" int sigsetmask( int );
extern "C" int sigblock( int );
// extern "C" int gethostname(char *host, unsigned const namelen);
#endif

#ifdef SVR4
#include <sys/systeminfo.h>
#endif

// ???
// #ifdef SUN_CC
// #include <sysent.h>
// #endif

#ifdef wx_xview
#include <xview/screen.h>
#include <xview/server.h>
extern Xv_Server xview_server;
#endif

extern XrmDatabase wxResourceDatabase;

#if USE_RESOURCES
void wxXMergeDatabases(wxApp *theApp, Display *display);
#endif

static char *GetHomeDir(char *dest);

// Get hostname e.g. blane
Bool wxGetHostName(char *buf, int maxSize)
{
#ifndef SVR4
  return (gethostname(buf, maxSize) != -1);
#else
  return (sysinfo(SI_HOSTNAME, buf, maxSize) != -1);
#endif
}

// Get user ID e.g. jacs
Bool wxGetUserId(char *buf, int maxSize)
{
  struct passwd *who = getpwuid(getuid());
  if (who)
  {
    strncpy(buf, who->pw_name, maxSize - 1);
    return TRUE;
  }
  else return FALSE;
}

// Get user name e.g. Julian Smart
Bool wxGetUserName(char *buf, int maxSize)
{
  struct passwd *who = getpwuid(getuid());
  if (who)
  {
    strncpy(buf, who->pw_gecos, maxSize - 1);
    return TRUE;
  }
  else return FALSE;
}


// Execute a command (e.g. another program) in a
// system-independent manner.
// Well - not quite. Note that under UNIX, the call will only
// return immediately if an ampersand is appended to the command.
// Under Windows, the call returns immediately.
Bool wxExecute(char *command)
{
  system(command);
  return TRUE;
}

// Get a temporary filename, opening and closing the file.
void wxGetTempFileName(char *prefix, char *buf)
{
  int suffix = 0;
  while (TRUE)
  {
    sprintf(buf, "/tmp/%s%d", prefix, suffix);
    if (!FileExists(buf))
    {
      FILE *fd = fopen(buf, "w");
      if (fd) fclose(fd);
      return;
    }
    suffix ++;
    if (suffix > 10000)
    {
      cerr << "wxWindows: error finding temporary file name.\n";
      buf[0] = 0;
      return;
    }
  }
}

Bool wxRemoveFile(char *file)
{
  int flag = unlink(file);

  if (flag == 0) return TRUE;
  return FALSE;
}

Bool wxMkdir(char *dir)
{
    // give default perms of owner read and write, group read and
    // others read. The interface to this func should be changed
    // to pass the perms info in.
  return (mkdir(dir, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH) == 0);
}

Bool wxRmdir(char *dir)
{
  return (rmdir(dir) == 0);
}

Bool DirExists(char *dir)
{
  DIR *dirstream = NULL;
  if (dirstream = opendir(dir))
  {
    closedir(dirstream);
    return TRUE;
  }
  else return FALSE;
}

// Get free memory in bytes, or -1 if cannot determine amount (e.g. on UNIX)
long wxGetFreeMemory(void)
{
  return -1;
}

// Sleep for nSecs seconds under UNIX, do nothing under Windows.
// XView implementation according to the Heller manual
void wxSleep(int nSecs)
{
#ifdef __sgi
  sleep(nSecs);
#else
#if defined(SVR4)
  struct sigset_t oldmask, mask;
  struct timeval tv;

  tv.tv_sec = nSecs;
  tv.tv_usec = 0;

  sigemptyset(&mask);
  sigaddset(&mask, SIGIO);
  sigaddset(&mask, SIGALRM);
  sigprocmask(SIG_BLOCK, &mask, &oldmask);
  if ((select(0,0,0,0,&tv)) == -1)
  {
    perror("select in wxSleep");
  }
//  sigprocmask(SIG_BLOCK, &oldmask, (sigset_t *) NULL); // Bug according to Kari
  sigprocmask(SIG_SETMASK, &oldmask, (sigset_t *) NULL);
#else
  int oldmask, mask;
  struct timeval tv;

  tv.tv_sec = nSecs;
  tv.tv_usec = 0;

  mask = sigmask(SIGIO);
  mask |= sigmask(SIGALRM);
  oldmask = sigblock(mask);
  if ((select(0,0,0,0,&tv)) == -1)
  {
    perror("select in wxSleep");
  }
  sigsetmask(oldmask);
#endif
#endif // __sgi
}

// Consume all events until no more left
#ifdef wx_xview
extern "C" int xv_input_pending(Display *, int);
extern "C" int ndis_dispatch(void);
#endif

void wxFlushEvents(void)
{
#ifdef wx_motif
  XSync(XtDisplay(wxTheApp->topLevel), FALSE);
  XEvent event;
  while (XtAppPending(wxTheApp->appContext))
  {
    XFlush(XtDisplay(wxTheApp->topLevel));
    XtAppNextEvent(wxTheApp->appContext, &event);
    XtDispatchEvent(&event);
  }
#endif
#ifdef wx_xview

  Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
  Xv_opaque root_window = xv_get(screen, XV_ROOT);
  Display *display = (Display *)xv_get(root_window, XV_DISPLAY);

  XSync(display, FALSE);
  XFlush(display);
/* Causes nasty internal problems. Pity, I thought I'd cracked it...
  while(XPending(display))
  {
    XEvent event;
    XPeekEvent(display, &event);
    xv_input_pending(display, 0);
    ndis_dispatch();
  }
*/
#endif
}

// Output a debug mess., in a system dependent fashion.

void wxDebugMsg(char *fmt ...)
{
#ifndef __sgi
  va_list ap;
  static char buffer[512];

  va_start(ap, fmt);

  vsprintf(buffer,fmt,ap) ;
  cerr << buffer ;

  va_end(ap);
#else
  fprintf(stderr, "Error: cannot use variable-argument functions on SGI!\n");
  return NULL;
#endif
}

// Non-fatal error: write error and continue
void wxError(char *msg, char *title)
{
  cerr << title << ": " << msg << "\n";
}

// Fatal error: pop up message box and abort
void wxFatalError(char *msg, char *title)
{
  cerr << title << ": " << msg << "\n";
  exit(1);
}

// Emit a beeeep...

void wxBell()
{
#ifdef wx_motif
  Display *display = XtDisplay(wxTheApp->topLevel) ;
#endif
#ifdef wx_xview
  Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
  Xv_opaque root_window = xv_get(screen, XV_ROOT);
  Display *display = (Display *)xv_get(root_window, XV_DISPLAY);
#endif

  XBell(display,100) ;

}

// Reading and writing resources (eg WIN.INI, .Xdefaults)

#if USE_RESOURCES
/*
 * We have a cache for writing different resource files,
 * which will only get flushed when we call wxFlushResources().
 * Build up a list of resource databases waiting to be written.
 *
 */

wxList wxResourceCache(wxKEY_STRING);

void wxFlushResources(void)
{
  char nameBuffer[512];
  
  wxNode *node = wxResourceCache.First();
  while (node)
  {
    char *file = node->key.string;
    // If file doesn't exit, create it first.
    if (!FileExists(file))
    {
      if (file[0] == '/')
        strcpy(nameBuffer, file);
      else
      {
        // Put in standard place for resource files if not absolute
        strcpy(nameBuffer, "/usr/lib/X11/app-defaults/");
        strcat(nameBuffer, FileNameFromPath(file));
      }
      if (!FileExists(nameBuffer))
      {
        FILE *fd = fopen(nameBuffer, "w");
        if (fd)
          fclose(fd);
      }
    }
    else strcpy(nameBuffer, file);
    
    XrmDatabase database = (XrmDatabase)node->Data();
    XrmPutFileDatabase(database, nameBuffer);
    XrmDestroyDatabase(database);
    wxNode *next = node->Next();
    delete node;
    node = next;
  }
}

Bool wxWriteResource(char *section, char *entry, char *value, char *file)
{
  char buffer[500];
  if (file)
    strcpy(buffer, file);
  else
  {
    (void) GetHomeDir(buffer);
    (void) strcat(buffer, "/.Xdefaults");
  }

  XrmDatabase database;
  wxNode *node = wxResourceCache.Find(buffer);
  if (node)
    database = (XrmDatabase)node->Data();
  else
  {
    database = XrmGetFileDatabase(buffer);
    wxResourceCache.Append(buffer, (wxObject *)database);
  }

  char resName[300];
  strcpy(resName, section);
  strcat(resName, ".");
  strcat(resName, entry);

  XrmPutStringResource(&database, resName, value);
  return TRUE;
}

Bool wxWriteResource(char *section, char *entry, float value, char *file)
{
  char buf[50];
  sprintf(buf, "%.4f", value);
  return wxWriteResource(section, entry, buf, file);
}

Bool wxWriteResource(char *section, char *entry, long value, char *file)
{
  char buf[50];
  sprintf(buf, "%ld", value);
  return wxWriteResource(section, entry, buf, file);
}

Bool wxWriteResource(char *section, char *entry, int value, char *file)
{
  char buf[50];
  sprintf(buf, "%d", value);
  return wxWriteResource(section, entry, buf, file);
}

Bool wxGetResource(char *section, char *entry, char **value, char *file)
{
  if (!wxResourceDatabase)
  {
#ifdef wx_motif
    Display *display = XtDisplay(wxTheApp->topLevel);
#endif
#ifdef wx_xview
    Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
    Xv_opaque root_window = xv_get(screen, XV_ROOT);
    Display *display = (Display *)xv_get(root_window, XV_DISPLAY);
#endif
    wxXMergeDatabases(wxTheApp, display);
  }

  XrmDatabase database;
  
  if (file)
  {
    wxNode *node = wxResourceCache.Find(file);
    if (node)
      database = (XrmDatabase)node->Data();
    else
    {
      database = XrmGetFileDatabase(file);
      wxResourceCache.Append(file, (wxObject *)database);
    }
  }
  else database = wxResourceDatabase;

  XrmValue xvalue;
  char *str_type[20];
  char buf[150];
  strcpy(buf, section);
  strcat(buf, ".");
  strcat(buf, entry);

  Bool success = XrmGetResource(database, buf, "*", str_type,
                  &xvalue);
  // Try different combinations of upper/lower case, just in case...
  if (!success)
  {
    if (isupper(buf[0]))
      buf[0] = tolower(buf[0]);
    else
      buf[0] = toupper(buf[0]);
    success = XrmGetResource(database, buf, "*", str_type,
                  &xvalue);
  }
  if (success)
  {
    if (!*value)
      *value = new char[xvalue.size + 1];
    strncpy(*value, xvalue.addr, (int)xvalue.size);
    return TRUE;
  }
  return FALSE;
}


Bool wxGetResource(char *section, char *entry, float *value, char *file)
{
  char *s = NULL;
  Bool succ = wxGetResource(section, entry, &s, file);
  if (succ)
  {
    *value = (float)strtod(s, NULL);
    delete[] s;
    return TRUE;
  }
  else return FALSE;
}

Bool wxGetResource(char *section, char *entry, long *value, char *file)
{
  char *s = NULL;
  Bool succ = wxGetResource(section, entry, &s, file);
  if (succ)
  {
    *value = strtol(s, NULL, 10);
    delete[] s;
    return TRUE;
  }
  else return FALSE;
}

Bool wxGetResource(char *section, char *entry, int *value, char *file)
{
  char *s = NULL;
  Bool succ = wxGetResource(section, entry, &s, file);
  if (succ)
  {
    *value = (int)strtol(s, NULL, 10);
    delete[] s; 
    return TRUE;
  }
  else return FALSE;
}

#ifdef XXXX
#ifdef wx_motif
/*
 * Not yet used but may be useful.
 *
 */
void wxSetDefaultResources(const Widget w, const char **resourceSpec, const char *name)
{
   int         i;	
   Display    *dpy = XtDisplay (w);	  // Retrieve the display pointer
   XrmDatabase rdb = NULL;             // A resource data base

   // Create an empty resource database
   rdb = XrmGetStringDatabase ("");

   // Add the Component resources, prepending the name of the component

   i = 0;
   while ( resourceSpec[i] != NULL )
   {
       char buf[1000];

       sprintf(buf, "*%s%s", name, resourceSpec[i++]);
       XrmPutLineResource( &rdb, buf );
   }

   // Merge them into the Xt database, with lowest precendence

   if ( rdb )
   {
#if (XlibSpecificationRelease>=5)
        XrmDatabase db = XtDatabase(dpy);
	XrmCombineDatabase(rdb, &db, FALSE);
#else
        XrmMergeDatabases ( dpy->db, &rdb );
        dpy->db = rdb;
#endif
    }
}
#endif
#endif // 0


/*
 * Merging defaults databases. We need to find resource information
 * from various sources and merge them before we query resources.
 *
 */

void wxXMergeDatabases(wxApp *theApp, Display *display)
{
  XrmDatabase homeDB, serverDB, applicationDB;
  char filenamebuf[1024];

  char *filename = &filenamebuf[0];
  char *environment;
  char *classname = theApp->wx_class;
  char name[256];
  (void) strcpy(name, "/usr/lib/X11/app-defaults/");
  (void) strcat(name, classname);

  /* Get application defaults file, if any */
  applicationDB = XrmGetFileDatabase(name);
  (void) XrmMergeDatabases(applicationDB, &wxResourceDatabase);

  /* Merge server defaults, created by xrdb, loaded as a property of the root
   * window when the server initializes and loaded into the display
   * structure on XOpenDisplay;
   * if not defined, use .Xdefaults.
   */

  if (XResourceManagerString(display) != NULL)
  {
    serverDB = XrmGetStringDatabase(XResourceManagerString(display));
  }
  else
  {
    (void) GetHomeDir(filename);
    (void) strcat(filename, "/.Xdefaults");
    serverDB = XrmGetFileDatabase(filename);
  }
  XrmMergeDatabases(serverDB, &wxResourceDatabase);

  /* Open XENVIRONMENT file, or if not defined, the .Xdefaults,
   * and merge into existing database
   */

  if ((environment = getenv("XENVIRONMENT")) == NULL)
  {
    int len;
    environment = GetHomeDir(filename);
    strcat(environment, "/.Xdefaults");
    len = strlen(environment);
#ifndef SVR4
    (void) gethostname(environment+len, 1024-len);
#else
    (void) sysinfo(SI_HOSTNAME, environment+len, 1024-len);
#endif
  }
  homeDB = XrmGetFileDatabase(environment);
  XrmMergeDatabases(homeDB, &wxResourceDatabase);
}

#endif // USE_RESOURCES

static char *GetHomeDir(char *dest)
{
    int uid;
    struct passwd *pw;
    register char *ptr;

    if ((ptr = getenv("HOME")) != NULL) {
        (void) strcpy(dest, ptr);

    } else {
        if ((ptr = getenv("USER")) != NULL) {
            pw = getpwnam(ptr);
        } else {
            uid = getuid();
            pw = getpwuid(uid);
        }
        if (pw) {
            (void) strcpy(dest, pw->pw_dir);
        } else {
            *dest = '\0';
        }
    }
    return dest;
}

