/*
 * File:	wx_win.cc
 * Purpose:	wxWindow class implementation
 * Author:	Julian Smart
 * Created:	1993
 * Updated:	
 * Copyright:	(c) 1993, AIAI, University of Edinburgh
 */

static const char sccsid[] = "%W% %G%";

#include "wx.h"
#pragma hdrstop

#include "wx_privt.h"
#include <iostream.h>
#include <shellapi.h>

/*
#include <windows.h>
#include "common.h"
#include "wx_win.h"
#include "wx_main.h"
#include "wx_utils.h"
#include "wx_gdi.h"
*/

#if CTL3D
#include <ctl3d.h>
#endif

#if !defined(APIENTRY)	// NT defines APIENTRY, 3.x not
#define APIENTRY far pascal
#endif
 
#ifdef WIN32
#define _EXPORT /**/
#else
#define _EXPORT _export
typedef signed short int SHORT ;
#endif
 
#if !defined(WIN32)	// 3.x uses FARPROC for dialogs
#define DLGPROC FARPROC
#endif

// Global variables
Bool wxShiftDown = FALSE;
Bool wxControlDown = FALSE;

wxMenu *wxCurrentPopupMenu = NULL;

// Find an item given the MS Windows id
wxWindow *wxWindow::FindItem(int id)
{
  if (!children)
    return NULL;
  wxNode *current = children->First();
  while (current)
  {
    wxObject *obj = (wxObject *)current->Data() ;
    if (wxSubType(obj->__type,wxTYPE_PANEL))
    {
      // Do a recursive search.
      wxPanel *panel = (wxPanel*)obj ;
      wxWindow *wnd = panel->FindItem(id) ;
      if (wnd)
        return wnd ;
    }
    else
    {
      wxItem *item = (wxItem *)current->Data();
      if (item->windows_id == id)
        return item;
      else
      {
        // In case it's a 'virtual' control (e.g. radiobox)
        if (item->subControls.Member((wxObject *)id))
          return item;
      }
    }
    current = current->Next();
  }
  return NULL;
}

// Default command handler
BOOL wxWindow::MSWCommand(UINT param, WORD id)
{
  return FALSE;
}

void wxWindow::PreDelete(HDC dc)
{
  mouseInWindow = FALSE ;
}

HWND wxWindow::GetHWND(void)
{
  HWND hWnd = 0;

  switch (wxWinType)
  {
    case wxTYPE_XWND:
    case wxTYPE_MDICHILD:
    {
      wxWnd *wnd = (wxWnd *)handle;
      hWnd = (HWND)wnd->handle;
      break;
    }
    default:
    {
      hWnd = (HWND)ms_handle;
      break;
    }
  }
  return hWnd;
}

// Constructor
wxWindow::wxWindow(void)
{
  handle = NULL;
}

// Destructor
wxWindow::~wxWindow(void)
{
  if (window_parent)
    window_parent->RemoveChild(this);

  wxbWindow::DestroyChildren();
  switch (wxWinType)
  {
    case wxTYPE_XWND:
    {
      if (handle)
      {
        wxWnd *wnd = (wxWnd *)handle;
        HDC dc = GetDC(wnd->handle);
        PreDelete(dc);
        ReleaseDC(wnd->handle, dc);

        wnd->DestroyWindow();
        delete wnd;
        handle = NULL;
      }
      break;
    }
    case wxTYPE_MDICHILD:
    {
      wxMDIChild *child = (wxMDIChild *)handle;
      child->DestroyWindow();
      delete child;
      handle = NULL;
      break;
    }
    case wxTYPE_HWND:
    {
      if (ms_handle)
        DestroyWindow((HWND)ms_handle);
      handle = NULL;
      break;
    }
    default:
      break;
    }

  delete children;
  children = NULL;
}

void wxWindow::SetFocus(void)
{
  switch (wxWinType)
  {
    case wxTYPE_XWND:
    {
      if (handle != NULL)
      {
        wxWnd *wnd = (wxWnd *)handle;
        ::SetFocus(wnd->handle);
      }
      break;
    }
    case wxTYPE_HWND:
    {
      if (ms_handle != NULL)
        ::SetFocus((HWND)ms_handle);
      break;
    }
    default:
      break;
  }
}

void wxWindow::Enable(Bool enable)
{
  switch (wxWinType)
  {
    case wxTYPE_XWND:
    {
      wxWnd *wnd = (wxWnd *)handle;
      ::EnableWindow(wnd->handle, enable);
      break;
    }
    case wxTYPE_HWND:
    {
      if (ms_handle)
        ::EnableWindow((HWND)ms_handle, enable);
      break;
    }
    default:
      break;
  }
}

void wxWindow::CaptureMouse(void)
{
  switch (wxWinType)
  {
    case wxTYPE_XWND:
    {
      wxWnd *wnd = (wxWnd *)handle;
      SetCapture(wnd->handle);
      break;
    }
    case wxTYPE_HWND:
    {
      if (ms_handle)
        SetCapture((HWND)ms_handle);
      break;
    }
    default:
      break;
  }
}

void wxWindow::ReleaseMouse(void)
{
  ReleaseCapture();
}

void wxWindow::DragAcceptFiles(Bool accept)
{
  switch (wxWinType)
  {
    case wxTYPE_XWND:
    {
      wxWnd *wnd = (wxWnd *)handle;
      ::DragAcceptFiles(wnd->handle, accept);
      break;
    }
    case wxTYPE_HWND:
    {
      if (ms_handle)
        ::DragAcceptFiles((HWND)ms_handle, accept);
      break;
    }
    default:
      break;
  }
}

// Get total size
void wxWindow::GetSize(int *x, int *y)
{
  wxWnd *wnd = (wxWnd *)handle;
  RECT rect;
  GetWindowRect(wnd->handle, &rect);
  *x = rect.right - rect.left;
  *y = rect.bottom - rect.top;
}

void wxWindow::GetPosition(int *x, int *y)
{
  HWND hWnd = GetHWND();
  HWND hParentWnd = 0;
  if (GetParent())
    hParentWnd = GetParent()->GetHWND();
  
  RECT rect;
  GetWindowRect(hWnd, &rect);

  // Since we now have the absolute screen coords,
  // if there's a parent we must subtract its top left corner
  POINT point;
  point.x = rect.left;
  point.y = rect.top;
  if (hParentWnd)
  {
    ::ScreenToClient(hParentWnd, &point);
  }
  *x = point.x;
  *y = point.y;
}

void wxWindow::ScreenToClient(int *x, int *y)
{
  HWND hWnd = GetHWND();
  POINT pt;
  pt.x = *x;
  pt.y = *y;
  ::ScreenToClient(hWnd, &pt);

  *x = pt.x;
  *y = pt.y;
}

void wxWindow::ClientToScreen(int *x, int *y)
{
  HWND hWnd = GetHWND();
  POINT pt;
  pt.x = *x;
  pt.y = *y;
  ::ClientToScreen(hWnd, &pt);

  *x = pt.x;
  *y = pt.y;
}

wxCursor *wxWindow::SetCursor(wxCursor *cursor)
{
  wxCursor *old_cursor = wx_cursor;
  wx_cursor = cursor;
  if (wx_cursor)
  {
    HWND hWnd = GetHWND();

    // Change the cursor NOW if we're within the correct window
    POINT point;
    ::GetCursorPos(&point);

    RECT rect;
    ::GetWindowRect(hWnd, &rect);

    if (::PtInRect(&rect, point))
      ::SetCursor(wx_cursor->ms_cursor);
  }

  wxFlushEvents();
  return old_cursor;
}

void wxWindow::SetColourMap(wxColourMap *cmap)
{
}

// Get size *available for subwindows* i.e. excluding menu bar etc.
// For XView, this is the same as GetSize
void wxWindow::GetClientSize(int *x, int *y)
{
  HWND hWnd = GetHWND();
  RECT rect;
  GetClientRect(hWnd, &rect);
  *x = rect.right;
  *y = rect.bottom;
}

void wxWindow::SetSize(int x, int y, int width, int height)
{
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  int currentW,currentH;
  GetSize(&currentW, &currentH);
  if (width == -1)
    width = currentW ;
  if (height == -1)
    height = currentH ;

  HWND hWnd = GetHWND();
  if (hWnd)
    MoveWindow(hWnd, x, y, width, height, TRUE);
  OnSize(width, height);
}

void wxWindow::SetClientSize(int width, int height)
{
  wxWindow *parent = GetParent();
  HWND hWnd = GetHWND();
  HWND hParentWnd = parent->GetHWND();

  RECT rect;
  GetClientRect(hWnd, &rect);

  RECT rect2;
  GetWindowRect(hWnd, &rect2);

  // Find the difference between the entire window (title bar and all)
  // and the client area; add this to the new client size to move the
  // window
  int actual_width = rect2.right - rect2.left - rect.right + width;
  int actual_height = rect2.bottom - rect2.top - rect.bottom + height;

  // If there's a parent, must subtract the parent's top left corner
  // since MoveWindow moves relative to the parent

  POINT point;
  point.x = rect2.left;
  point.y = rect2.top;
  if (parent)
  {
    ::ScreenToClient(hParentWnd, &point);
  }

  MoveWindow(hWnd, point.x, point.y, actual_width, actual_height, TRUE);
  OnSize(actual_width, actual_height);
}

void wxWindow::Show(Bool show)
{
  HWND hWnd = GetHWND();
  int cshow;
  if (show)
    cshow = SW_SHOW;
  else
    cshow = SW_HIDE;
  ShowWindow(hWnd, cshow);
  if (show)
    BringWindowToTop(hWnd);
}

float wxWindow::GetCharHeight(void)
{
  TEXTMETRIC lpTextMetric;
  HWND hWnd = GetHWND();
  HDC dc = GetDC(hWnd);

  GetTextMetrics(dc, &lpTextMetric);
  ReleaseDC(hWnd, dc);

  return (float)lpTextMetric.tmHeight;
}

float wxWindow::GetCharWidth(void)
{
  TEXTMETRIC lpTextMetric;
  HWND hWnd = GetHWND();
  HDC dc = GetDC(hWnd);

  GetTextMetrics(dc, &lpTextMetric);
  ReleaseDC(hWnd, dc);

  return (float)lpTextMetric.tmAveCharWidth;
}

void wxWindow::GetTextExtent(char *string, float *x, float *y,
                           float *descent, float *externalLeading)
{
  HWND hWnd = GetHWND();
  HDC dc = GetDC(hWnd);

  SIZE sizeRect;
  TEXTMETRIC tm;
  GetTextExtentPoint(dc, string, strlen(string), &sizeRect);
  GetTextMetrics(dc, &tm);

  ReleaseDC(hWnd, dc);

  *x = (float)sizeRect.cx;
  *y = (float)sizeRect.cy;
  if (descent) *descent = (float)tm.tmDescent;
  if (externalLeading) *externalLeading = (float)tm.tmExternalLeading;
}

// Hook for new window just as it's being created,
// when the window isn't yet associated with the handle
wxWnd *wxWndHook = NULL;

// Main Windows 3 window proc
LONG APIENTRY _EXPORT
  wxWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
  wxWnd *wnd = (wxWnd *)GetWindowLong(hWnd, 0);
  if (!wnd)
  {
    if (wxWndHook)
    {
      wnd = wxWndHook;
      wnd->handle = hWnd;
    }
    else wnd = wxFindWinFromHandle(hWnd);
  }
#if (DEBUG > 1)
  wxDebugMsg("hWnd = %d, wnd->handle = %d, msg = %d\n", hWnd, wnd ? wnd->handle : 0, message);
#endif
  // Stop right here if we don't have a valid handle
  // in our wxWnd object.
  if (wnd && !wnd->handle)
  {
    wnd->handle = hWnd;
    LONG res = wnd->DefWindowProc(message, wParam, lParam );
    wnd->handle = NULL;
    return res;
  }

  if (wnd)
  {
    wnd->last_msg = message;
    wnd->last_wparam = wParam;
    wnd->last_lparam = lParam;

    if (message == WM_SETFONT)
      return 0;
    else if (message == WM_INITDIALOG)
      return TRUE;
  }

  switch (message)
  {
        case WM_ACTIVATE:
        {
            WORD state = LOWORD(wParam);
#ifdef WIN32
            WORD minimized = HIWORD(wParam);
            HWND hwnd = (HWND)lParam;
#else
            WORD minimized = HIWORD(lParam);
            HWND hwnd = (HWND)LOWORD(lParam);
#endif
            wnd->OnActivate(state, minimized, hwnd);
            return 0;
//            if (!wnd->OnActivate(state, minimized, hwnd))
//              return wnd->DefWindowProc(message, wParam, lParam );
            break;
        }
        case WM_SETFOCUS:
        {
            HWND hwnd = (HWND)wParam;
//            return wnd->OnSetFocus(hwnd);

            if (wnd->OnSetFocus(hwnd))
              return 0;
            else return wnd->DefWindowProc(message, wParam, lParam );
            break;
        }
        case WM_KILLFOCUS:
        {
            HWND hwnd = (HWND)lParam;
//            return wnd->OnKillFocus(hwnd);
            if (wnd->OnKillFocus(hwnd))
              return 0;
            else
              return wnd->DefWindowProc(message, wParam, lParam );
            break;
        }
	case WM_CREATE:
	{
          if (wnd)
            wnd->OnCreate((LPCREATESTRUCT)lParam);
          return 0;
          break;
	}
	case WM_PAINT:
	{
          if (wnd->OnPaint())
            return 0;
          else return wnd->DefWindowProc(message, wParam, lParam );
          break;
        }

        case WM_SIZE:
        {
            if (wnd)
            {
              int width = LOWORD(lParam);
              int height = HIWORD(lParam);
              wnd->OnSize(width, height, wParam);
            }
            else return DefWindowProc( hWnd, message, wParam, lParam );
            break;
        }

        case WM_RBUTTONDOWN:
        {
            int x = (int)LOWORD(lParam);
            int y = (int)HIWORD(lParam);
            wnd->OnRButtonDown(x, y, wParam);
            break;
        }
        case WM_RBUTTONUP:
        {
            int x = (int)LOWORD(lParam);
            int y = (int)HIWORD(lParam);
            wnd->OnRButtonUp(x, y, wParam);
            break;
        }
        case WM_RBUTTONDBLCLK:
        {
            int x = (int)LOWORD(lParam);
            int y = (int)HIWORD(lParam);
            wnd->OnRButtonDClick(x, y, wParam);
            break;
        }
        case WM_MBUTTONDOWN:
        {
            int x = (int)LOWORD(lParam);
            int y = (int)HIWORD(lParam);
            wnd->OnMButtonDown(x, y, wParam);
            break;
        }
        case WM_MBUTTONUP:
        {
            int x = (int)LOWORD(lParam);
            int y = (int)HIWORD(lParam);
            wnd->OnMButtonUp(x, y, wParam);
            break;
        }
        case WM_MBUTTONDBLCLK:
        {
            int x = (int)LOWORD(lParam);
            int y = (int)HIWORD(lParam);
            wnd->OnMButtonDClick(x, y, wParam);
            break;
        }
        case WM_LBUTTONDOWN:
        {
            int x = (int)LOWORD(lParam);
            int y = (int)HIWORD(lParam);
            wnd->OnLButtonDown(x, y, wParam);
            break;
        }
        case WM_LBUTTONUP:
        {
            int x = (int)LOWORD(lParam);
            int y = (int)HIWORD(lParam);
            wnd->OnLButtonUp(x, y, wParam);
            break;
        }
        case WM_LBUTTONDBLCLK:
        {
            int x = (int)LOWORD(lParam);
            int y = (int)HIWORD(lParam);
            wnd->OnLButtonDClick(x, y, wParam);
            break;
        }
        case WM_MOUSEMOVE:
        {
            int x = (int)LOWORD(lParam);
            int y = (int)HIWORD(lParam);
            wnd->OnMouseMove(x, y, wParam);
            break;
        }
        case WM_DESTROY:
        {
            if (wnd)
            {
              if (wnd->OnDestroy())
                return 0;
              else return wnd->DefWindowProc(message, wParam, lParam );
            }
            else return ::DefWindowProc( hWnd, message, wParam, lParam );
            break;
        }
/*
        case WM_SYSCOMMAND:
            break;
*/
        case WM_COMMAND:
	{
            WORD id = LOWORD(wParam);
            HWND hwnd = (HWND)(UINT)lParam;
#ifdef WIN32
            WORD cmd = HIWORD(wParam);
#else
            WORD cmd = HIWORD(lParam);
#endif
            if (!wnd->OnCommand(id, cmd, hwnd))
              return wnd->DefWindowProc(message, wParam, lParam );
            break;
	 }
        case WM_MENUSELECT:
        {
            WORD id = LOWORD(wParam);
#ifdef WIN32
            WORD flags = HIWORD(wParam);
            HMENU sysmenu = (HMENU)lParam;
#else
            WORD flags = LOWORD(lParam);
            HMENU sysmenu = (HMENU)HIWORD(lParam);
#endif
            wnd->OnMenuSelect(wParam, flags, sysmenu);
            break;
        }
        case WM_KEYDOWN:
        {
            if (wParam == VK_SHIFT)
              wxShiftDown = TRUE;
            else if (wParam == VK_CONTROL)
              wxControlDown = TRUE;
            // Avoid duplicate messages to OnChar
            else if ((wParam != VK_ESCAPE) && (wParam != VK_SPACE) && (wParam != VK_RETURN) && (wParam != VK_DELETE))
	    {
              wnd->OnChar(wParam);
	    }
            break;
        }
        case WM_KEYUP:
        {
            if (wParam == VK_SHIFT)
              wxShiftDown = FALSE;
            else if (wParam == VK_CONTROL)
              wxControlDown = FALSE;
            break;
        }
        case WM_CHAR: // Always an ASCII character
        {
          wnd->OnChar(wParam, TRUE);
          break;
        }
        case WM_HSCROLL:
        {
            WORD code = LOWORD(wParam);
#ifdef WIN32
            WORD pos = HIWORD(wParam);
            HWND control = (HWND)lParam;
#else
            WORD pos = LOWORD(lParam);
            HWND control = (HWND)HIWORD(lParam);
#endif
            wnd->OnHScroll(code, pos, control);
            break;
        }
        case WM_VSCROLL:
        {
            WORD code = LOWORD(wParam);
#ifdef WIN32
            WORD pos = HIWORD(wParam);
            HWND control = (HWND)lParam;
#else
            WORD pos = LOWORD(lParam);
            HWND control = (HWND)HIWORD(lParam);
#endif
            wnd->OnVScroll(code, pos, control);
            break;
        }
#ifdef WIN32
        case WM_CTLCOLORBTN:
	{
          int nCtlColor = CTLCOLOR_BTN;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
        case WM_CTLCOLORDLG:
	{
          int nCtlColor = CTLCOLOR_DLG;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);\
          break;
	}
        case WM_CTLCOLORLISTBOX:
	{
          int nCtlColor = CTLCOLOR_LISTBOX;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
        case WM_CTLCOLORMSGBOX:
	{
          int nCtlColor = CTLCOLOR_MSGBOX;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
        case WM_CTLCOLORSCROLLBAR:
	{
          int nCtlColor = CTLCOLOR_SCROLLBAR;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
        case WM_CTLCOLORSTATIC:
	{
          int nCtlColor = CTLCOLOR_STATIC;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
        case WM_CTLCOLOREDIT:
	{
          int nCtlColor = CTLCOLOR_EDIT;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
#else
        case WM_CTLCOLOR:
        {
          HWND control = (HWND)LOWORD(lParam);
          int nCtlColor = (int)HIWORD(lParam);
          HDC pDC = (HDC)wParam;
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
        }
#endif
        case WM_SYSCOLORCHANGE:
        {
#if CTL3D
          Ctl3dColorChange();
#else
          return ::DefWindowProc( hWnd, message, wParam, lParam );
#endif
          break;
        }
        case WM_ERASEBKGND:
        {
          // Prevents flicker when dragging
          if (IsIconic(hWnd)) return 1;

          if (!wnd->OnEraseBkgnd((HDC)wParam))
            return wnd->DefWindowProc(message, wParam, lParam );
          else return 1;
          break;
        }
        case WM_MDIACTIVATE:
        {
#ifdef WIN32
            HWND hWndActivate = (HWND)LOWORD(wParam);
            HWND hWndDeactivate = (HWND)LOWORD(lParam);
            BOOL activate = (hWndActivate == hWnd);
            return wnd->OnMDIActivate(activate, hWndActivate, hWndDeactivate);
#else
            return wnd->OnMDIActivate((BOOL)wParam, (HWND)LOWORD(lParam),
                                               (HWND)HIWORD(lParam));
#endif
        }
        case WM_DROPFILES:
        {
            wnd->OnDropFiles(wParam);
            break;
	}
        case WM_CLOSE:
        {
            if (wnd->OnClose())
              return 0L;
            else
              return 1L;
            break;
        }

        default:
            if (wnd)
              return wnd->DefWindowProc(message, wParam, lParam );
            else return DefWindowProc( hWnd, message, wParam, lParam );
    }
    return 0; // Success: we processed this command.
}

// Dialog window proc
LONG APIENTRY _EXPORT
  wxDlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
  wxWnd *wnd = wxFindWinFromHandle(hWnd);

  if (!wnd && wxWndHook)
  {
    wnd = wxWndHook;
    wnd->handle = hWnd;
  }

  if (wnd)
  {
    wnd->last_msg = message;
    wnd->last_wparam = wParam;
    wnd->last_lparam = lParam;
  }

  if (message == WM_SETFONT)
    return 0;
  else if (message == WM_INITDIALOG)
    return 0;

  switch (message)
  {
#if CTL3D
        case WM_SETTEXT:
        case WM_NCPAINT:
        case WM_NCACTIVATE:
        {

          SetWindowLong(hWnd, DWL_MSGRESULT,
                        Ctl3dDlgFramePaint(hWnd, message, wParam, lParam));
          return TRUE;
          break;
        }
#endif
        case WM_ACTIVATE:
        {
            WORD state = LOWORD(wParam);
#ifdef WIN32
            WORD minimized = HIWORD(wParam);
            HWND hwnd = (HWND)lParam;
#else
            WORD minimized = HIWORD(lParam);
            HWND hwnd = (HWND)LOWORD(lParam);
#endif
            wnd->OnActivate(state, minimized, hwnd);
            return 0;
//            if (!wnd->OnActivate(state, minimized, hwnd))
//              return wnd->DefWindowProc(message, wParam, lParam );
            break;
	 }

        case WM_SETFOCUS:
        {
            HWND hwnd = (HWND)wParam;
            return wnd->OnSetFocus(hwnd);
//            if (!wnd->OnSetFocus(hwnd))
//              return wnd->DefWindowProc(message, wParam, lParam );
            break;
        }
        case WM_KILLFOCUS:
        {
            HWND hwnd = (HWND)lParam;
            return wnd->OnKillFocus(hwnd);
//            if (!wnd->OnKillFocus(hwnd))
//              return wnd->DefWindowProc(message, wParam, lParam );
            break;
        }
        case WM_CREATE:
            if (wnd)
              wnd->OnCreate((LPCREATESTRUCT)lParam);
            return 0;
	    break;
        case WM_SIZE:
        {
            if (wnd)
            {
              int width = LOWORD(lParam);
              int height = HIWORD(lParam);
              wnd->OnSize(width, height, wParam);
            }
            else return FALSE;
            break;
        }
/*
        case WM_DESTROY:
            if (wnd)
            {
              if (wnd->OnDestroy())
                return 0;
            }
            return FALSE;
            break;
*/
        case WM_COMMAND:
	{
            WORD id = LOWORD(wParam);
            HWND hwnd = (HWND)(UINT)lParam;
#ifdef WIN32
            WORD cmd = HIWORD(wParam);
#else
            WORD cmd = HIWORD(lParam);
#endif
            if (!wnd->OnCommand(id, cmd, hwnd))
              return wnd->DefWindowProc(message, wParam, lParam );
            break;
	}
/* Do we intercept keystrokes from dialogs, or would this cause problems?
        case WM_KEYDOWN:
        {
            if (wParam == VK_SHIFT)
              wxShiftDown = TRUE;
            else if (wParam == VK_CONTROL)
              wxControlDown = TRUE;
            else if ((wParam != VK_ESCAPE) && (wParam != VK_SPACE) && (wParam != VK_RETURN) && (wParam != VK_DELETE))
              wnd->OnChar(wParam);
            break;
        }
        case WM_KEYUP:
        {
            if (wParam == VK_SHIFT)
              wxShiftDown = FALSE;
            else if (wParam == VK_CONTROL)
              wxControlDown = FALSE;
            break;
        }
*/
        case WM_HSCROLL:
        {
            WORD code = LOWORD(wParam);
#ifdef WIN32
            WORD pos = HIWORD(wParam);
            HWND control = (HWND)lParam;
#else
            WORD pos = LOWORD(lParam);
            HWND control = (HWND)HIWORD(lParam);
#endif
            wnd->OnHScroll(code, pos, control);
            break;
        }
        case WM_VSCROLL:
        {
            WORD code = LOWORD(wParam);
#ifdef WIN32
            WORD pos = HIWORD(wParam);
            HWND control = (HWND)lParam;
#else
            WORD pos = LOWORD(lParam);
            HWND control = (HWND)HIWORD(lParam);
#endif
            wnd->OnVScroll(code, pos, control);
            break;
        }
#ifdef WIN32
        case WM_CTLCOLORBTN:
	{
          int nCtlColor = CTLCOLOR_BTN;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
        case WM_CTLCOLORDLG:
	{
          int nCtlColor = CTLCOLOR_DLG;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
        case WM_CTLCOLORLISTBOX:
	{
          int nCtlColor = CTLCOLOR_LISTBOX;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
        case WM_CTLCOLORMSGBOX:
	{
          int nCtlColor = CTLCOLOR_MSGBOX;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
        case WM_CTLCOLORSCROLLBAR:
	{
          int nCtlColor = CTLCOLOR_SCROLLBAR;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
        case WM_CTLCOLORSTATIC:
	{
          int nCtlColor = CTLCOLOR_STATIC;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
        case WM_CTLCOLOREDIT:
	{
          int nCtlColor = CTLCOLOR_EDIT;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
	}
#else
        case WM_CTLCOLOR:
        {
          HWND control = (HWND)LOWORD(lParam);
          int nCtlColor = (int)HIWORD(lParam);
          HDC pDC = (HDC)wParam;
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor,
                                        message, wParam, lParam);
          break;
        }
#endif
        case WM_SYSCOLORCHANGE:
        {
#if CTL3D
          Ctl3dColorChange();
#else
          return ::DefWindowProc( hWnd, message, wParam, lParam );
#endif
          break;
        }
        case WM_ERASEBKGND:
        {
            return wnd->OnEraseBkgnd((HDC)wParam);
            break;
        }
        case WM_CLOSE:
        {
            return !wnd->OnClose();
            break;
        }
        case WM_DROPFILES:
        {
            wnd->OnDropFiles(wParam);
            break;
	}
        default:
        {
          return FALSE;
	}
    }
    return FALSE;
}

wxList *wxWinHandleList = NULL;
wxWnd *wxFindWinFromHandle(HWND hWnd)
{
  wxNode *node = wxWinHandleList->Find((long)hWnd);
  if (!node)
    return NULL;
  return (wxWnd *)node->Data();
}

/* wxWnd class used to implement all Windows 3 windows
 */
wxWnd::wxWnd(void)

{
  x_scrolling_enabled = TRUE;
  y_scrolling_enabled = TRUE;
  last_msg = 0;
  last_wparam = 0;
  last_lparam = 0;
  accelerator_table = NULL;
  hMenu = 0;

  xscroll_pixels_per_line = 0;
  yscroll_pixels_per_line = 0;
  xscroll_lines = 0;
  yscroll_lines = 0;
  xscroll_lines_per_page = 0;
  yscroll_lines_per_page = 0;
  xscroll_position = 0;
  yscroll_position = 0;
  background_brush = GetStockObject( LTGRAY_BRUSH );
  last_x_pos = -1.0;
  last_y_pos = -1.0;
  last_event = -1;
  is_canvas = FALSE;
  cdc = NULL;
  ldc = NULL ;
  dc_count = 0 ;

}

wxWnd::~wxWnd(void)
{
  wxWinHandleList->DeleteObject(this);
}

HDC wxWnd::GetHDC(void)
{
  if (cdc)
    return(cdc) ;
  if (dc_count==0)
    ldc = ::GetDC(handle) ;
//wxDebugMsg("Get, Count %d\n",dc_count) ;
  dc_count++ ;
  return(ldc) ;

}

void wxWnd::ReleaseHDC(void)
{
  if (cdc)
    return ;
  dc_count-- ;
//wxDebugMsg("Release, Count %d\n",dc_count) ;
  if (dc_count==0)
    ::ReleaseDC(handle,ldc) ;
}

// Default destroyer - override if you destroy it in some other way
// (e.g. with MDI child windows)
void wxWnd::DestroyWindow(void)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::DestroyWindow %d\n", handle);
#endif
  DetachWindowMenu();
  SetWindowLong(handle, 0, (long)0);
  HWND oldHandle = handle;
  handle = NULL;
  ::DestroyWindow(oldHandle);
  // Menu is destroyed explicitly by wxMDIChild::DestroyWindow,
  // or when Windows HWND is deleted if MDI parent or
  // SDI frame.
/*
  if (hMenu)
  {
    ::DestroyMenu(hMenu);
    hMenu = 0;
  }
 */
}

void wxWnd::Create(wxWnd *parent, char *wclass, wxWindow *wx_win, char *title,
                    int x, int y, int width, int height,
                    DWORD style, char *dialog_template, DWORD extendedStyle)
{
  wx_window = wx_win;
  is_dialog = (dialog_template != NULL);
  int x1 = 0;
  int y1 = 0;
  int x2 = 100;
  int y2 = 100;

  // Find parent's size, if it exists, to set up a possible default
  // panel size the size of the parent window
  RECT parent_rect;
  if (parent)
  {
    GetWindowRect(parent->handle, &parent_rect);

    // Convert from screen coordinates to parent coordinates
    x2 = parent_rect.right - parent_rect.left;
    y2 = parent_rect.bottom - parent_rect.top;
  }

  if (x > -1) x1 = x;
  if (y > -1) y1 = y;
  if (width > -1) x2 = x1 + width;
  if (height > -1) y2 = y1 + height;

  HWND hParent = NULL;
  if (parent)
    hParent = parent->handle;

  wxWndHook = this;

  if (is_dialog)
  {
    // MakeProcInstance doesn't seem to be needed in C7. Is it needed for
    // other compilers???
//    DLGPROC dlgproc = (DLGPROC)MakeProcInstance(wxWndProc, wxhInstance);

    handle = ::CreateDialog(wxhInstance, dialog_template, hParent,
                            (DLGPROC)wxDlgProc);
    if (handle == 0)
       MessageBox(NULL, "Can't find dummy dialog template!\nCheck resource include path for finding wx.rc.",
                  "wxWindows Error", MB_ICONEXCLAMATION | MB_OK);
    else MoveWindow(handle, x1, y1, x2 - x1, y2 - y1, FALSE);
  }
  else
  {
    handle = CreateWindowEx(extendedStyle, wclass,
                title,
                style,
                x1, y1,
                x2 - x1, y2 - y1,
                hParent, NULL, wxhInstance,
                NULL);
  }
  wxWndHook = NULL;
  wxWinHandleList->Append((long)handle, this);

#ifdef DEBUG
  wxDebugMsg("wxWnd::Create %d\n", handle);
#endif

  // Can't do this for dialogs!!!!
  if (!is_dialog) SetWindowLong(handle, 0, (long)this);
}

void wxWnd::OnCreate(LPCREATESTRUCT cs)
{
}

BOOL wxWnd::OnPaint(void)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnPaint %d\n", handle);
#endif
  return 1;
}

BOOL wxWnd::OnClose(void)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnClose %d\n", handle);
#endif
  return FALSE;
}

BOOL wxWnd::OnDestroy(void)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnDestroy %d\n", handle);
#endif
  return TRUE;
}

void wxWnd::OnSize(int x, int y, UINT flag)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnSize %d\n", handle);
#endif
}

// Deal with child commands from buttons etc.

BOOL wxWnd::OnCommand(WORD id, WORD cmd, HWND control)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnCommand %d\n", handle);
#endif
  return FALSE;
}

void wxWnd::OnMenuSelect(WORD item, WORD flags, HMENU sysmenu)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnMenuSelect %d\n", handle);
#endif
}

BOOL wxWnd::OnActivate(BOOL state, BOOL minimized, HWND activate)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnActivate %d\n", handle);
#endif
  if (wx_window)
  {
    wx_window->OnActivate(((state == WA_ACTIVE) || (state == WA_CLICKACTIVE)));
    return 0;
  }
  else return TRUE;
}

BOOL wxWnd::OnSetFocus(HWND hwnd)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnSetFocus %d\n", handle);
#endif
  if (wx_window)
  {
    wx_window->OnSetFocus();
//    return 0;
    return TRUE;
  }
  else return FALSE;
}

BOOL wxWnd::OnKillFocus(HWND hwnd)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnKillFocus %d\n", handle);
#endif
  if (wx_window)
  {
    wx_window->OnKillFocus();
    return TRUE;
  }
  else return FALSE;
}

void wxWnd::OnDropFiles(WPARAM wParam)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnDropFiles %d\n", handle);
#endif

  HANDLE hFilesInfo = (HANDLE)wParam;
  POINT dropPoint;
  DragQueryPoint(hFilesInfo, (LPPOINT) &dropPoint);

  // Get the total number of files dropped
  WORD gwFilesDropped = DragQueryFile (hFilesInfo,
				   (UINT)-1,
                                   NULL,
                                   0);

  char **files = new char *[gwFilesDropped];
  for (int wIndex=0; wIndex < (int)gwFilesDropped; wIndex++)
  {
    DragQueryFile (hFilesInfo, wIndex, (LPSTR) wxBuffer, 1000);
    files[wIndex] = copystring(wxBuffer);
  }
  DragFinish (hFilesInfo);
  if (wx_window)
    wx_window->OnDropFiles(gwFilesDropped, files, dropPoint.x, dropPoint.y);

  for (int i = 0; i < (int)gwFilesDropped; i++)
    delete[] files[i];
  delete[] files;
}

void wxWnd::OnVScroll(WORD code, WORD pos, HWND control)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnVScroll %d\n", handle);
#endif
}

void wxWnd::OnHScroll(WORD code, WORD pos, HWND control)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnHScroll %d\n", handle);
#endif
}

void wxWnd::CalcScrolledPosition(int x, int y, int *xx, int *yy)
{
  *xx = x - xscroll_position * xscroll_pixels_per_line;
  *yy = y - yscroll_position * yscroll_pixels_per_line;
}

void wxWnd::CalcUnscrolledPosition(int x, int y, float *xx, float *yy)
{
  *xx = (float)(x + xscroll_position * xscroll_pixels_per_line);
  *yy = (float)(y + yscroll_position * yscroll_pixels_per_line);
}

HBRUSH wxWnd::OnCtlColor(HDC pDC, HWND pWnd, UINT nCtlColor,
                         UINT message, WPARAM wParam, LPARAM lParam)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnCtlColour %d\n", handle);
#endif
#if (!CTL3D)
  if ((nCtlColor == CTLCOLOR_STATIC || nCtlColor == CTLCOLOR_BTN) && background_brush)
  {
    SetBkColor(pDC, RGB(200, 200, 200));
    return background_brush;
  }
  else return NULL;
#else
  HBRUSH hbrush = Ctl3dCtlColorEx(message, wParam, lParam);
  if (hbrush != (HBRUSH) 0)
    return hbrush;
  else
    return (HBRUSH)::DefWindowProc(pWnd, message, wParam, lParam);
#endif
}

BOOL wxWnd::OnEraseBkgnd(HDC pDC)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnEraseBkgnd %d\n", handle);
#endif
  return FALSE;
}

void wxWnd::OnLButtonDown(int x, int y, UINT flags)
{
}

void wxWnd::OnLButtonUp(int x, int y, UINT flags)
{
}

void wxWnd::OnLButtonDClick(int x, int y, UINT flags)
{
}

void wxWnd::OnMButtonDown(int x, int y, UINT flags)
{
}

void wxWnd::OnMButtonUp(int x, int y, UINT flags)
{
}

void wxWnd::OnMButtonDClick(int x, int y, UINT flags)
{
}

void wxWnd::OnRButtonDown(int x, int y, UINT flags)
{
}

void wxWnd::OnRButtonUp(int x, int y, UINT flags)
{
}

void wxWnd::OnRButtonDClick(int x, int y, UINT flags)
{
}

void wxWnd::OnMouseMove(int x, int y, UINT flags)
{
}

void wxWnd::OnMouseEnter(int x, int y, UINT flags)
{
}

void wxWnd::OnMouseLeave(int x, int y, UINT flags)
{
}

void wxWnd::OnChar(WORD wParam, Bool isASCII)
{
}

LONG wxWnd::DefWindowProc(UINT nMsg, WPARAM wParam, LPARAM lParam)
{
  return ::DefWindowProc(handle, nMsg, wParam, lParam);
}

BOOL wxWnd::ProcessMessage(MSG* pMsg)
{
  return FALSE;
}

BOOL wxWnd::OnMDIActivate(BOOL flag, HWND activate, HWND deactivate)
{
#ifdef DEBUG
  wxDebugMsg("wxWnd::OnMDIActivate %d\n", handle);
#endif
  return 1;
}

void wxWnd::DetachWindowMenu(void)
{
  if (hMenu)
  {
    int N = GetMenuItemCount(hMenu);
    int i;
    for (i = 0; i < N; i++)
    {
      char buf[100];
      int chars = GetMenuString(hMenu, i, buf, 100, MF_BYPOSITION);
      if ((chars > 0) && (strcmp(buf, "&Window") == 0))
      {
        RemoveMenu(hMenu, i, MF_BYPOSITION);
      }
    }
  }
}

/*
 * Subwindow - used for panels and canvases
 *
 */

wxSubWnd::wxSubWnd(wxWnd *parent, char *wclass, wxWindow *wx_win,
                           int x, int y, int width, int height,
                           DWORD style, char *dialog_template)

{
  Create(parent, wclass, wx_win, NULL, x, y, width, height, style, dialog_template);
  mouse_in_window = FALSE ;
}

wxSubWnd::~wxSubWnd(void)
{
}


BOOL wxSubWnd::OnPaint(void)
{
#ifdef DEBUG
  wxDebugMsg("wxSubWnd::OnPaint %d\n", handle);
#endif
  RECT rect;
  if (GetUpdateRect(handle, &rect, FALSE))
  {
    PAINTSTRUCT ps;
    // Hold a pointer to the dc so long as the OnPaint() message
    // is being processed
    cdc = BeginPaint(handle, &ps);
    if (wx_window)
      wx_window->OnPaint();

    cdc = NULL;
    EndPaint(handle, &ps);
    return TRUE;
  }
  return FALSE;
}

void wxSubWnd::OnSize(int w, int h, UINT flag)
{
#ifdef DEBUG
  wxDebugMsg("wxSubWnd::OnSize %d\n", handle);
#endif
  // Recalculate scroll bar range and position
  if (xscroll_lines > 0)
  {
    int nMaxWidth = xscroll_lines*xscroll_pixels_per_line;
    int nHscrollMax = max(0, (int)(2 + (nMaxWidth - w)/xscroll_pixels_per_line));
    xscroll_position = min(nHscrollMax, xscroll_position);

    SetScrollRange(handle, SB_HORZ, 0, nHscrollMax, FALSE);
    SetScrollPos(handle, SB_HORZ, xscroll_position, TRUE );
  }	
  if (yscroll_lines > 0)
  {
    int nMaxHeight = yscroll_lines*yscroll_pixels_per_line;
    int nVscrollMax = max(0, (int)(2 + (nMaxHeight - h)/yscroll_pixels_per_line));
    yscroll_position = min(nVscrollMax, yscroll_position);

    SetScrollRange(handle, SB_VERT, 0, nVscrollMax, FALSE);
    SetScrollPos(handle, SB_VERT, yscroll_position, TRUE );
  }	

  // Store DC for duration of size message
  cdc = GetDC(handle);
  if (wx_window)
    wx_window->OnSize(w, h);

  ReleaseDC(handle, cdc);
  cdc = NULL;
}

// Deal with child commands from buttons etc.
BOOL wxSubWnd::OnCommand(WORD id, WORD cmd, HWND control)
{
#ifdef DEBUG
  wxDebugMsg("wxSubWnd::OnCommand %d\n", handle);
#endif
  if (wxCurrentPopupMenu)
  {
    BOOL succ = wxCurrentPopupMenu->MSWCommand(cmd, id);
    wxCurrentPopupMenu = NULL;
    return succ;
  }
  wxWindow *item = wx_window->FindItem(id);
  if (item)
  {
    Bool value = item->MSWCommand(cmd, id);
    return value;
  }
  else
    return FALSE;
}

void wxSubWnd::OnLButtonDown(int x, int y, UINT flags)
{
#ifdef WIN32
  // DClick not clean supported on Win3.1, except if someone know
  // how to emulate Sleep()...
  // This means that your app will receive Down-Up-Dclick sequences
  // rather than Dclick
  if (wx_window && wx_window->doubleClickAllowed)
  {
    UINT time = GetDoubleClickTime() ;
#ifdef WIN32
    Sleep(time) ;
#endif
    MSG dummy ;
    if (PeekMessage(&dummy,handle,
                    WM_LBUTTONDBLCLK,WM_LBUTTONDBLCLK,
                    PM_NOREMOVE)
       )
    {
      PeekMessage(&dummy,handle,WM_LBUTTONUP,WM_LBUTTONUP,PM_REMOVE);
      return; 
    }
  }
#endif
//wxDebugMsg("LButtonDown\n") ;
  wxMouseEvent event(wxEVENT_TYPE_LEFT_DOWN);

  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  last_x_pos = event.x; last_y_pos = event.y; last_event = wxEVENT_TYPE_LEFT_DOWN;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnLButtonUp(int x, int y, UINT flags)
{
//wxDebugMsg("LButtonUp\n") ;
  wxMouseEvent event(wxEVENT_TYPE_LEFT_UP);
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  last_x_pos = event.x; last_y_pos = event.y; last_event = wxEVENT_TYPE_LEFT_UP;

  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnLButtonDClick(int x, int y, UINT flags)
{
//wxDebugMsg("LButtonDClick\n") ;
  wxMouseEvent event(wxEVENT_TYPE_LEFT_DCLICK);
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  last_x_pos = event.x; last_y_pos = event.y; last_event = wxEVENT_TYPE_LEFT_DCLICK;

  if (wx_window && wx_window->doubleClickAllowed) wx_window->OnEvent(event);
}

void wxSubWnd::OnMButtonDown(int x, int y, UINT flags)
{
#ifdef WIN32
  // DClick not clean supported on Win3.1, except if someone know
  // how to emulate Sleep()...
  // This means that your app will receive Down-Up-Dclick sequences
  // rather than Dclick
  if (wx_window && wx_window->doubleClickAllowed)
  {
    UINT time = GetDoubleClickTime() ;
#ifdef WIN32
    Sleep(time) ;
#endif
    MSG dummy ;
    if (PeekMessage(&dummy,handle,
                    WM_MBUTTONDBLCLK,WM_MBUTTONDBLCLK,
                    PM_NOREMOVE)
       )
    {
      PeekMessage(&dummy,handle,WM_MBUTTONUP,WM_MBUTTONUP,PM_REMOVE);
      return; 
    }
  }
#endif

//wxDebugMsg("MButtonDown\n") ;
  wxMouseEvent event(wxEVENT_TYPE_MIDDLE_DOWN);

  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  last_x_pos = event.x; last_y_pos = event.y; last_event = wxEVENT_TYPE_LEFT_DOWN;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnMButtonUp(int x, int y, UINT flags)
{
//wxDebugMsg("MButtonUp\n") ;
  wxMouseEvent event(wxEVENT_TYPE_MIDDLE_UP);
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  last_x_pos = event.x; last_y_pos = event.y; last_event = wxEVENT_TYPE_LEFT_UP;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnMButtonDClick(int x, int y, UINT flags)
{
//wxDebugMsg("MButtonDClick\n") ;
  wxMouseEvent event(wxEVENT_TYPE_MIDDLE_DCLICK);
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  last_x_pos = event.x; last_y_pos = event.y; last_event = wxEVENT_TYPE_LEFT_DCLICK;
  if (wx_window && wx_window->doubleClickAllowed) wx_window->OnEvent(event);
}

void wxSubWnd::OnRButtonDown(int x, int y, UINT flags)
{
#ifdef WIN32
  // DClick not clean supported on Win3.1, except if someone know
  // how to emulate Sleep()...
  // This means that your app will receive Down-Up-Dclick sequences
  // rather than Dclick
  if (wx_window && wx_window->doubleClickAllowed)
  {
    UINT time = GetDoubleClickTime() ;
#ifdef WIN32
    Sleep(time) ;
#endif
    MSG dummy ;
    if (PeekMessage(&dummy,handle,
                    WM_RBUTTONDBLCLK,WM_RBUTTONDBLCLK,
                    PM_NOREMOVE)
       )
    {
      PeekMessage(&dummy,handle,WM_RBUTTONUP,WM_RBUTTONUP,PM_REMOVE);
      return; 
    }
  }
#endif

//wxDebugMsg("RButtonDown\n") ;
  wxMouseEvent event(wxEVENT_TYPE_RIGHT_DOWN);
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  last_x_pos = event.x; last_y_pos = event.y; last_event = wxEVENT_TYPE_RIGHT_DOWN;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnRButtonUp(int x, int y, UINT flags)
{
//wxDebugMsg("RButtonUp\n") ;
  wxMouseEvent event(wxEVENT_TYPE_RIGHT_UP);
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  last_x_pos = event.x; last_y_pos = event.y; last_event = wxEVENT_TYPE_RIGHT_UP;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnRButtonDClick(int x, int y, UINT flags)
{
//wxDebugMsg("RButtonDClick\n") ;
  wxMouseEvent event(wxEVENT_TYPE_RIGHT_DCLICK);
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  last_x_pos = event.x; last_y_pos = event.y; last_event = wxEVENT_TYPE_RIGHT_DCLICK;
  if (wx_window && wx_window->doubleClickAllowed) wx_window->OnEvent(event);
}

#define LEFT_MARGIN 3

void wxSubWnd::OnMouseMove(int x, int y, UINT flags)
{
//wxDebugMsg("Client 0x%08x Move Msg %d,%d\n",this,x,y) ;

#if MOUSE_EXIT_FIX
  HWND hunder ;
  POINT pt ;
  // See if we Leave/Enter the window.
  GetCursorPos(&pt) ;
  hunder = WindowFromPoint(pt) ;
  if (hunder==handle)
  {
    // I'm in the Window, but perhaps in NC area.
    RECT wind ;
    RECT nc ;
    GetClientRect(handle,&wind) ;
    GetWindowRect(handle,&nc) ;
    pt.x -= nc.left ;
    pt.y -= nc.top ;
    wind.left += LEFT_MARGIN ; // In case Window was maximized.
    if (!PtInRect(&wind,pt))
      hunder = NULL ; // So, I simulate a Leave event...
  }

  if (hunder!=handle)
  {
    if (mouse_in_window)
    {
      mouse_in_window = FALSE ;
      ReleaseCapture() ;
      OnMouseLeave(x,y,flags) ;
      return ;
    }
  }
  else
  {
    // Event was triggered while I'm really into my client area.
    // Do an Enter if not done.
    if (!mouse_in_window)
    {
      mouse_in_window = TRUE ;
      SetCapture(handle) ;
      // Set cursor
      if (wx_window->wx_cursor)
        ::SetCursor(wx_window->wx_cursor->ms_cursor);
      OnMouseEnter(x,y,flags) ;
      return ;
    }
  }
#endif
    
  // 'normal' move event...
  // Set cursor
  if (wx_window->wx_cursor)
    ::SetCursor(wx_window->wx_cursor->ms_cursor);

  wxMouseEvent event(wxEVENT_TYPE_MOTION);
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  // Window gets a click down message followed by a mouse move
  // message even if position isn't changed!  We want to discard
  // the trailing move event if x and y are the same.
  if ((last_event == wxEVENT_TYPE_RIGHT_DOWN || last_event == wxEVENT_TYPE_LEFT_DOWN ||
       last_event == wxEVENT_TYPE_MIDDLE_DOWN) &&
      (last_x_pos == event.x && last_y_pos == event.y))
  {
    last_x_pos = event.x; last_y_pos = event.y;
    last_event = wxEVENT_TYPE_MOTION;
    return;
  }

  last_event = wxEVENT_TYPE_MOTION;
  last_x_pos = event.x; last_y_pos = event.y;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnMouseEnter(int x, int y, UINT flags)
{
//wxDebugMsg("Client 0x%08x Enter %d,%d\n",this,x,y) ;

  // Set cursor
  if (wx_window->wx_cursor)
    ::SetCursor(wx_window->wx_cursor->ms_cursor);

  wxMouseEvent event(wxEVENT_TYPE_ENTER_WINDOW);
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  last_event = wxEVENT_TYPE_ENTER_WINDOW;
  last_x_pos = event.x; last_y_pos = event.y;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnMouseLeave(int x, int y, UINT flags)
{
//wxDebugMsg("Client 0x%08x Leave %d,%d\n",this,x,y) ;

  // Set cursor
  if (wx_window->wx_cursor)
    ::SetCursor(wx_window->wx_cursor->ms_cursor);

  wxMouseEvent event(wxEVENT_TYPE_LEAVE_WINDOW);
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.shiftDown = (flags & MK_SHIFT);
  event.controlDown = (flags & MK_CONTROL);
  event.leftDown = (flags & MK_LBUTTON);
  event.middleDown = (flags & MK_MBUTTON);
  event.rightDown = (flags & MK_RBUTTON);

  last_event = wxEVENT_TYPE_LEAVE_WINDOW;
  last_x_pos = event.x; last_y_pos = event.y;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnChar(WORD wParam, Bool isASCII)
{
  int id;
  if (isASCII)
    id = wParam;
  else
    if ((id = wxCharCodeMSWToWX(wParam)) == 0)
      id = -1;

  if ((id > -1) && wx_window)
  {
    wxKeyEvent event(wxEVENT_TYPE_CHAR);
    if (wxShiftDown)
      event.shiftDown = TRUE;
    if (wxControlDown)
      event.controlDown = TRUE;
    event.eventObject = wx_window;
    event.keyCode = id;

    POINT pt ;
    GetCursorPos(&pt) ;
    RECT rect ;
    GetWindowRect(handle,&rect) ;
    pt.x -= rect.left ;
    pt.y -= rect.top ;
    float fx,fy ;
    fx = (float)pt.x ;
    fy = (float)pt.y ;
    DeviceToLogical(&fx,&fy) ;
    CalcUnscrolledPosition((int)fx,(int)fy,&event.x,&event.y) ;

    wx_window->OnChar(event);
  }
}

void wxSubWnd::OnVScroll(WORD wParam, WORD pos, HWND control)
{
	short nScrollInc;
	
	switch ( wParam )
	{
		case SB_TOP:
                        nScrollInc = -yscroll_position;
			break;

		case SB_BOTTOM:
                        nScrollInc = yscroll_lines - yscroll_position;
			break;
			
		case SB_LINEUP:
			nScrollInc = -1;
			break;

		case SB_LINEDOWN:
			nScrollInc = 1;
			break;

		case SB_PAGEUP:
                        nScrollInc = min(-1, -yscroll_lines_per_page);
			break;

		case SB_PAGEDOWN:
                        nScrollInc = max(1, yscroll_lines_per_page);
			break;

		case SB_THUMBTRACK:
                        nScrollInc = pos - yscroll_position;
			break;

		default:
			nScrollInc = 0;
	}

        int w, h;
        RECT rect;
        GetWindowRect(handle, &rect);
        w = rect.right - rect.left;
        h = rect.bottom - rect.top;

        int nMaxHeight = yscroll_lines*yscroll_pixels_per_line;
        int nVscrollMax = max(0, (int)(2 + (nMaxHeight - h)/yscroll_pixels_per_line));
//        int nVscrollMax = max(0, (int)(yscroll_lines + 2 -  h/yscroll_pixels_per_line));
        if ( nScrollInc = max( -yscroll_position,
                        min( nScrollInc, nVscrollMax - yscroll_position ) ) )
	{
          yscroll_position += nScrollInc;

          if (y_scrolling_enabled)
            ScrollWindow(handle, 0, -yscroll_pixels_per_line * nScrollInc, NULL, NULL );
          else
            InvalidateRect(handle, NULL, FALSE);
          SetScrollPos(handle, SB_VERT, yscroll_position, TRUE );
	}
}

void wxSubWnd::OnHScroll( WORD wParam, WORD pos, HWND control)
{
  if (control)
  {
    wxSliderEvent(control, wParam, pos);
  }
  else
  {
	int nScrollInc;
	switch ( wParam )
	{
		case SB_LINEUP:
			nScrollInc = -1;
			break;

		case SB_LINEDOWN:
			nScrollInc = 1;
			break;

		case SB_PAGEUP:
                        nScrollInc = -xscroll_lines_per_page;
			break;

		case SB_PAGEDOWN:
                        nScrollInc = xscroll_lines_per_page;
			break;

                case SB_THUMBTRACK:
                        nScrollInc = pos - xscroll_position;
			break;

		default:
			nScrollInc = 0;
	}
        int w, h;
        RECT rect;
        GetWindowRect(handle, &rect);
        w = rect.right - rect.left;
        h = rect.bottom - rect.top;
        int nMaxWidth = xscroll_lines*xscroll_pixels_per_line;
        int nHscrollMax = max(0, (int)(2 + (nMaxWidth - w)/xscroll_pixels_per_line));

        if ( nScrollInc = max( -xscroll_position,
                         min( nScrollInc, nHscrollMax - xscroll_position ) ) )
	{
          xscroll_position += nScrollInc;
          if (x_scrolling_enabled)
            ScrollWindow(handle, -xscroll_pixels_per_line * nScrollInc, 0, NULL, NULL );
          else
            InvalidateRect(handle, NULL, FALSE);
          SetScrollPos(handle, SB_HORZ, xscroll_position, TRUE );
	}
  }
}

void wxGetCharSize(HWND wnd, int *x, int *y,wxFont *the_font)
{
  TEXTMETRIC tm;
  HDC dc = GetDC(wnd);
  HFONT fnt =0;
  HFONT was = 0;
  if (the_font&&(fnt=the_font->GetInternalFont(dc)))
    was = SelectObject(dc,fnt) ;
  GetTextMetrics(dc, &tm);
  if (the_font && fnt && was)
    SelectObject(dc,was) ;
  ReleaseDC(wnd, dc);
  *x = tm.tmAveCharWidth;
  *y = tm.tmHeight + tm.tmExternalLeading;
}

// Returns 0 if was a normal ASCII value, not a special key. This indicates that
// the key should be ignored by WM_KEYDOWN and processed by WM_CHAR instead.
int wxCharCodeMSWToWX(int keySym)
{
  int id = 0;
  switch (keySym)
  {
    case VK_CANCEL:             id = WXK_CANCEL; break;
    case VK_BACK:               id = WXK_BACK; break;
    case VK_TAB:	        id = WXK_TAB; break;
    case VK_CLEAR:		id = WXK_CLEAR; break;
    case VK_RETURN:		id = WXK_RETURN; break;
    case VK_SHIFT:		id = WXK_SHIFT; break;
    case VK_CONTROL:		id = WXK_CONTROL; break;
    case VK_MENU :		id = WXK_MENU; break;
    case VK_PAUSE:		id = WXK_PAUSE; break;
    case VK_SPACE:		id = WXK_SPACE; break;
    case VK_ESCAPE:		id = WXK_ESCAPE; break;
    case VK_PRIOR:		id = WXK_PRIOR; break;
    case VK_NEXT :		id = WXK_NEXT; break;
    case VK_END:		id = WXK_END; break;
    case VK_HOME :		id = WXK_HOME; break;
    case VK_LEFT :		id = WXK_LEFT; break;
    case VK_UP:		        id = WXK_UP; break;
    case VK_RIGHT:		id = WXK_RIGHT; break;
    case VK_DOWN :		id = WXK_DOWN; break;
    case VK_SELECT:		id = WXK_SELECT; break;
    case VK_PRINT:		id = WXK_PRINT; break;
    case VK_EXECUTE:		id = WXK_EXECUTE; break;
    case VK_INSERT:		id = WXK_INSERT; break;
    case VK_DELETE:		id = WXK_DELETE; break;
    case VK_HELP :		id = WXK_HELP; break;
    case VK_NUMPAD0:		id = WXK_NUMPAD0; break;
    case VK_NUMPAD1:		id = WXK_NUMPAD1; break;
    case VK_NUMPAD2:		id = WXK_NUMPAD2; break;
    case VK_NUMPAD3:		id = WXK_NUMPAD3; break;
    case VK_NUMPAD4:		id = WXK_NUMPAD4; break;
    case VK_NUMPAD5:		id = WXK_NUMPAD5; break;
    case VK_NUMPAD6:		id = WXK_NUMPAD6; break;
    case VK_NUMPAD7:		id = WXK_NUMPAD7; break;
    case VK_NUMPAD8:		id = WXK_NUMPAD8; break;
    case VK_NUMPAD9:		id = WXK_NUMPAD9; break;
    case VK_MULTIPLY:		id = WXK_MULTIPLY; break;
    case VK_ADD:		id = WXK_ADD; break;
    case VK_SUBTRACT:		id = WXK_SUBTRACT; break;
    case VK_DECIMAL:		id = WXK_DECIMAL; break;
    case VK_DIVIDE:		id = WXK_DIVIDE; break;
    case VK_F1:		id = WXK_F1; break;
    case VK_F2:		id = WXK_F2; break;
    case VK_F3:		id = WXK_F3; break;
    case VK_F4:		id = WXK_F4; break;
    case VK_F5:		id = WXK_F5; break;
    case VK_F6:		id = WXK_F6; break;
    case VK_F7:		id = WXK_F7; break;
    case VK_F8:		id = WXK_F8; break;
    case VK_F9:		id = WXK_F9; break;
    case VK_F10:		id = WXK_F10; break;
    case VK_F11:		id = WXK_F11; break;
    case VK_F12:		id = WXK_F12; break;
    case VK_F13:		id = WXK_F13; break;
    case VK_F14:		id = WXK_F14; break;
    case VK_F15:		id = WXK_F15; break;
    case VK_F16:		id = WXK_F16; break;
    case VK_F17:		id = WXK_F17; break;
    case VK_F18:		id = WXK_F18; break;
    case VK_F19:		id = WXK_F19; break;
    case VK_F20:		id = WXK_F20; break;
    case VK_F21:		id = WXK_F21; break;
    case VK_F22:		id = WXK_F22; break;
    case VK_F23:		id = WXK_F23; break;
    case VK_F24:		id = WXK_F24; break;
    case VK_NUMLOCK:		id = WXK_NUMLOCK; break;
    case VK_SCROLL:		id = WXK_SCROLL; break;
    default:
    {
      return 0;
    }
  }
  return id;
}

int wxCharCodeWXToMSW(int id, Bool *isVirtual)
{
  *isVirtual = TRUE;
  int keySym = 0;
  switch (id)
  {
    case WXK_CANCEL:            keySym = VK_CANCEL; break;
    case WXK_CLEAR:		keySym = VK_CLEAR; break;
    case WXK_SHIFT:		keySym = VK_SHIFT; break;
    case WXK_CONTROL:		keySym = VK_CONTROL; break;
    case WXK_MENU :		keySym = VK_MENU; break;
    case WXK_PAUSE:		keySym = VK_PAUSE; break;
    case WXK_PRIOR:		keySym = VK_PRIOR; break;
    case WXK_NEXT :		keySym = VK_NEXT; break;
    case WXK_END:		keySym = VK_END; break;
    case WXK_HOME :		keySym = VK_HOME; break;
    case WXK_LEFT :		keySym = VK_LEFT; break;
    case WXK_UP:		keySym = VK_UP; break;
    case WXK_RIGHT:		keySym = VK_RIGHT; break;
    case WXK_DOWN :		keySym = VK_DOWN; break;
    case WXK_SELECT:		keySym = VK_SELECT; break;
    case WXK_PRINT:		keySym = VK_PRINT; break;
    case WXK_EXECUTE:		keySym = VK_EXECUTE; break;
    case WXK_INSERT:		keySym = VK_INSERT; break;
    case WXK_DELETE:		keySym = VK_DELETE; break;
    case WXK_HELP :		keySym = VK_HELP; break;
    case WXK_NUMPAD0:		keySym = VK_NUMPAD0; break;
    case WXK_NUMPAD1:		keySym = VK_NUMPAD1; break;
    case WXK_NUMPAD2:		keySym = VK_NUMPAD2; break;
    case WXK_NUMPAD3:		keySym = VK_NUMPAD3; break;
    case WXK_NUMPAD4:		keySym = VK_NUMPAD4; break;
    case WXK_NUMPAD5:		keySym = VK_NUMPAD5; break;
    case WXK_NUMPAD6:		keySym = VK_NUMPAD6; break;
    case WXK_NUMPAD7:		keySym = VK_NUMPAD7; break;
    case WXK_NUMPAD8:		keySym = VK_NUMPAD8; break;
    case WXK_NUMPAD9:		keySym = VK_NUMPAD9; break;
    case WXK_MULTIPLY:		keySym = VK_MULTIPLY; break;
    case WXK_ADD:		keySym = VK_ADD; break;
    case WXK_SUBTRACT:		keySym = VK_SUBTRACT; break;
    case WXK_DECIMAL:		keySym = VK_DECIMAL; break;
    case WXK_DIVIDE:		keySym = VK_DIVIDE; break;
    case WXK_F1:		keySym = VK_F1; break;
    case WXK_F2:		keySym = VK_F2; break;
    case WXK_F3:		keySym = VK_F3; break;
    case WXK_F4:		keySym = VK_F4; break;
    case WXK_F5:		keySym = VK_F5; break;
    case WXK_F6:		keySym = VK_F6; break;
    case WXK_F7:		keySym = VK_F7; break;
    case WXK_F8:		keySym = VK_F8; break;
    case WXK_F9:		keySym = VK_F9; break;
    case WXK_F10:		keySym = VK_F10; break;
    case WXK_F11:		keySym = VK_F11; break;
    case WXK_F12:		keySym = VK_F12; break;
    case WXK_F13:		keySym = VK_F13; break;
    case WXK_F14:		keySym = VK_F14; break;
    case WXK_F15:		keySym = VK_F15; break;
    case WXK_F16:		keySym = VK_F16; break;
    case WXK_F17:		keySym = VK_F17; break;
    case WXK_F18:		keySym = VK_F18; break;
    case WXK_F19:		keySym = VK_F19; break;
    case WXK_F20:		keySym = VK_F20; break;
    case WXK_F21:		keySym = VK_F21; break;
    case WXK_F22:		keySym = VK_F22; break;
    case WXK_F23:		keySym = VK_F23; break;
    case WXK_F24:		keySym = VK_F24; break;
    case WXK_NUMLOCK:		keySym = VK_NUMLOCK; break;
    case WXK_SCROLL:		keySym = VK_SCROLL; break;
    default:
    {
      *isVirtual = FALSE;
      keySym = id;
      break;
    }
  }
  return keySym;
}
