/*
 * File:	wb_item.cc
 * Purpose:	Panel items implementation
 * Author:	Julian Smart
 * Created:	1993
 * Updated:	
 * Copyright:	(c) 1993, AIAI, University of Edinburgh
 */

static const char sccsid[] = "%W% %G%";


/* When implementing a new item, be sure to:
 *
 * - add the item to the parent panel
 * - set window_parent to the parent
 * - NULL any extra child window pointers not created for this item
 *   (e.g. label control that wasn't needed)
 * - delete any extra child windows in the destructor (e.g. label control)
 * - implement GetSize and SetSize
 * - to find panel position if coordinates are (-1, -1), use GetPosition
 * - call AdvanceCursor after creation, for panel layout mechanism.
 *
 */

/*
 Motif notes

 A panel is a form.
 Each item is created on a RowColumn or Form of its own, to allow a label to
 be positioned. wxListBox and wxMultiText have forms, all the others have RowColumns.
 This is to allow labels to be positioned to the top left (can't do it with a
 RowColumn as far as I know).
 AttachWidget positions widgets relative to one another (left->right, top->bottom)
 unless the x, y coordinates are given (more than -1).
 */

/* Uncomment for Borland/UNIX combination
#ifdef wx_msw
#include "wx.h"
#pragma hdrstop
#else

#include "common.h"

#include "wx_item.h"
#include "wx_lbox.h"
#include "wx_buttn.h"
#include "wx_choic.h"
#include "wx_check.h"
#include "wx_messg.h"
#include "wx_slidr.h"
#include "wx_menu.h"
#include "wx_txt.h"
#include "wx_mtxt.h"
#include "wx_stdev.h"
#include "wx_utils.h"
#include "wx_main.h"
#include "wx_frame.h"
#endif
*/

#include "wx.h"                 // For MSC 7/UNIX combination

#include "wx_stdev.h"

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>

// Item members
wxbItem::wxbItem(void)
{
  __type = wxTYPE_ITEM;
}

wxbItem::~wxbItem(void)
{
  wxPanel *parent = (wxPanel *)GetParent();
  if (parent && (parent->defaultItem == this))
    parent->defaultItem = NULL;
}

void wxbItem::SetClientSize(int width, int height)
{
  SetSize(-1, -1, width, height);
}

int wxbItem::GetLabelPosition(void)
{
  return labelPosition;
}

void wxbItem::SetLabelPosition(int pos)
{
  labelPosition = pos;
}

void wxbItem::Centre(int direction)
{
  int x, y, width, height, panel_width, panel_height, new_x, new_y;

  wxPanel *panel = (wxPanel *)GetParent();
  if (!panel)
    return;

  panel->GetClientSize(&panel_width, &panel_height);
  GetSize(&width, &height);
  GetPosition(&x, &y);

  new_x = x;
  new_y = y;

  if (direction == wxHORIZONTAL || direction == wxBOTH)
    new_x = (int)((panel_width - width)/2);

  if (direction == wxVERTICAL || direction == wxBOTH)
    new_y = (int)((panel_height - height)/2);

  SetSize(new_x, new_y, width, height);
  int temp_x, temp_y;
  GetPosition(&temp_x, &temp_y);
  GetPosition(&temp_x, &temp_y);
}

wxbButton::wxbButton(void)
{
  __type = wxTYPE_BUTTON;
  window_parent = NULL;
  labelPosition = wxHORIZONTAL;
  buttonFont = NULL ;
  labelFont = NULL ;
}

wxbButton::wxbButton(wxPanel *panel, wxFunction Function, char *label,
		   int x, int y, int width, int height, int style, char *name)
{
  __type = wxTYPE_BUTTON;
  window_parent = panel;
  labelPosition = wxHORIZONTAL;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}

wxbButton::wxbButton(wxPanel *panel, wxFunction Function, wxBitmap *bitmap,
		   int x, int y, int width, int height, int style, char *name)
{
  __type = wxTYPE_BUTTON;
  window_parent = panel;
  labelPosition = wxHORIZONTAL;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}

wxbButton::~wxbButton(void)
{
}

void wxbButton::Command(wxCommandEvent& event)
{
  ProcessCommand(event);
}

void wxbButton::ProcessCommand(wxCommandEvent& event)
{
  if (wxNotifyEvent(event, TRUE))
    return;

  wxFunction fun = callback;
  if (fun)
  {
    (void)(*(fun))(*this, event);
  }

  wxNotifyEvent(event, FALSE);
}

// Menus

// Construct a menu with optional title (then use append)
wxbMenu::wxbMenu(char *Title, wxFunction func)
{
  __type = wxTYPE_MENU;
  no_items = 0;
  menu_bar = NULL;
  if (Title)
    title = copystring(Title);
  else
    title = NULL;
}

// The wxWindow destructor will take care of deleting the submenus.
wxbMenu::~wxbMenu(void)
{
  if (title)
    delete[] title;
}

// Finds the item id matching the given string, -1 if not found.
int wxbMenu::FindItem(char *itemString)
{
  char buf1[200];
  char buf2[200];
  wxStripMenuCodes(itemString, buf1);

  wxNode *node = menuItems.First();
  while (node)
  {
    wxMenuItem *item = (wxMenuItem *)node->Data();
    if (item->subMenu)
    {
      int ans = item->subMenu->FindItem(itemString);
      if (ans > -1)
        return ans;
    }
    if ((item->itemId > -1) && item->itemName)
    {
      wxStripMenuCodes(item->itemName, buf2);
      if (strcmp(buf1, buf2) == 0)
        return item->itemId;
    }
    node = node->Next();
  }
  return -1;
}

wxMenuItem *wxbMenu::FindItemForId(int itemId)
{
  wxNode *node = menuItems.First();
  while (node)
  {
    wxMenuItem *item = (wxMenuItem *)node->Data();

    if (item->itemId == itemId)
      return item;

    if (item->subMenu)
    {
      wxMenuItem *ans = item->subMenu->FindItemForId(itemId);
      if (ans)
        return ans;
    }

    node = node->Next();
  }
  return NULL;
}

void wxbMenu::SetHelpString(int itemId, char *helpString)
{
  wxMenuItem *item = FindItemForId(itemId);
  if (item)
  {
    if (item->helpString)
      delete[] item->helpString;
    item->helpString = copystring(helpString);
  }
}

char *wxbMenu::GetHelpString(int itemId)
{
  wxMenuItem *item = FindItemForId(itemId);
  if (item)
    return item->helpString;
  else return NULL;
}

void wxbMenu::ProcessCommand(wxCommandEvent& event)
{
  if (wxNotifyEvent(event, TRUE))
    return;

  wxFunction fun = callback;
  if (fun)
  {
    (void)(*(fun))(*this, event);
  }

  wxNotifyEvent(event, FALSE);
}

// Menu Bar

wxbMenuBar::wxbMenuBar(void)
{
  __type = wxTYPE_MENU_BAR;
  n = 0;
  menus = NULL;
  titles = NULL;
  menu_bar_frame = NULL;
}

wxbMenuBar::wxbMenuBar(int N, wxMenu *Menus[], char *Titles[])
{
  __type = wxTYPE_MENU_BAR;
  n = N;
  menus = Menus;
  titles = Titles;
  menu_bar_frame = NULL;
  for (int i = 0; i < N; i++)
    menus[i]->menu_bar = (wxMenuBar *)this;
}

wxbMenuBar::~wxbMenuBar(void)
{
}

void wxbMenuBar::Append(wxMenu *menu, char *title)
{
  n ++;
  wxMenu **new_menus = new wxMenu *[n];
  char **new_titles = new char *[n];

  int i;
  for (i = 0; i < n - 1; i++)
    {
      new_menus[i] = menus[i];
      menus[i] = NULL;
      new_titles[i] = titles[i];
      titles[i] = NULL;
    }
  if (menus)
   {
     delete[] menus;
     delete[] titles;
   }
  menus = new_menus;
  titles = new_titles;

  menus[n-1] = menu;
  titles[n-1] = copystring(title);

  menu->menu_bar = (wxMenuBar *)this;
}

// Find the menu menuString, item itemString, and return the item id.
// Returns -1 if none found.
int wxbMenuBar::FindMenuItem(char *menuString, char *itemString)
{
  char buf1[200];
  char buf2[200];
  wxStripMenuCodes(menuString, buf1);
  for (int i = 0; i < n; i++)
  {
    wxStripMenuCodes(titles[i], buf2);
    if (strcmp(buf1, buf2) == 0)
      return menus[i]->FindItem(itemString);
  }
  return -1;
}

wxMenuItem *wxbMenuBar::FindItemForId(int Id)
{
  wxMenuItem *item = NULL;
  for (int i = 0; i < n; i++)
    if (item = menus[i]->FindItemForId(Id))
      return item;
  return NULL;
}

void wxbMenuBar::SetHelpString(int Id, char *helpString)
{
  for (int i = 0; i < n; i++)
  {
    if (menus[i]->FindItemForId(Id))
    {
      menus[i]->SetHelpString(Id, helpString);
      return;
    }
  }
}

char *wxbMenuBar::GetHelpString(int Id)
{
  for (int i = 0; i < n; i++)
  {
    if (menus[i]->FindItemForId(Id))
      return menus[i]->GetHelpString(Id);
  }
  return NULL;
}

void wxbFrame::SetMenuBar(wxMenuBar *menu_bar)
{
}

// Single check box item
wxbCheckBox::wxbCheckBox(void)
{
  __type = wxTYPE_CHECK_BOX;
  buttonFont = NULL ;
  labelFont = NULL ;
}

wxbCheckBox::wxbCheckBox(wxPanel *panel, wxFunction func, char *Title,
                       int x, int y, int width, int height, int style, char *name)
{
  __type = wxTYPE_CHECK_BOX;
  window_parent = panel;
  labelPosition = panel->label_position;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}

wxbCheckBox::wxbCheckBox(wxPanel *panel, wxFunction func, wxBitmap *bitmap,
                       int x, int y, int width, int height, int style, char *name)
{
  __type = wxTYPE_CHECK_BOX;
  window_parent = panel;
  labelPosition = panel->label_position;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}

wxbCheckBox::~wxbCheckBox(void)
{
}

void wxbCheckBox::Command(wxCommandEvent& event)
{
  SetValue(event.commandInt);
  ProcessCommand(event);
}

void wxbCheckBox::ProcessCommand(wxCommandEvent& event)
{
  if (wxNotifyEvent(event, TRUE))
    return;

  wxFunction fun = callback;
  if (fun)
  {
    (void)(*(fun))(*this, event);
  }

  wxNotifyEvent(event, FALSE);
}

wxbChoice::wxbChoice(void)
{
  __type = wxTYPE_CHOICE;
  buttonFont = NULL ;
  labelFont = NULL ;
}

wxbChoice::wxbChoice(wxPanel *panel, wxFunction func, char *Title,
                   int x, int y, int width, int height, int N, char **Choices,
                   int style, char *name)
{
  __type = wxTYPE_CHOICE;
  window_parent = panel;
  labelPosition = panel->label_position;
  no_strings = N;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}

wxbChoice::~wxbChoice(void)
{
}

char *wxbChoice::GetStringSelection(void)
{
  int sel = GetSelection();
  if (sel > -1)
    return this->GetString(sel);
  else
    return NULL;
}

Bool wxbChoice::SetStringSelection(char *s)
{
  int sel = FindString(s);
  if (sel > -1)
  {
    SetSelection(sel);
    return TRUE;
  }
  else return FALSE;
}

void wxbChoice::Command(wxCommandEvent& event)
{
  SetSelection(event.commandInt);
  ProcessCommand(event);
}

void wxbChoice::ProcessCommand(wxCommandEvent& event)
{
  if (wxNotifyEvent(event, TRUE))
    return;

  wxFunction fun = callback;
  if (fun)
  {
    (void)(*(fun))(*this, event);
  }

  wxNotifyEvent(event, FALSE);
}

// Listbox item
wxbListBox::wxbListBox(void)
{
  __type = wxTYPE_LIST_BOX;
  selected = -1;
  selections = 0;
  no_items = 0;
  buttonFont = NULL ;
  labelFont = NULL ;
}

wxbListBox::wxbListBox(wxPanel *panel, wxFunction func,
                       char *Title, Bool Multiple,
                       int x, int y, int width, int height,
                       int N, char **Choices, int style, char *name)
{
  __type = wxTYPE_LIST_BOX;
  selected = -1;
  selections = 0;
  multiple = Multiple;
  window_parent = panel;
  no_items = 0;
  labelPosition = panel->label_position;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}

wxbListBox::~wxbListBox(void)
{
}

int wxbListBox::Number(void)
{
  return no_items;
}

// For single selection items only
char *wxbListBox::GetStringSelection(void)
{
  int sel = GetSelection();
  if (sel > -1)
    return this->GetString(sel);
  else
    return NULL;
}

Bool wxbListBox::SetStringSelection(char *s)
{
  int sel = FindString(s);
  if (sel > -1)
  {
    SetSelection(sel);
    return TRUE;
  }
  else return FALSE;
}

// Is this the right thing? Won't setselection generate a command
// event too? No! It'll just generate a setselection event.
// But we still can't have this being called whenever a real command
// is generated, because it sets the selection, which will already
// have been done! (Unless we have an optional argument for calling
// by the actual window system, or a separate function, ProcessCommand)
void wxbListBox::Command(wxCommandEvent& event)
{
  if (event.extraLong)
    SetSelection(event.commandInt);
  else
  {
    Deselect(event.commandInt);
    return;
  }
  ProcessCommand(event);
}

void wxbListBox::ProcessCommand(wxCommandEvent& event)
{
  if (wxNotifyEvent(event, TRUE))
    return;

  wxFunction fun = callback;

  if (fun)
  {
    (void)(*(fun))(*this, event);
  }

  wxNotifyEvent(event, FALSE);
}

// Radiobox item
wxbRadioBox::wxbRadioBox(void)
{
  __type = wxTYPE_RADIO_BOX;
  selected = -1;
  no_items = 0;
  buttonFont = NULL ;
  labelFont = NULL ;
}

wxbRadioBox::wxbRadioBox(wxPanel *panel, wxFunction func,
                       char *Title,
                       int x, int y, int width, int height,
                       int N, char **Choices,
                       int majorDim,int style, char *name)
{
  __type = wxTYPE_RADIO_BOX;
  selected = -1;
  window_parent = panel;
  no_items = 0;
  labelPosition = panel->label_position;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}

#ifndef __turboc
wxbRadioBox::wxbRadioBox(wxPanel *panel, wxFunction func,
                       char *Title,
                       int x, int y, int width, int height,
                       int N, wxBitmap **Choices,
                       int majorDim,int style, char *name)
{
  __type = wxTYPE_RADIO_BOX;
  selected = -1;
  window_parent = panel;
  no_items = 0;
  labelPosition = panel->label_position;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}
#endif

wxbRadioBox::~wxbRadioBox(void)
{
}

int wxbRadioBox::Number(void)
{
  return no_items;
}

// For single selection items only
char *wxbRadioBox::GetStringSelection(void)
{
  int sel = GetSelection();
  if (sel > -1)
    return this->GetString(sel);
  else
    return NULL;
}

Bool wxbRadioBox::SetStringSelection(char *s)
{
  int sel = FindString(s);
  if (sel > -1)
  {
    SetSelection(sel);
    return TRUE;
  }
  else return FALSE;
}

void wxbRadioBox::Command(wxCommandEvent& event)
{
  SetSelection(event.commandInt);
  ProcessCommand(event);
}

void wxbRadioBox::ProcessCommand(wxCommandEvent& event)
{
  if (wxNotifyEvent(event, TRUE))
    return;

  wxFunction fun = callback;

  if (fun)
  {
    (void)(*(fun))(*this, event);
  }

  wxNotifyEvent(event, FALSE);
}

// Message
wxbMessage::wxbMessage(void)
{
  __type = wxTYPE_MESSAGE;
  buttonFont = NULL ;
  labelFont = NULL ;
}

wxbMessage::wxbMessage(wxPanel *panel, char *label, int x, int y, int style, char *name)
{
  __type = wxTYPE_MESSAGE;
  window_parent = panel;
  labelPosition = panel->label_position;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}

wxbMessage::~wxbMessage(void)
{
}

// Text item
wxbText::wxbText(void)
{
  __type = wxTYPE_TEXT;
  buttonFont = NULL ;
  labelFont = NULL ;
}

wxbText::wxbText(wxPanel *panel, wxFunction Function, char *label, char *value,
               int x, int y, int width, int height, int style, char *name)
{
  __type = wxTYPE_TEXT;
  window_parent = panel;
  labelPosition = panel->label_position;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}

wxbText::~wxbText(void)
{
}


void wxbText::Command(wxCommandEvent& event)
{
  SetValue(event.commandString);
  ProcessCommand(event);
}

void wxbText::ProcessCommand(wxCommandEvent& event)
{
  if (wxNotifyEvent(event, TRUE))
    return;

  wxFunction fun = callback;
  if (fun)
  {
    (void)(*(fun))(*this, event);
  }

  wxNotifyEvent(event, FALSE);
}

// Multi-line Text item
wxbMultiText::wxbMultiText(void)
{
  __type = wxTYPE_MULTI_TEXT;
  buttonFont = NULL ;
  labelFont = NULL ;
}

wxbMultiText::wxbMultiText(wxPanel *panel, wxFunction Function, char *label, char *value,
               int x, int y, int width, int height, int style, char *name)
{
  __type = wxTYPE_MULTI_TEXT;
  window_parent = panel;
  labelPosition = panel->label_position;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}

void wxbMultiText::Command(wxCommandEvent& event)
{
  SetValue(event.commandString);
  ProcessCommand(event);
}

void wxbMultiText::ProcessCommand(wxCommandEvent& event)
{
  if (wxNotifyEvent(event, TRUE))
    return;

  wxFunction fun = callback;
  if (fun)
  {
    (void)(*(fun))(*this, event);
  }
  wxNotifyEvent(event, FALSE);
}

wxbSlider::wxbSlider(void)
{
  __type = wxTYPE_SLIDER;
  buttonFont = NULL ;
  labelFont = NULL ;
}

wxbSlider::wxbSlider(wxPanel *panel, wxFunction func, char *label, int value,
           int min_value, int max_value, int width, int x, int y, int style, char *name)
{
  __type = wxTYPE_SLIDER;
  window_parent = panel;
  labelPosition = panel->label_position;
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
}

wxbSlider::~wxbSlider(void)
{
}

void wxbSlider::Command(wxCommandEvent& event)
{
  SetValue(event.commandInt);
  ProcessCommand(event);
}

void wxbSlider::ProcessCommand(wxCommandEvent& event)
{
  if (wxNotifyEvent(event, TRUE))
    return;

  wxFunction fun = callback;
  if (fun)
  {
    (void)(*(fun))(*this, event);
  }
  wxNotifyEvent(event, FALSE);
}



