/*
 * lcdbios.h
 *
 * Simulacin de las funciones de LCDBIOS del onHand/Ruputer.
 * Slo las generales y las de GVRAM; las de LCD no interesan.
 *
 * Para usarla, la primera cosa que debe hacer el programa
 * es un
 *         screen(1);
 *
 * Historia:
 *      11/12/00 Termino una primera aproximacin de las funciones.
 *               Estn hechas slo las triviales, quedan 19 "stubs".
 *      25/12/00 Hago una ms: mpread().
 *      29/12/00 Empiezo con lo de las fuentes: gv_kput()
 *      30/12/00 He acabado de depurar lo de las fuentes; gv_kput() ya funciona
 *               y adems he corregido gv_aput(), que no era exactamente tal
 *               y como yo crea (no es estricto lo de primero vert y luego
 *               horiz, sino que va por bytes en vert... un poco raro).
 *               Estn metidas todas las monoespaciadas y las de tipo hora,
 *               mediante el fichero autogenerado lcdfonts.h.
 *      01/01/01 Saco el cuerpo de gv_kput() a mi_gv_kput() y lo generalizo para
 *               que tambin lo pueda usar ruptool.c:getUsWdot().
 *      03/01/01 Empiezo a implementar la parte de E/S de teclado con gii.
 *               Aunque parezca extrao, tambin le pongo la inicializacin
 *               del teclado en screen(), que es la funcin que aseguro que
 *               que se llama al inicio del programa.
 *      07/01/01 Contino con la implementacin del teclado aadiendo la
 *               palabra de estado (equivalente al contenido de los puertos
 *               0x50-0x51 del onHand/Ruputer pero sin los valores de las teclas).
 *      08/01/01 Termino con las estructuras de datos para el teclado, poniendo
 *               el bffer de teclas, la palabra de esado actual de los botones
 *               y las teclas por defecto.
 *      09/02/01 Implemento las reas de parpadeo (BlinkAreas) y las funciones
 *               asociadas, aunque ignoro el valor de velocidad de parpadeo.
 *      11/02/01 Comento en la funcin de salida el que espere una tecla.
 *      13/02/01 Cambio los #define de COPY_PUT, etc a 00:Overwrite,
 *               01:XOR, 10:OR y 11:AND, que es como estn en gv_put().
 *      19/04/01 Aado el que haga honor a XOR_PUT/OR_PUT en gv_square() en
 *               el caso de rectngulos no rellenos. En gv_aput()
 *               XOR_PUT/OR_PUT estan intercambiados en el onhand :-?. Hago
 *               la macros KOR_PUT para gv_kput()/gv_aput()/... que no tienen
 *               XOR_PUT/AND_PUT y adems est en un bit diferente. Quito la
 *               funcionalidad de XOR/AND de kput/aput y dejo la funcionalidad
 *               completa en mi_gv_aput().
 *      21/04/01 Aado soporte para usar la estructura
 *               pSystemCtl->EventManager->eventReq con los tipos de suceso
 *               ("evento") EVENT_SEC, EVENT_MIN, EVENT_HOUR. Estn aqu en vez
 *               de en rupsys.h ya que set trata enporque porque los implemento
 *               por medio de rutina_timer().
 *      22/04/01 Termino de implementar los sucesos EVENT_SEC/MIN/HOUR por
 *               medio de una estructura auxiliar y un gii_event de
 *               pulsacin de la tecla espacio. Defino PCOPY_PUT y dems para
 *               usar con pintar un pxel en negro con pset (COPY_PUT a secas
 *               lo pinta en blanco).
 *       9/06/01 Hago que si est definida la mcro NO_SIGALRM no inicialice
 *               el timer para bliks y events (y facilitar la depuracin).
 *      30/09/01 Implemento que haga un ZOOM, y que por defecto sea 2 (segn un
 *               #define).
 *       4/10/01 Pongo en la estructura de teclado una mscara de teclas en las
 *               que tiene que permitir repeticin si permanecen pulsadas. Aado
 *               el cdigo en la rutina de timer que inserta las teclas de la
 *               mscara (bueno, un fake de ellas) cada vez que se invoca la
 *               rutina. Hago que mientras Teclado.ContadorNoRepeticion sea
 *               mayor que cero no se inserte nada (sino que disminuya dicho
 *               contador).
 *      26/10/01 Aado la gestin de DTAs y la de los directorios a la
 *               estructura de SimFicheros.
 *      27/01/10 Aado el campo simdtaSearchAtrib a las estructuras/defines que
 *               se usan para manipular los DTA.
 *       5/12/01 Arreglos cosmticos para que no de ningn warning cuando se
 *               compila con -Wall.
 *       4/01/02 Hago que la rutina del timer llame a input_buffer(0).
 *       6/01/02 Hago que use SimTeclado.BanderaInputBuffer para evitar
 *               reentrada en input_buffer(0) desde la rutina_timer() (es un
 *               que est a 1 si se encuentra dentro de dicha funcin).
 *
 * NOTA: Parece que el LCD driver tiene como nombre SED1560 (bioscall.def y en TCGctrl.s en WallCrush).
 */

#ifndef _LCDBIOS_H
#define _LCDBIOS_H

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <dirent.h>
#include <asm/bitops.h>
#include <ggi/ggi.h>
#include <ggi/types.h>
#include <ggi/gii.h>
#include <ggi/keyboard.h>


#include "lcdfonts.h"

#ifndef ZOOM
#define ZOOM 2
#endif

/* Definicin de estructuras de la LCDBIOS*/
struct moji {
        int w;
        int h;
        long add;
};

typedef struct {
        int x1;
        int y1;
        int x2;
        int y2;
} RECT;

typedef struct {
        RECT rect;
        int color;
} BLINK_DRAW;

typedef struct {
        int x;
        int y;
} POINT;

typedef struct {
        POINT pos;
        char *str;
        int font;
        int gap;
        int lop;
} KANJI_PUT;

typedef struct {
        POINT pos;
        unsigned short code;
        int lop;
} SIICODE_PUT;

typedef struct {
        POINT pos;
        int width;
        int hight;
        unsigned long addr;
        int lop;
} FONT_PUT;

typedef struct {
        POINT pos;
        char *name;
        int lop;
} MMAP_PUT;

typedef struct {
        POINT st;
        POINT ed;
        int lop;
        unsigned char txt;
} LINE_DRAW;

typedef struct {
        RECT rect;
        int lop;
        unsigned long txt;
} SQUARE_DRAW;

typedef struct {
        int xr;
        int yr;
} RADS;

typedef struct {
        int d1;
        int d2;
} DEGR;

typedef struct {
        POINT pos;
        RADS rad;
        DEGR deg;
        int lop;
        unsigned long txt;
        unsigned char *buf;
} CIRCLE_DRAW;

typedef struct {
        POINT pos;
        int lop;
} POINT_DRAW;

typedef struct {
        RECT rect;
        char *buf;
} GVRAM_GET;

typedef struct {
        POINT pos;
        char *buf;
        int lop;
} GVRAM_PUT;

/* Prototipos de funciones de la LCDBIOS: generales*/
int screen(int mode);
void funcion_salida(void);
void input_ayuda(void);
int getscreen(void);
int cls(int mode);
int lcdon(int power);
int getlcdon(void);
int mpread(char *buf,char *name);
struct moji *mojisize(unsigned short s_code);
int getbufsize(int x1,int y1,int x2,int y2);
int pgetbufsize(RECT *rct);
int reverse(int x1,int y1,int x2,int y2 );
int preverse(RECT *rct);
int revstart(int num);
int revstop(int num);
int revend(int num);
int revdraw(void);
int blink(int x1,int y1,int x2,int y2,int color);
int pblink(BLINK_DRAW *blk);
int blstop(int num);
int blstart(int num);
int blend(int num);
int lcdfreeze(int mode);
int lcdvol(int vol);
int getlcdvol(void);

/* Prototipos de funciones de la LCDBIOS: GVRAM*/
int gv_place(int x,int y);
int pv_place(POINT *ppt);
int gv_kput(int x,int y,char *str,int font,int gap,int lop);
int pv_kput(KANJI_PUT *kpt);
int gv_sput(int x,int y,unsigned short code,int lop);
int pv_sput(SIICODE_PUT *spt);
int gv_aput(int x,int y,int w,int h,unsigned long addr,int lop);
int pv_aput(FONT_PUT *apt);
int gv_mmap(int x,int y,char *name,int lop);
int pv_mmap(MMAP_PUT *mpt);
int gv_line(int x1,int y1,int x2,int y2,int lop,unsigned char style);
int pv_line(LINE_DRAW *lpt);
int gv_square(int x1,int y1,int x2,int y2,int lop,unsigned long style);
int pv_square(SQUARE_DRAW *spt);
int gv_circle(int x,int y,int xr,int yr,int d1,int d2,int lop,unsigned long style,unsigned char *buf);
int pv_circle(CIRCLE_DRAW *cpt);
int gv_pset(int x,int y,int lop);
int pv_pset(POINT_DRAW *ppt);
int gv_point(int x,int y);
int pv_point(POINT *ppt);
int gv_get(int x1,int y1,int x2,int y2,char *buf);
int pv_get(GVRAM_GET *gpt);
int gv_put(int x,int y,char *buf,int logic);
int pv_put(GVRAM_PUT *gpt);
int gv_scroll(int lines);
int gv_reverse(int x1,int y1,int x2,int y2);
int pv_reverse(RECT *rpt);
int gv_clear(int x1,int y1,int x2,int y2);
int pv_clear(RECT *cpt);

/* funciones de ayuda */
int mi_gv_aput(int x,int y,int w,int h,unsigned long addr,int lop);
int miHLine(int x1,int y1,int ex, int lop);
int miVLine(int x1,int y1,int ey, int lop);

/* Datos globales */
typedef struct {
        char *Altura;
        char *NumRangos;
        char *Rangos;
        short *PosAbs;
        char *Datos;
} sFuente;

typedef struct {
        char Activada;
        char Periodo;
        char Operacion;
        int x1,y1,x2,y2;
} sBlinkArea;

#define MAX_NUM_BLINK_AREAS 16

struct sSimPantalla {
        int Init;
        int PosX,PosY;
        char Congelado;
        char Parpadeando;
        ggi_visual_t Salida;
        ggi_color ColorBlanco,ColorNegro;
        ggi_pixel Blanco,Negro;
        sFuente Fuente[61];
        int NumBlinkActivos;
        sBlinkArea Blink[MAX_NUM_BLINK_AREAS];
        struct itimerval TimerSistema;
} SimPantalla = {0};

#define MAX_NUM_TECLAS 20

struct sSimTeclado {
        ggi_visual_t Entrada;
        unsigned short PalabraEstado;
        unsigned short NumBuffer;
        unsigned short Buffer[MAX_NUM_TECLAS];
        unsigned short BotonesActivos;
        unsigned short MascaraBotonesRepeticion;
        unsigned char ContadorNoRepeticion;
        unsigned long BanderaInputBuffer;
        unsigned long NumEvSec,NumEvMin,NumEvHour;
        char *TeclasBits;
} SimTeclado = {0};

#define INPUT_ARR 'W'
#define INPUT_ABJ 'X'
#define INPUT_DER 'D'
#define INPUT_IZQ 'A'
#define INPUT_FILER 'Z'
#define INPUT_ENTER 'E'
#define INPUT_EL    'Q'
#define INPUT_MENU  'C'
char TeclasDefecto[]={INPUT_FILER,INPUT_MENU,INPUT_EL,INPUT_ENTER,INPUT_IZQ,INPUT_ARR,INPUT_DER,INPUT_ABJ,'\0'};


#define SIMDTADESP(x,desp) (((char *)x)+(desp))

#define SIMDTADIR(x) (*((DIR **)SIMDTADESP(x,simdtaDIR)))
#define SIMDTAPATH(x) (*((char **)SIMDTADESP(x,simdtaPath)))
#define SIMDTAMASK(x) (*((char **)SIMDTADESP(x,simdtaMask)))
#define SIMDTASEARCHATRIB(x) (*((char *)SIMDTADESP(x,simdtaSearchatrib)))
#define SIMDTARESERVED(x) (*((char **)SIMDTADESP(x,simdtaReserved)))
#define SIMDTATIME(x) (*((short *)SIMDTADESP(x,simdtaTime)))
#define SIMDTADATE(x) (*((short *)SIMDTADESP(x,simdtaDate)))
#define SIMDTASIZE(x) (*((long *)SIMDTADESP(x,simdtaSize)))
#define SIMDTAATRIB(x) (*((char *)SIMDTADESP(x,simdtaAtrib)))
#define SIMDTANAME(x) ((char *)SIMDTADESP(x,simdtaName))

enum EnumSimDespDta {
        simdtaDIR=0,
        simdtaPath=4,
        simdtaMask=8,
        simdtaSearchatrib=12,
        simdtaReserved=13,
        simdtaTime=22,
        simdtaDate=24,
        simdtaSize=26,
        simdtaAtrib=30,
        simdtaName=31,
};

#define MAX_NUM_FICHEROS 20
struct sSimFicheros {
        int NumAbiertos;
        char BanderaEnUso[MAX_NUM_FICHEROS];
        FILE *Manejadores[MAX_NUM_FICHEROS];
        char DirectorioRaiz[256];
        char SubdirectorioActual[256];
        char DtaInicial[44], *DtaActual;
} SimFicheros = {0};

#define EVENT_SEC 0x00040000L
#define EVENT_MIN 0x20000000L
#define EVENT_HOUR 0x40000000L

typedef struct {
        long eventReq;
} EVENTMANAGE;

typedef struct {
        volatile EVENTMANAGE *EventManager;
} SYSTEMCTL;

volatile EVENTMANAGE miEventManager = {0};
volatile SYSTEMCTL SystemCtl={&miEventManager},*pSystemCtl=&SystemCtl;


/* Definicin de macros */
#define COLOR2BIT(color) ((SimPantalla.Blanco==(color))?0:1)
#define BIT2COLOR(bit) (((bit)==0)?SimPantalla.Blanco:SimPantalla.Negro)
#ifndef COPY_PUT
#define COPY_PUT 0
#define XOR_PUT  1
#define OR_PUT   2
#define AND_PUT  3
#endif

#ifndef KCOPY_PUT
#define KCOPY_PUT 0
#define KOR_PUT   1
#endif

#ifndef PCOPY_PUT
#define PCOPY_PUT (COPY_PUT | 4)
#define PXOR_PUT  (XOR_PUT | 4)
#define POR_PUT   (OR_PUT | 4)
#define PAND_PUT  (AND_PUT | 4)
#endif

/* Fake de las funciones lcd* del Ruputer */

#define DEFINE_COLOR(miggi_color,mir,mig,mib) miggi_color.r=mir,miggi_color.g=mig,miggi_color.b=mib,miggi_color.a=0;
#define DEFINE_COLOR256(miggi_color,mir,mig,mib) miggi_color.r=((unsigned int)mir)<<8,miggi_color.g=((unsigned int)mig)<<8,miggi_color.b=((unsigned int)mib)<<8,miggi_color.a=0;
#define DEFINE_MODE(miggi_mode,miframes,mix,miy,mixv,miyv) miggi_mode.frames=miframes,miggi_mode.visible.x=mix,miggi_mode.visible.y=miy,miggi_mode.virt.x=mixv,miggi_mode.virt.y=miyv,miggi_mode.size.x=miggi_mode.size.y=miggi_mode.dpp.x=miggi_mode.dpp.y=GGI_AUTO,miggi_mode.graphtype=GT_AUTO;
#define AJUSTA_TIMER(timer,valorsec,valorusec) timer.it_interval.tv_sec=timer.it_value.tv_sec=valorsec,timer.it_interval.tv_usec=timer.it_value.tv_usec=valorusec
#define TAMPANTALLAX 102
#define TAMPANTALLAY 64
#define TAMBUFFERX 320
#define TAMBUFFERY 240

void rutina_timer(int senial);

void prepara_pantalla(void);

int
screen(int mode)
{
        /* Set screen mode to LCD mode (mode:0) or GVRAM mode (mode:1) */
        if(SimPantalla.Init==0) {
                ggi_mode Modo;
                SimPantalla.Init=1;
                if(ggiInit()!=0) {
                        ggiPanic("No se pudo inicializar GGI");
                        return(1);
                }
                if((SimPantalla.Salida=ggiOpen(NULL))==NULL) {
                        ggiPanic("No se pudo abrir la visual de salida");
                        return(1);
                }
                DEFINE_MODE(Modo,2,TAMPANTALLAX*ZOOM,TAMPANTALLAY*ZOOM,TAMBUFFERX*ZOOM,TAMBUFFERY*ZOOM);
                if(ggiSetMode(SimPantalla.Salida,&Modo)!=0) {
                        ggiPanic("No se pudo poner a 102x64");
                        return(1);
                }
                DEFINE_COLOR256(SimPantalla.ColorBlanco,202,220,202);
                DEFINE_COLOR256(SimPantalla.ColorNegro,0,0,0);
                SimPantalla.Blanco=ggiMapColor(SimPantalla.Salida,&SimPantalla.ColorBlanco);
                SimPantalla.Negro=ggiMapColor(SimPantalla.Salida,&SimPantalla.ColorNegro);
                ggiSetDisplayFrame(SimPantalla.Salida,0);
                ggiSetWriteFrame(SimPantalla.Salida,0);
                ggiSetReadFrame(SimPantalla.Salida,0);
                ggiSetGCBackground(SimPantalla.Salida,SimPantalla.Blanco);
                ggiSetGCForeground(SimPantalla.Salida,SimPantalla.Blanco);
                ggiFillscreen(SimPantalla.Salida);
                ggiSetGCForeground(SimPantalla.Salida,SimPantalla.Negro);
                SimPantalla.PosX=0,
                SimPantalla.PosY=0;
                SimPantalla.Congelado=0;
                ggiSetOrigin(SimPantalla.Salida,SimPantalla.PosX*ZOOM,SimPantalla.PosY*ZOOM);
                atexit(funcion_salida);
                /* Inicializacin de las fuentes */
                {
                        struct {
                                int num;
                                char *ptr;
                        } inits[] = {{0,letra_0},{1,letra_1},{2,letra_2},
                                     {5,letra_5},{6,letra_6},{7,letra_7},
                                     {8,letra_8}, {9,letra_9},{10,letra_10},
                                     {11,letra_11},{12,letra_12},{16,letra_16},
                                     {17,letra_17},{19,letra_19},{20,letra_20},
                                     {24,letra_24},{25,letra_25},{26,letra_26},
                                     {32,letra_32},{33,letra_33},{40,letra_40},
                                     {41,letra_41},{48,letra_48},{49,letra_49},
                                     {50,letra_50},{56,letra_56},{57,letra_57},
                                     {58,letra_58},{59,letra_59},{60,letra_60},{-1,NULL}};
                        int j,i,N;
                        sFuente *Fuente;
                        memset(SimPantalla.Fuente,0,sizeof(sFuente)*61);
                        for(j=0;inits[j].num!=-1;j++) {
                                Fuente=SimPantalla.Fuente+inits[j].num;
                                Fuente->Altura=inits[j].ptr;
                                Fuente->NumRangos=Fuente->Altura+1;
                                Fuente->Rangos=Fuente->NumRangos+1;
                                Fuente->PosAbs=(short *) (Fuente->Rangos+2*(*(Fuente->NumRangos)));
                                for(i=0,N=0;i<*(Fuente->NumRangos);i++)
                                        N+=Fuente->Rangos[i*2+1]-Fuente->Rangos[i*2]+1;
                                Fuente->Datos=(char *) (Fuente->PosAbs+N+1);
                        }
                }
                /* Inicializacin del Blink */
                SimPantalla.NumBlinkActivos=0;
                SimPantalla.Parpadeando=0;
                memset(SimPantalla.Blink,0,sizeof(sBlinkArea)*MAX_NUM_BLINK_AREAS);
#ifndef NO_SIGALRM
                AJUSTA_TIMER(SimPantalla.TimerSistema,0,125000);
                signal(SIGALRM,rutina_timer);
                setitimer(ITIMER_REAL,&(SimPantalla.TimerSistema),NULL);
#endif
                /* Inicializacin del teclado */
                SimTeclado.Entrada=SimPantalla.Salida;
                SimTeclado.PalabraEstado=0x1000;
                SimTeclado.NumBuffer=0;
                SimTeclado.TeclasBits=TeclasDefecto;
                SimTeclado.BotonesActivos=0;
                input_ayuda();
                ggiSetEventMask(SimTeclado.Entrada,emKeyboard | emPointer);

                /* Inicializacin del sim de ficheros */
                if(getcwd(SimFicheros.DirectorioRaiz,sizeof(SimFicheros.DirectorioRaiz))==NULL)
                        SimFicheros.DirectorioRaiz[0]='.';
                SimFicheros.DtaActual=SimFicheros.DtaInicial;
        }
        return(0);
}

#define DESACTIVA_TIMER() printf("Desactivando timer..."),fflush(stdout),AJUSTA_TIMER(SimPantalla.TimerSistema,0,0),\
                        setitimer(ITIMER_REAL,&(SimPantalla.TimerSistema),NULL), \
                        signal(SIGALRM,SIG_IGN),printf("ok\n"),fflush(stdout);
#define ACTIVA_TIMER() printf("Activando timer..."),fflush(stdout),AJUSTA_TIMER(SimPantalla.TimerSistema,0,500000), \
                      signal(SIGALRM,rutina_timer),\
                      setitimer(ITIMER_REAL,&(SimPantalla.TimerSistema),NULL),printf("ok\n"),fflush(stdout);



void
funcion_salida(void)
{
        if(SimPantalla.Init) {
#ifndef NO_SIGALARM
                AJUSTA_TIMER(SimPantalla.TimerSistema,0,0);
                setitimer(ITIMER_REAL,&(SimPantalla.TimerSistema),NULL);
                signal(SIGALRM,SIG_IGN);
#endif
                ggiClose(SimPantalla.Salida);
                SimPantalla.Init=0;
        }
#if 0
        {
                char Letra;
                printf("Pulse Enter para salir\n");
                scanf("%c",&Letra);
        }
#endif
        ggiExit();
}


/* pequeo hack ya que no s cmo construir un evento KEYEVENT correcto */
char Plantilla_gii_event[]={
0x24,0x05,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0xd3,0xea,0xe2,0x3a
,0x4e,0xff,0x0e,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x20,0x00,0x00,0x00
,0x39,0x00,0x00,0x00,0x54,0x5a,0x69,0x66,0x00,0x00,0x00,0x00,0xf9,0xce,0x00,0x00
,0x8c,0xfb,0xff,0xbf,0x00,0xa5,0x04,0x08,0x28,0x76,0x05,0x08,0x55,0x00,0x00,0x00
,0x09,0x00,0x00,0x00,0xf9,0xce,0x00,0x00,0x78,0x60,0x10,0x40,0xa1,0x00,0x00,0x00
,0x60,0x51,0x10,0x40,0x7a,0x45,0x0a,0x40,0x78,0x60,0x10,0x40,0x77,0x1d,0x05,0x08
,0x01,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
,0x01,0x00,0x00,0x00,0xbe,0x46,0x0a,0x40,0x02,0x00,0x00,0x00,0x09,0x00,0x00,0x00
,0x55,0x00,0x00,0x00,0x0a,0x00,0x00,0x00,0x56,0x00,0x00,0x00,0x80,0xfb,0xff,0xbf
,0x00,0x00,0x00,0x01,0x00,0x01,0x01,0x01,0x00,0x01,0x01,0x00,0x00,0x01,0x00,0x01
,0xbc,0xfb,0xff,0xbf,0x6e,0xa5,0x04,0x08,0x52,0x00,0x00,0x00,0x01,0x00,0x00,0x00
,0x04,0x00,0x00,0x00,0x09,0x00,0x00,0x00,0x6f,0x1d,0x05,0x08,0x02,0x00,0x00,0x00
,0xec,0xfb,0xff,0xbf,0x4f,0x3b,0x0a,0x40,0x58,0xfc,0xff,0xbf,0x30,0x30,0x01,0x40
,0xec,0xfb,0xff,0xbf,0xe4,0x05,0x02,0x40,0x28,0x76,0x05,0x08,0x64,0x00,0x00,0x00
,0x0a,0x00,0x00,0x00,0x18,0xfc,0xff,0xbf,0x68,0x43,0x05,0x08,0x28,0x76,0x05,0x08
,0x0c,0xfc,0xff,0xbf,0x7e,0x1e,0x07,0x40};

void input_buffer(char BanderaEsperarTecla); /* esto es de wbios.h, suponemos que lo ha incluido */


void
rutina_timer(int senial)
{
        int mi_gv_pset(int x,int y,int lop);
        int i,x,y;
        static time_t HoraSegundos=0;
        static struct tm HoraAnt;
        struct tm Hora;
        sBlinkArea *Area;
        unsigned char BanderaEv;
        unsigned short BitTeclado;
        static char Divisor=0;
        int NumTecla;
        ggi_event EvTecla;
        /* si no es una seal nuestra, salimos */
        if(senial!=SIGALRM)
                return;
        /* Primero insertamos las pulsaciones de teclas que haya pendientes */
        if(((SimTeclado.BanderaInputBuffer)&1)==0)
                input_buffer(0);
        /* implementa los pSystemCtl->EventManager->eventReq */
        if(HoraSegundos==0) {
                HoraSegundos=time(NULL);
                memcpy(&HoraAnt,localtime(&HoraSegundos),sizeof(struct tm));
        }
        HoraSegundos=time(NULL);
        memcpy(&Hora,localtime(&HoraSegundos),sizeof(struct tm));
        BanderaEv=0;
        if((pSystemCtl->EventManager->eventReq&EVENT_SEC)!=0 && Hora.tm_sec!=HoraAnt.tm_sec) {
                SimTeclado.NumEvSec++;
                BanderaEv++;
        }
        if((pSystemCtl->EventManager->eventReq&EVENT_MIN)!=0 && Hora.tm_min!=HoraAnt.tm_min) {
                SimTeclado.NumEvMin++;
                BanderaEv++;
        }
        if((pSystemCtl->EventManager->eventReq&EVENT_HOUR)!=0 && Hora.tm_hour!=HoraAnt.tm_hour) {
                SimTeclado.NumEvHour++;
                BanderaEv++;
        }
        while(BanderaEv--) {
                memcpy(&EvTecla,Plantilla_gii_event,sizeof(gii_event));
                EvTecla.any.type=evKeyPress;
                EvTecla.key.sym=' ';
                ggiEventSend(SimTeclado.Entrada,&EvTecla);
                EvTecla.any.type=evKeyRelease;
                ggiEventSend(SimTeclado.Entrada,&EvTecla);
        }
        for(BitTeclado=1,NumTecla=0;BitTeclado<=0x80;BitTeclado<<=1,NumTecla++) {
                if((SimTeclado.BotonesActivos & BitTeclado) & (SimTeclado.MascaraBotonesRepeticion & BitTeclado)) {
                        if(SimTeclado.ContadorNoRepeticion>0)
                                SimTeclado.ContadorNoRepeticion--;
                        else {
                                memcpy(&EvTecla,Plantilla_gii_event,sizeof(gii_event));
                                EvTecla.any.type=evKeyPress;
                                EvTecla.key.sym=SimTeclado.TeclasBits[NumTecla]^0x80;
                                ggiEventSend(SimTeclado.Entrada,&EvTecla);
                                EvTecla.any.type=evKeyRelease;
                                ggiEventSend(SimTeclado.Entrada,&EvTecla);
                        }
                }
        }
        memcpy(&HoraAnt,&Hora,sizeof(struct tm));
        /* implementa el parpadeo */
        if(SimPantalla.Congelado)
                return;
        Divisor++;
        Divisor%=4;
        if(SimPantalla.NumBlinkActivos>0 && Divisor==0) {
                /* copiamos el contenido y ejecutamos las tareas de parpadeo */
                if(SimPantalla.Parpadeando==0) {
                        SimPantalla.Parpadeando=1;
                        ggiSetReadFrame(SimPantalla.Salida,0);
                        ggiSetWriteFrame(SimPantalla.Salida,1);
                        ggiCopyBox(SimPantalla.Salida,0,0,TAMBUFFERX*ZOOM,TAMBUFFERY*ZOOM,0,0);
                        ggiSetReadFrame(SimPantalla.Salida,1);
                        for(i=0;i<MAX_NUM_BLINK_AREAS;i++) {
                                Area=SimPantalla.Blink+i;
                                if(Area->Activada==1) {
                                        for(y=Area->y1;y<=Area->y2;y++) {
                                                for(x=Area->x1;x<=Area->x2;x++) {
                                                        mi_gv_pset(x,y,Area->Operacion);
                                                }
                                        }

                                }
                        }
                        ggiSetDisplayFrame(SimPantalla.Salida,1);
                        ggiSetReadFrame(SimPantalla.Salida,0);
                        ggiSetWriteFrame(SimPantalla.Salida,0);
                } else {
                        SimPantalla.Parpadeando=0;
                        ggiSetDisplayFrame(SimPantalla.Salida,0);
                }
        } else if(SimPantalla.Parpadeando) {
                SimPantalla.Parpadeando=0;
                ggiSetDisplayFrame(SimPantalla.Salida,0);
        }
}

void
prepara_pantalla(void)
{
        if(SimPantalla.Parpadeando!=0) {
                ggiSetDisplayFrame(SimPantalla.Salida,0);
                SimPantalla.Parpadeando=0;
        }
}


void
input_ayuda(void)
{
        printf("                       %c    %c\n",INPUT_ARR,INPUT_ENTER);
        printf("Las teclas son:      %c   %c  %c\n",INPUT_IZQ,INPUT_DER,INPUT_EL);
        printf("                  %c    %c    %c\n",INPUT_FILER,INPUT_ABJ,INPUT_MENU);
}

int
getscreen(void)
{
        /* Get the current screen mode */
        if(SimPantalla.Init==0)
                return(-1);
        else
                return(1);
}

int
cls(int mode)
{
        /* Clear the screen (b0: work screen, b1: whole GVRAM) */
        if(SimPantalla.Init==0)
                return(-1);
        prepara_pantalla();
        ggiSetGCForeground(SimPantalla.Salida,SimPantalla.Blanco);
        if(mode & 2 || mode==4)
                ggiFillscreen(SimPantalla.Salida);
        else if(mode & 1)
                ggiDrawBox(SimPantalla.Salida,SimPantalla.PosX*ZOOM,SimPantalla.PosY*ZOOM,TAMPANTALLAX*ZOOM,TAMPANTALLAY*ZOOM);
        ggiSetGCForeground(SimPantalla.Salida,SimPantalla.Negro);
        return(0);
}

int
lcdon(int power)
{
        /* Power-on (power:1) or off (power:0) the LCD display */
        printf("lcdon(): STUB\n");
        return(0);
}

int
getlcdon(void)
{
        /* Get the LCD power status (1 if LCD display is on, 0 otherwise) */
        printf("lcdoff(): STUB\n");
        return(1);
}

int
mpread(char *buf,char *name)
{
        /* Load a MonoMap file into the specified memory buffer */
        FILE *Fich;
        Fich=fopen(name,"r");
        if(!Fich) {
                printf("mpread(): Error opening file %s\n",name);
                return(0);
        }
        fread(buf,4,1,Fich);
        fread(buf+4,getbufsize(1,1,((short *)buf)[0],((short *)buf)[1]),1,Fich);
        fclose(Fich);
        return(1);
}

struct moji *
mojisize(unsigned short s_code)
{
        /* Get the character width, height and memory address of the specified ?font? ?icon? */
        static struct moji Datos= {0,0,0};
        static char Letra;
        Datos.add=(unsigned long) &Letra;
        printf("mojisize(): STUB\n");
        return(&Datos);
}

int
getbufsize(int x1,int y1,int x2,int y2)
{
        /* Returns the buffer size required to hold a MMP of the specified area of LCD display */
        int tamx,tamy,imag,info,total;
        tamx=abs((x2-x1)+1);
        tamy=abs((y2-y1)+1);
        imag=((tamy+7)/8)*tamx;
        info=4;
        total=info+imag;
        return(total);
}

int pgetbufsize(RECT *rct )
{
        /* Returns the buffer size required to hold a MMP of the specified area of LCD display */
        return(getbufsize(rct->x1,rct->y1,rct->x2,rct->y2));
}

int
reverse(int x1,int y1,int x2,int y2 )
{
        /* Sets a reverse area in the LCD and returns a handler to it */
        printf("reverse(): STUB\n");
        return(0);
}

int
preverse(RECT *rct)
{
        /* Sets a reverse area in the LCD and returns a handler to it */
        return(reverse(rct->x1,rct->y1,rct->x2,rct->y2));
}

int
revstart(int num)
{
        /* Restart the reversing of the specified reverse area (only required if it has been revstopped) */
        printf("revstart(): STUB\n");
        return(0);
}

int
revstop(int num)
{
        /* Temporary reverse stop of the specified reverse area */
        printf("revstop(): STUB\n");
        return(0);
}

int
revend(int num)
{
        /* Release the specified reverse area handler */
        printf("revend(): STUB\n");
        return(0);
}

int
revdraw(void)
{
        /* Redraw all active reverse areas */
        printf("revdraw(): STUB\n");
        return(0);
}

int
blink(int x1,int y1,int x2,int y2,int color)
{
        /* Set a blink area in the LCD and returns a handler to it */
        int i,b;
        sBlinkArea *Area;
        if(SimPantalla.Init==0)
                return(0);
        prepara_pantalla();
        if(SimPantalla.NumBlinkActivos>=MAX_NUM_BLINK_AREAS)
                return(-1);
        for(i=0;i<MAX_NUM_BLINK_AREAS;i++) {
                Area=SimPantalla.Blink+i;
                if(Area->Activada==0) {
                        break;
                }
        }
        if(i>=MAX_NUM_BLINK_AREAS)
                return(-1);
        SimPantalla.NumBlinkActivos++;
        Area->x1=(x1<x2)?x1:x2;
        Area->x2=(x1>x2)?x1:x2;
        Area->y1=(y1<y2)?y1:y2;
        Area->y2=(y1>y2)?y1:y2;
        Area->Periodo=(color & (128+64))>>6;
        b=((color & (16+32))>>4);
        if(b==1)
                Area->Operacion=(4 | XOR_PUT);
        else if(b==2)
                Area->Operacion=0;
        else
                Area->Operacion=OR_PUT;
        Area->Activada=1;
        return(i);
}

int
pblink(BLINK_DRAW *blk)
{
        /* Set a blink area in the LCD and returns a handler to it */
        return(blink(blk->rect.x1,blk->rect.y1,blk->rect.x2,blk->rect.y2,blk->color));
}

int
blstop(int num)
{
        /* Temporary stop blinking the blink area */
        if(SimPantalla.Init==0)
                return(0);
        prepara_pantalla();
        if(num<0 || num>MAX_NUM_BLINK_AREAS || SimPantalla.Blink[num].Activada==0)
                return(-1);
        SimPantalla.Blink[num].Activada=2;
        return(0);
}

int
blstart(int num)
{
        /* Resume blinking of the specified blink area (only required if it has been blstopped) */
        if(SimPantalla.Init==0)
                return(0);
        prepara_pantalla();
        if(num<0 || num>MAX_NUM_BLINK_AREAS || SimPantalla.Blink[num].Activada==0)
                return(-1);
        SimPantalla.Blink[num].Activada=1;
        return(0);
}

int
blend(int num)
{
        /* Release the specified blink area handler */
        if(SimPantalla.Init==0)
                return(0);
        prepara_pantalla();
        if(num<0 || num>MAX_NUM_BLINK_AREAS || SimPantalla.Blink[num].Activada==0)
                return(-1);
        memset(SimPantalla.Blink+num,0,sizeof(sBlinkArea));
        SimPantalla.NumBlinkActivos--;
        return(0);
}

int
lcdfreeze(int mode)
{
        /* Stop (mode=1) or resume (mode=0) the LCD display refresh from video memory */
        if(SimPantalla.Init==0)
                return(-1);
        prepara_pantalla();
        if(mode) {
                SimPantalla.Congelado=1;
                ggiSetReadFrame(SimPantalla.Salida,0);
                ggiSetWriteFrame(SimPantalla.Salida,1);
                ggiCopyBox(SimPantalla.Salida,0,0,TAMBUFFERX*ZOOM,TAMBUFFERY*ZOOM,0,0);
                ggiSetDisplayFrame(SimPantalla.Salida,1);
                ggiSetWriteFrame(SimPantalla.Salida,0);
        } else {
                SimPantalla.Congelado=0;
                ggiSetDisplayFrame(SimPantalla.Salida,0);
        }
        return(0);
}


int
lcdvol(int vol)
{
        /* Sets the LCD contrast level */
        printf("lcdvol(): STUB\n");
        return(0);
}

int
getlcdvol(void)
{
        /* Get the LCD contrast level */
        printf("getlcdvol(): STUB\n");
        return(0);
}


/* funciones de manipulacin de GVRAM */

int
gv_place(int x,int y)
{
        /* Sets the (upper-left) LCD screen origin to the specified GVRAM coordinate */
        if(SimPantalla.Init==0)
                return(0);
        ggiSetOrigin(SimPantalla.Salida,x*ZOOM,y*ZOOM);
        return(0);
}

int
pv_place(POINT *ppt)
{
        /* Sets the (upper-left) LCD screen origin to the specified GVRAM coordinate */
        return(gv_place(ppt->x,ppt->y));
}

int
mi_gv_kput(int x, int y, char *str, int font, int gap, int lop,char BanderaSoloCuenta)
{
        int i,j,n,a,w,r;
        int resultado;
        sFuente *Fuente;
        if(SimPantalla.Init==0)
                return(0);
        prepara_pantalla();
        if(font<0 || font >60 || SimPantalla.Fuente[font].Altura==NULL)
                return(0);
        Fuente=SimPantalla.Fuente+font;
        for(i=0,resultado=0;str[i]!='\0';i++) {
                for(j=0,n=0,r=0;j<*(Fuente->NumRangos);r+=(Fuente->Rangos[j*2+1]-Fuente->Rangos[j*2]+1),j++) {
                        if(str[i]>=Fuente->Rangos[j*2] && str[i]<=Fuente->Rangos[j*2+1]) {
                                n=str[i]-Fuente->Rangos[j*2]+r;
                                goto encontrado;
                        }
                }
                continue;
                encontrado:
                a=Fuente->PosAbs[n+1]-Fuente->PosAbs[n];
                w=a/((*(Fuente->Altura)+7)/8);
                if(!BanderaSoloCuenta)
                        gv_aput(x,y,w,*(Fuente->Altura),(unsigned long) (Fuente->Datos+Fuente->PosAbs[n]),lop);
                x+=w+gap;
                resultado+=w+gap;
        }
        return(resultado);
}

int
gv_kput(int x,int y,char *str,int font,int gap,int lop)
{
        /* Draws the specified string in GVRAM using the font, character spacing and logical operation (0:COPY,1:OR,3:XOR) */
        mi_gv_kput(x,y,str,font,gap,lop,0);
        return(0);
}

int
pv_kput(KANJI_PUT *kpt)
{
        /* Draws the specified string in GVRAM using the font, character spacing and logical operation (0:COPY,1:OR,3:XOR) */
        return(gv_kput(kpt->pos.x,kpt->pos.y,kpt->str,kpt->font,kpt->gap,kpt->lop));
}

int
gv_sput(int x,int y,unsigned short code,int lop)
{
        /* Puts a SII pictograph into the specified position. Code is between 0xF6C0 (63184) and 0xF93F (63807). */
        printf("gv_sput(): STUB\n");
        return(0);
}

int
pv_sput(SIICODE_PUT *spt)
{
        /* Puts a SII pictograph into the specified position. Code is between 0xF6C0 (63184) and 0xF93F (63807). */
        return(gv_sput(spt->pos.x,spt->pos.y,spt->code,spt->lop));
}

int
mi_gv_aput(int x,int y,int w,int h,unsigned long addr,int lop)
{
        /* Draws the first character of the user-supplied font in the specified position */
        char CopyOp[2][2]={{0,1},{0,1}};
        char XorOp[2][2]={{0,1},{1,0}};
        char OrOp[2][2]={{0,1},{1,1}};
        char AndOp[2][2]={{0,0},{0,1}};
        char *Op;
        int px,py,x2,y2,tamy,nbit,punto1,punto2,nbyte,limy,i,j;
        unsigned char *Ptr;
        if(SimPantalla.Init==0)
                return(0);
        prepara_pantalla();
        switch(lop&3) {
        case XOR_PUT:
                Op=(char *)XorOp;
                break;
        case OR_PUT:
                Op=(char *)OrOp;
                break;
        case AND_PUT:
                Op=(char *)AndOp;
                break;
        default:
                Op=(char *)CopyOp;
                break;
        }
        x2=w+x-1;
        y2=h+y-1;
        tamy=(h+7)/8;
        for(nbyte=0,Ptr=((unsigned char *)addr);nbyte<tamy;nbyte++) {
                limy=(((nbyte+1)*8)>h?(h-nbyte*8):8);
                for(px=x;px<=x2;px++,Ptr++) {
                        for(py=y+nbyte*8,nbit=0;nbit<limy;py++,nbit++) {
                                punto1=gv_point(px,py);
                                punto2=(*Ptr>>nbit)&0x1;
                                for(i=0;i<ZOOM;i++) {
                                        for(j=0;j<ZOOM;j++)
                                                ggiPutPixel(SimPantalla.Salida,px*ZOOM+i,py*ZOOM+j,BIT2COLOR(Op[(punto1<<1)|punto2]));
                                }
                        }
                }
        }
        return(0);
}

int
gv_aput(int x,int y,int w,int h,unsigned long addr,int lop)
{
        switch(lop & 3) {
        case KOR_PUT:
                mi_gv_aput(x,y,w,h,addr,(lop & ~3) | OR_PUT );
                break;
        default:
                mi_gv_aput(x,y,w,h,addr,(lop & ~3) | COPY_PUT );
                break;
        }
        return(0);
}


int
pv_aput(FONT_PUT *apt)
{
        /* Draws the first character of the user-supplied font in the specified position */
        return(gv_aput(apt->pos.x,apt->pos.y,apt->width,apt->hight,apt->addr,apt->lop));
}

int
gv_mmap(int x,int y,char *name,int lop)
{
        /* Load a MonoMap file directly into GVRAM in the specified position */
        printf("gv_mmap(): STUB\n");
        return(0);
}

int
pv_mmap(MMAP_PUT *mpt)
{
        /* Load a MonoMap file directly into GVRAM in the specified position */
        return(gv_mmap(mpt->pos.x,mpt->pos.y,mpt->name,mpt->lop));
}

int
gv_line(int x1,int y1,int x2,int y2,int lop,unsigned char style)
{
        char BanderaDibuja;
        int i,j;
        /* Draws a line from x1,y1 to x2,y2 using the specified logical operation (0:COPY,1:OR,3:XOR) */
        if(SimPantalla.Init==0)
                return(0);
        prepara_pantalla();
        if((style&0xff)!=0xff)
                printf("gv_line(): line style not implemented, drawing solid line\n");
        BanderaDibuja=0;
        if(lop==0) {
                BanderaDibuja=1;
        } else if(lop==1) {
                printf("gv_line(): OR_PUT not implemented, using COPY_PUT\n");
                BanderaDibuja=1;
        } else if(lop==3) {
                printf("gv_line(): XOR_PUT not implemented, using COPY_PUT\n");
                BanderaDibuja=1;
        }
        if(BanderaDibuja) {
                for(i=0;i<ZOOM;i++) {
                        for(j=0;j<ZOOM;j++)
                                ggiDrawLine(SimPantalla.Salida,x1*ZOOM+i,y1*ZOOM+j,x2*ZOOM+i,y2*ZOOM+j);
                }
        }
        return(0);
}

int
pv_line(LINE_DRAW *lpt)
{
        /* Draws a line from x1,y1 to x2,y2 using the specified logical operation (0:COPY,1:OR,3:XOR) */
        return(gv_line(lpt->st.x,lpt->st.y,lpt->ed.x,lpt->ed.y,lpt->lop,lpt->txt));
}


int
gv_square(int x1,int y1,int x2,int y2,int lop,unsigned long style)
{
        /* Draws a rectangle (lop&3: logical operation; lop&4:0 not filled, LSB of style is line draw format; lop&4:1 filled, style is
           tile pattern). */
        if(SimPantalla.Init==0)
                return(-1);
        prepara_pantalla();
        if((lop & 3)!=0 && (lop & 4)==1)
                printf("gv_square(): OR_PUT and XOR_PUT not implemented when drawing a filled squere, using COPY_PUT\n");
        if((lop & 4)==0 && (style&0xff)!=0xff)
                printf("gv_square(): line style not implemented, drawing solid line\n");
        if((lop & 4)==1 && (style&0xffffffff)!=0xffffffff)
                printf("gv_square(): fill style not implemented, filling with solid pattern\n");
        if(lop&4) {
                ggiDrawBox(SimPantalla.Salida,x1*ZOOM,y1*ZOOM,(x2-x1+1)*ZOOM,(y2-y1+1)*ZOOM);
        } else {
                miHLine(x1+1,y1,x2-x1-1,(lop & ~4) | (1<<2));
                miHLine(x1+1,y2,x2-x1-1,(lop & ~4) | (1<<2));
                miVLine(x1,y1,y2-y1+1,(lop & ~4) | (1<<2));
                miVLine(x2,y1,y2-y1+1,(lop & ~4) | (1<<2));
        }
        return(0);
}

int
pv_square(SQUARE_DRAW *spt)
{
        /* Draws a rectangle (lop&3: logical operation; lop&4:0 not filled, LSB of style is line draw format; lop&4:1 filled, style is
           tile pattern). */
        return(gv_square(spt->rect.x1,spt->rect.y1,spt->rect.x2,spt->rect.y2,spt->lop,spt->txt));
}

int
gv_circle(int x,int y,int xr,int yr,int d1,int d2,int lop,unsigned long style,unsigned char *buf)
{
        /* Display a circle/oval, circle/oval arc or fan at position x, y, with radius xr and yr, start/end angles d1 and d2, logical
           operation lop b1,b0 (00:COPY, 01: XOR, 10:OR, 11:AND), b2 (0: do not fill, LSB of style is line draw format, 1: fill,
           style is fill pattern) and b4,b3 (0X: Circle/oval, 10:Arc, 11:Fan). Buf is an 8000 bytes buffer needed to draw a fan (not
           needed elsewhere). */
        printf("gv_circle(): STUB\n");
        return(0);
}


int
pv_circle(CIRCLE_DRAW *cpt)
{
        /* Display a circle/oval, circle/oval arc or fan at position x, y, with radius xr and yr, start/end angles d1 and d2, logical
           operation lop b1,b0 (00:COPY, 01: XOR, 10:OR, 11:AND), b2 (0: do not fill, LSB of style is line draw format, 1: fill,
           style is fill pattern) and b4,b3 (0X: Circle/oval, 10:Arc, 11:Fan). Buf is an 8000 bytes buffer needed to draw a fan (not
           needed elsewhere). */
        return(gv_circle(cpt->pos.x,cpt->pos.y,cpt->rad.xr,cpt->rad.yr,cpt->deg.d1,cpt->deg.d2,cpt->lop,cpt->txt,cpt->buf));
}

int
gv_pset(int x,int y,int lop)
{
        int mi_gv_pset(int x,int y,int lop);
        /* Draws the specified pixel using the specified logical operation and color in lop b1,b0 (00: COPY, 01: XOR, 10: OR,
           11:AND) and b2 (0: Clear, 1:Black) */
        if(SimPantalla.Init==0)
                return(0);
        prepara_pantalla();
        return(mi_gv_pset(x,y,lop));
}

int
mi_gv_pset(int x,int y,int lop)
{
        int i,j;
        if((lop&3)==0) {
                for(i=0;i<ZOOM;i++) {
                        for(j=0;j<ZOOM;j++)
                                ggiPutPixel(SimPantalla.Salida,x*ZOOM+i,y*ZOOM+j,BIT2COLOR((lop>>2)&1));
                }
        } else {
                ggi_pixel estado;
                int bit1,bit2;
                ggiGetPixel(SimPantalla.Salida,x*ZOOM,y*ZOOM,&estado);
                bit1=COLOR2BIT(estado);
                bit2=COLOR2BIT((lop>>2)&1);
                switch(lop&3) {
                case XOR_PUT:
                        estado=BIT2COLOR(bit1^bit2);
                        break;
                case OR_PUT:
                        estado=BIT2COLOR(bit1|bit2);
                        break;
                case AND_PUT:
                        estado=BIT2COLOR(bit1&bit2);
                        break;
                }
                for(i=0;i<ZOOM;i++) {
                        for(j=0;j<ZOOM;j++)
                                ggiPutPixel(SimPantalla.Salida,x*ZOOM+i,y*ZOOM+j,estado);
                }
        }
        return(0);

}

int
pv_pset(POINT_DRAW *ppt)
{
        /* Draws the specified pixel using the specified logical operation and color in lop b1,b0 (00: COPY, 01: XOR, 10: OR,
           11:AND) and b2 (0: Clear, 1:Black) */
        return(gv_pset(ppt->pos.x,ppt->pos.y,ppt->lop));
}

int MiraGvPoint=0;

int
gv_point(int x,int y)
{
        /* Get the color (1:Black, 0:White) of the specified position */
        ggi_pixel estado;
        if(SimPantalla.Init==0)
                return(0);
        ggiGetPixel(SimPantalla.Salida,x*ZOOM,y*ZOOM,&estado);
        if(MiraGvPoint) {
                printf("%c",COLOR2BIT(estado)*'#' + (1-COLOR2BIT(estado))*'');
                fflush(stdout);
        }
        return(COLOR2BIT(estado));
}

int
pv_point(POINT *ppt)
{
        /* Get the color (1:Black, 0:White) of the specified position */
        return(gv_point(ppt->x,ppt->y));
}

int
gv_get(int x1,int y1,int x2,int y2,char *buf)
{
        /* Stores the specified screen rectangle into the specified MonoMap memory buffer */
        int px,py,w,h,tamy,nbit,punto1,nbyte,limy;
        unsigned char *Ptr;
        if(SimPantalla.Init==0)
                return(0);
        memset(buf,0,getbufsize(x1,y1,x2,y2));
        w=x2-x1+1;
        h=y2-y1+1;
        tamy=(h+7)/8;
        ((unsigned short *)buf)[0]=w;
        ((unsigned short *)buf)[1]=h;
        for(nbyte=0,Ptr=(((unsigned char *)buf)+4);nbyte<tamy;nbyte++) {
                limy=(((nbyte+1)*8)>h?(h-nbyte*8):8);
                for(px=x1;px<=x2;px++,Ptr++) {
                        for(py=y1+nbyte*8,nbit=0;nbit<limy;py++,nbit++) {
                                punto1=gv_point(px,py);
                                *Ptr|=(punto1<<nbit);
                        }
                }
        }
        return(0);
}

int
pv_get(GVRAM_GET *gpt)
{
        /* Stores the specified screen rectangle into the specified MonoMap memory buffer */
        return(gv_get(gpt->rect.x1,gpt->rect.y1,gpt->rect.x2,gpt->rect.y2,gpt->buf));
}

int
gv_put(int x,int y,char *buf,int logic)
{
        /* Draws the specified monomap buffer into the specified screen coordinate with the specified logical operation (0:COPY,
           1: XOR, 2:OR, 3:AND) */
        return(mi_gv_aput(x,y,((unsigned short *) buf)[0],((unsigned short *) buf)[1],(unsigned long) (((unsigned short *) buf)+2),logic));
}

int
pv_put(GVRAM_PUT *gpt)
{
        /* Draws the specified monomap buffer into the specified screen coordinate with the specified logical operation (0:COPY,
           1: XOR, 2:OR, 3:AND) */
        return(gv_put(gpt->pos.x,gpt->pos.y,gpt->buf,gpt->lop));
}

int
gv_scroll(int lines)
{
        /* Scroll GVRAM the specified number of pixels (lines) */
        if(SimPantalla.Init==0)
                return(-1);
        prepara_pantalla();
        ggiCopyBox(SimPantalla.Salida,0,lines*ZOOM,TAMPANTALLAX*ZOOM,(TAMPANTALLAY-lines)*ZOOM,0,0);
        ggiSetGCForeground(SimPantalla.Salida,SimPantalla.Blanco);
        ggiDrawBox(SimPantalla.Salida,0,(TAMPANTALLAY-lines)*ZOOM,TAMPANTALLAX*ZOOM,lines*ZOOM);
        ggiSetGCForeground(SimPantalla.Salida,SimPantalla.Negro);
        return(0);
}

int
gv_reverse(int x1,int y1,int x2,int y2)
{
        /* Reverse the contents of the specified area of GVRAM */
        int x,y,i,j;
        int Color;
        if(SimPantalla.Init==0)
                return(0);
        prepara_pantalla();
        for(x=x1;x<=x2;x++) {
                for(y=y1;y<=y2;y++) {
                        Color=1-gv_point(x,y);
                        for(i=0;i<ZOOM;i++) {
                                for(j=0;j<ZOOM;j++)
                                        ggiPutPixel(SimPantalla.Salida,x*ZOOM+i,y*ZOOM+j,BIT2COLOR(Color));
                        }
                }
        }
        return(0);
}

int
pv_reverse(RECT *rpt)
{
        /* Reverse the contents of the specified area of GVRAM */
        return(gv_reverse(rpt->x1,rpt->y1,rpt->x2,rpt->y2));
}

int
gv_clear(int x1,int y1,int x2,int y2)
{
        /* Clear (set to 0) the specified area of GVRAM */
        if(SimPantalla.Init==0)
                return(-1);
        prepara_pantalla();
        ggiSetGCForeground(SimPantalla.Salida,SimPantalla.Blanco);
        ggiDrawBox(SimPantalla.Salida,x1*ZOOM,y1*ZOOM,(x2-x1+1)*ZOOM,(y2-y1+1)*ZOOM);
        ggiSetGCForeground(SimPantalla.Salida,SimPantalla.Negro);
        return(0);
}

int
pv_clear(RECT *cpt)
{
        /* Clear (set to 0) the specified area of GVRAM */
        return(gv_clear(cpt->x1,cpt->y1,cpt->x2,cpt->y2));
}

/* funciones de ayuda */
int
miHLine(int x1,int y1,int ex, int lop)
{
        int i;
        for(i=0;i<ex;i++)
                mi_gv_pset(x1+i,y1,lop);
        return(0);
}

int
miVLine(int x1,int y1,int ey, int lop)
{
        int i;
        for(i=0;i<ey;i++)
                mi_gv_pset(x1,y1+i,lop);
        return(0);
}

#endif
