/* exchHions-2

Copyright Jrgen Prenzel 2001
The program is licenced through the GPL (see LICENCE)

*/

#include <stdio.h>
#include <math.h>

#define MAXIT 100

static double actv(int z, double I);

int akeh(double ph0, double ph1, double ctal, double str, double* exh){

  /* Berechnung von sogenanntem austauschbarem Wasserstoff in
     Bodenproben. Ausgegangen wird von der Ansuerung des
     Extraktionsmittels (pH0->pH1). Es wird eine Korrektur angebracht,
     weil auch extrahiertes Aluminium den Extrakt ansuern kann. Dabei
     wird davon ausgegangen, da das Al als dreiwertig am Austauscher
     vorliegt und im Extrakt hydrolisiert. Aktivittskoeffizienten
     werden bercksichtigt, aber die Ionestrke wird als konstant
     angenommen. Das Ergebnis wird als Konzentration von H-Ionen, die
     vom Austauscher stammen sollen, angegeben.  Diese kann negativ
     herauskommen, was auf Unzulnglichkeiten des Modellansatzes
     hinweist.

     Argumente mit erlaubtem Wertebereich:
     1.) pH des Extraktionsmittels vor der Extraktion ( 0..14 )
     2.) pH im Extrakt                                ( 0..14 )
     3.) Al im Extrakt (Gesamt-Al in mol/l)          ( 0...  )
     4.) Ionenstrke im Extrakt (mol/l)               ( 0..10 )
     5.) Rckgabewert: H+ vom Austauscher (mol/l) (kann <0 sein!)  

     Funktionswert:  0 => Kein Fehler, Konvergenz
                     1 => unerlaubte Argumente  
                     2 => keine Konvergenz

  */


  double x;  /*   Al+++ - Aktivitt (mol/l) */
  double f;  /*  Al-Bilanz, Polynom in x, dessens Nullstelle gesucht wird */
  double df; /*   Ableitung von f */
  double cal,ccpl; /*  Konzentrationen in mol/l */
  double pal,poh,pcpl,pk;  /*  -log10 */

  typedef struct {
    int iAl,iOH;
    double pK;
  } ALOHCPL;
#define NCPL 7
  const ALOHCPL alohs[NCPL] = { /* Liste der Al-OH-Komplexe */
    {1,1,9.03},                 /*   nach Baes & Mesmer */
    {1,2,18.7},
    {1,3,27.0},
    {1,4,33.0},
    {2,2,20.3},
    {3,4,42.06},
    {13,32,349.27},
};
  ALOHCPL aloh;
  double dpal,dcpl,dpcpl,pctal,sumoh;
  int l1,l2,zcpl,k;
  const int verbose = 1;

/*  ------------------------------------------------------------- */


  *exh = 0.0;

  if(ph0 < 0 || ph0 > 14 || ph1 < 0 || ph1 > 14 || ctal < 0  
     || str < 0 || str > 10){
    return 1;
  }
  if(ph1 >= ph0){
    return 0;
  }
  if(ctal <= 0.0){
    *exh = 1.e6 * (pow(10,-ph1)/actv(1,str) 
		   - pow(10,-ph0)/actv(1,str));
    return 0;
  }


  ctal = ctal*1.e-6; /*  -> mol/l */
  /* .................................... Startpunktsuche  ........ */
  x = ctal;
  poh = 14.0 - ph1;
  pctal -log10(ctal);
  for(l1=0;l1<100;l1++){
    pal = -log10(x*actv(3,str));
    k = 0;
    for(l2=0;l2<NCPL;l2++){
      zcpl = 3*alohs[l2].iAl - alohs[l2].iOH ;
      pcpl = alohs[l2].iAl * pal + alohs[l2].iOH * poh  - alohs[l2].pK;
      if(pcpl < pctal){ /*  Komplex ist mehr als Gesamt-Al */
	x = x/2;
	k = 1;
	break;
      }
    }
    if(k == 0){
      break;
    }
  }


  /* .................................... Newton-Iteration ........ */
  k = 0;
  for(l1=0;l1<MAXIT;l1++){
    f = x/actv(3,str) - ctal;
    df = 1.0/actv(3,str);
    pal = -log10(x);
    dpal = -1/log(10)/x;
    poh = 14.0 - ph1;
    for(l2=0;l2<NCPL;l2++){
      zcpl = 3*alohs[l2].iAl - alohs[l2].iOH ;
      pcpl = alohs[l2].iAl * pal + alohs[l2].iOH * poh  - alohs[l2].pK;
      dpcpl = alohs[l2].iAl * dpal;
      if(pcpl < 100.0 && pcpl > -100.0){
	ccpl = pow(10.0,-pcpl)/actv(zcpl,str);
        dcpl = -log(10)*exp(-log(10)*pcpl)*dpcpl/actv(zcpl,str);
	f += ccpl*alohs[l2].iAl;
	df += dcpl*alohs[l2].iAl;
      }
      else{
	if(pcpl < -100.0){
	  f += 1e100; /* unsinnige Zwischenergebnisse  */
	}
      }
    }
    if(fabs(f/ctal) < 1.e-12){ /*  Erfolgskriterium */
      k = 1;
      break;
    }
    x = x - f/df; /*  Newton-Formel */
  }

  if(k == 0){
    return 2;
  }

  sumoh = 0.0;
  for(l2=0;l2<NCPL;l2++){
    zcpl = 3*alohs[l2].iAl - alohs[l2].iOH ;
    pcpl = alohs[l2].iAl * pal + alohs[l2].iOH * poh  - alohs[l2].pK;
      ccpl = pow(10.0,-pcpl)/actv(zcpl,str);
      sumoh += ccpl*alohs[l2].iOH;      
  }
  *exh = 1.e6 * (pow(10,-ph1)/actv(1,str) 
		 - pow(10,-ph0)/actv(1,str)
		 - sumoh);
  return 0;
}


static double actv(int z, double I){
  /* Aktivittskoeffizienten nach der Davies-Formel. Quelle
   z.B. Lindsay(1979)*/

  const double A = 0.5;
  double x,s;
  s = sqrt(I);
  x = -A * z*z * (s/(1.0+s) - 0.3*I);
  return pow(10.0,x);
}








