/* MakeTeXPK -- create a missing PK and/or TFM file.

   Copyright (C) 1994, 1995 Ralph Schleicher  */

/* This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Program arguments:

	font		base name of the METAFONT parameter file
	dpi		required resolution
	res		base resolution of the device
	mag		magnification of the font
	[mode]		device name from `modes.mf', defaults to `localfont'

   Program options:

        -f, --force	compute the font even if it exists

   Environment variables:

	MAKETEXPK	directory for PK files, defaults to `./pk'
	MAKETEXTFM	directory for TFM files, defaults to `./tfm'
	MAKETEXLOG	directory for LOG files, defaults to `./log'

   Features:

	DPI will be computed from RES and MAG if DPI is equal to zero.
	MAG may be given TeX encoded -- `magstep0', `magstephalf', ... and
	`magstep5' are supported.

   I make use of some OS/2 extensions of the REXX language, here.  */


call rxfuncadd 'sysloadfuncs', 'rexxutil', 'sysloadfuncs';
call sysloadfuncs;

prog = 'MakeTeXPK';
exec = '@';

force = 0;

parse arg all;
do while all \= ''
  parse var all option rest;
  if left(option, 1) \= '-' then
    leave;
  select
    when option = '-f' | option = '--force' then
      force = 1;
    otherwise
      call usage;
  end;
  all = rest;
end;

parse var all font dpi res mag mode;
if font = '' | dpi = '' | res = '' | mag = '' then
  call usage;
if \datatype(dpi, 'number') then
  call usage dpi||': Expect numerical argument';
if \datatype(res, 'number') then
  call usage res||': Expect numerical argument';
parse upper var mag temp;
select
  when temp = 'MAGSTEP0' then
    mag = 1.0;
  when temp = 'MAGSTEPHALF' then
    mag = 1.095;
  when temp = 'MAGSTEP1' then
    mag = 1.2;
  when temp = 'MAGSTEP2' then
    mag = 1.44;
  when temp = 'MAGSTEP3' then
    mag = 1.728;
  when temp = 'MAGSTEP4' then
    mag = 2.074;
  when temp = 'MAGSTEP5' then
    mag = 2.488;
  otherwise
    nop;
end;
if dpi = 0 then
  do
    dpi = res * mag;
    int = trunc(dpi);
    if (dpi - int) >= 0.5 then
      int = int + 1;
    dpi = int;
  end;
if mode = '' then
  mode = 'localfont';

pk_dir = os2name(getenv('MAKETEXPK', './pk')||'/'||mode||'/'||dpi||'dpi');
if makedir(pk_dir) \= 0 then
  call usage unixname(pk_dir)||': Cannot create directory';
tfm_dir = os2name(getenv('MAKETEXTFM', './tfm'));
if makedir(tfm_dir) \= 0 then
  call usage unixname(tfm_dir)||': Cannot create directory';
log_dir = os2name(getenv('MAKETEXLOG', './log'));
if makedir(log_dir) \= 0 then
  call usage unixname(log_dir)||': Cannot create directory';

gf_file = font||'.'||dpi||'gf';
pk_file = font||'.'||dpi||'pk';
tfm_file = font||'.tfm';
log_file = font||'.log';

if \force then
  do
    pk_stream = stream(pk_dir||'\'||pk_file, 'command', 'query exists');
    if pk_stream \= '' then
      do
    	say unixname(pk_dir||'\'||pk_file); exit 0;
      end;
  end;

parse upper var font temp;
if substr(temp, 1, 2) = 'CM' then
  virmf = 'cmmf';
else if substr(temp, 1, 2) = 'DC' then
  virmf = 'dxmf';
else
  virmf = 'mf';
exec virmf '\mode:='||mode||'; mag:='||mag||'; batchmode; input' font '>nul 2>nul';
if rc \= 0 then
  do
    call copy log_file, log_dir||'\'||log_file;
    call cleanup gf_file tfm_file log_file;
    call usage virmf||': See the log file for how to fix this error';
  end;

exec 'gftopk' gf_file pk_file;
if rc \= 0 then
  do
    call cleanup gf_file pk_file tfm_file log_file;
    call usage 'gftopk: I think this program is broken';
  end;

call copy pk_file, pk_dir||'\'||pk_file;
call copy tfm_file, tfm_dir||'\'||tfm_file;

call cleanup gf_file pk_file tfm_file log_file;

say unixname(pk_dir||'\'||pk_file);

exit 0;


usage: procedure expose prog
  parse arg message;
  if message = '' then
    do
      say 'Usage:  '||prog||' [-f] <font> <dpi> <res> <mag> [<mode>]';
      say '';
      say '	<font>	base name of the font';
      say '	<dpi>	magnified resolution';
      say '	<res>	resolution of the device';
      say '	<mag>	required magnification';
      say '	<mode>	METAFONT mode';
      say '';
      say '	-f, --force';
      say '		compute the font even if it already exists';
    end;
  else
    say prog||':' message;
  exit 1;

cleanup: procedure
  parse arg files;
  do while files \= ''
    parse var files file rest;
    call sysfiledelete file;
    files = rest;
  end;
  return 0;

copy: procedure expose exec
  parse arg from, to;
  if strip(from) = '' | strip(to) = '' then
    return 1;
  exec 'copy' '"'||from||'"' '"'||to||'" >nul 2>&1';
  return rc;

/* REXXLIB: os2name unixname makedir */

getenv: procedure
  parse arg name, default;
  variable = value(name, , 'OS2ENVIRONMENT');
  if variable = '' then
    variable = default;
  return variable;

toupper: procedure
  parse arg string;
  upper = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
  lower = 'abcdefghijklmnopqrstuvwxyz';
  return translate(string, upper, lower);

tilde: procedure
  parse arg string, mode;
  home = getenv('HOME');
  if home = '' then
    return string;
  if mode = '' then
    mode = 'E';
  else
    mode = left(toupper(mode), 1);
  select
    when mode = 'E' then
      if left(string, 1) = '~' then
	string = home||substr(string, 2);
    when mode = 'R' then
      do
	home_len = length(home); string_len = length(string);
	if string_len >= home_len & left(string, home_len) = home then
	  string = '~'||substr(string, home_len + 1);
      end;
    otherwise
      nop;
  end;
  return string;

os2name: procedure
  parse arg name;
  name = tilde(name, 'e');
  name = translate(name, '\', '/');
  name = strip(name, 't', '\');
  return name;

unixname: procedure
  parse arg name;
  name = tilde(name, 'r');
  name = translate(name, '/', '\');
  name = strip(name, 't', '/');
  return name;

isalpha: procedure
  parse arg str;
  if str = '' then
    return 0;
  do until str = ''
    char = left(str, 1);
    if \((char >= 'A' & char <= 'Z') | (char >= 'a' & char <= 'z')) then
      return 0;
    str = substr(str, 2);
  end;
  return 1;

isdrive: procedure
  parse upper arg name;
  if length(name) > 0 then
    do
      name = substr(name, 1, 1)||':';
      map = sysdrivemap('C:');
      do while map \= ''
	parse var map drive map;
	if name = drive then
	  return 1;
      end;
    end;
  return 0;

makedir: procedure
  parse arg dirs;
  do while dirs \= ''
    parse var dirs dir dirs;
    dir = os2name(dir);
    parse var dir drive ':' rest;
    if length(drive) = 1 & isalpha(drive) then
      do
	if isdrive(dir) then
	  do
	    all = substr(dir, 1, 2);
	    dir = substr(dir, 3);
	  end;
	else
	  return 1;
      end;
    else
      all = '';
    if left(dir, 1) = '\' then
      do
        all = all||'\';
        dir = substr(dir, 2);
      end;
    do until dir = ''
      parse var dir sub '\' rest;
      dir = rest;
      all = all||sub;
      rc = sysmkdir(all);
      if \(rc = 0 | rc = 5) then
        return 1;
      all = all||'\';
    end;
  end;
  return 0;
